// Utilities to create clients to our flavour of API
package api

import (
	pb "a.yandex-team.ru/yp/go/proto/hq"
	"bytes"
	"github.com/golang/protobuf/proto"
	"golang.org/x/net/context"
	"golang.org/x/net/context/ctxhttp"
	"io/ioutil"
	"net/http"
)

var statusCastFailed = &pb.Status{
	Status:  StatusFailure,
	Code:    StatusCodeInternalError,
	Reason:  StatusReasonInternalError,
	Message: "Failed to cast input message",
}

// CallAPIMethod performs HTTP request, serializing input, receiving response and loading it into out
func CallAPIMethod(ctx context.Context, client *http.Client, url string, in proto.Message, out proto.Message) *pb.Status {
	b, err := proto.Marshal(in)
	if err != nil {
		return &pb.Status{
			Status:  StatusFailure,
			Code:    StatusCodeInternalError,
			Reason:  StatusReasonInternalError,
			Message: err.Error(),
		}
	}
	r, err := http.NewRequest("POST", url, bytes.NewReader(b))
	if err != nil {
		return &pb.Status{
			Status:  StatusFailure,
			Code:    StatusCodeInternalError,
			Reason:  StatusReasonInternalError,
			Message: err.Error(),
		}
	}
	r.Header.Add(headerContentType, headerValueProtobuf)
	r.Header.Add(headerAccept, headerValueProtobuf)
	resp, err := ctxhttp.Do(ctx, client, r)
	if err != nil {
		return &pb.Status{
			Status:  StatusFailure,
			Code:    StatusCodeServiceUnavailable,
			Reason:  StatusReasonServiceUnavailable,
			Message: err.Error(),
		}
	}
	b, err = ioutil.ReadAll(r.Body)
	resp.Body.Close()
	if err != nil {
		return &pb.Status{
			Status:  StatusFailure,
			Code:    StatusCodeServiceUnavailable,
			Reason:  StatusReasonServiceUnavailable,
			Message: "Failed to read response " + err.Error(),
		}
	}
	if resp.StatusCode != 200 {
		// Need to deserialize Status
		status := &pb.Status{}
		err := proto.Unmarshal(b, status)
		if err != nil {
			return &pb.Status{
				Status:  StatusFailure,
				Code:    StatusCodeServiceUnavailable,
				Reason:  StatusReasonServiceUnavailable,
				Message: "Failed to read response " + err.Error(),
			}
		}
		return status
	}
	err = proto.Unmarshal(b, out)
	if err != nil {
		return &pb.Status{
			Status:  StatusFailure,
			Code:    StatusCodeServiceUnavailable,
			Reason:  StatusReasonServiceUnavailable,
			Message: "Failed to load response " + err.Error(),
		}
	}
	return nil
}
