package api

import (
	"bytes"
	"net/http"
	"sync"

	"github.com/golang/protobuf/jsonpb"
	"github.com/golang/protobuf/proto"

	pb "a.yandex-team.ru/yp/go/proto/hq"
)

const smallBufSize = 4096

var statusInternalBadParam = pb.Status{
	Status:  StatusFailure,
	Code:    StatusCodeInternalError,
	Reason:  "",
	Message: "Invalid object passed to reader",
}

var smallBufPool = sync.Pool{
	New: func() interface{} {
		return make([]byte, 0, smallBufSize)
	},
}

// ReadRequest reads API request from http POST request into provided req.
// Request object MUST have Unmarshal method (generated by protobuf compiler).
// If returned value is non nil - error happened.
func ReadRequest(r *http.Request, req interface{}) *pb.Status {
	defer func() { _ = r.Body.Close() }()
	if r.Header.Get(headerContentType) == headerValueProtobuf {
		m, ok := req.(proto.Message)
		if !ok {
			return &statusInternalBadParam
		}
		b := smallBufPool.Get().([]byte)
		buf := bytes.NewBuffer(b)
		defer func() {
			b := buf.Bytes()
			smallBufPool.Put(b[0:0])
		}()
		_, err := buf.ReadFrom(r.Body)
		if err != nil {
			return &pb.Status{
				Status:  StatusFailure,
				Code:    StatusCodeInvalid,
				Reason:  StatusReasonInvalid,
				Message: err.Error(),
			}
		}
		err = proto.Unmarshal(buf.Bytes(), m)
		if err != nil {
			return &pb.Status{
				Status:  StatusFailure,
				Code:    StatusCodeInvalid,
				Reason:  StatusReasonInvalid,
				Message: err.Error(),
			}
		}
		return nil
	} else {
		err := jsonpb.Unmarshal(r.Body, req.(proto.Message))
		if err != nil {
			return &pb.Status{
				Status:  StatusFailure,
				Code:    StatusCodeInvalid,
				Reason:  StatusReasonInvalid,
				Message: err.Error(),
			}
		}
		return nil
	}
}
