package api

import "net/http"

// Values of Status.Status
const (
	StatusSuccess = "Success"
	StatusFailure = "Failure"
)

// StatusReason is an enumeration of possible failure causes.  Each StatusReason
// must map to a single HTTP status code, but multiple reasons may map
// to the same HTTP status code.
const (
	// StatusReasonUnknown means the server has declined to indicate a specific reason.
	// The details field may contain other information about this error.
	// Status code 500.
	StatusCodeUnknown   = http.StatusInternalServerError
	StatusReasonUnknown = ""

	// StatusReasonUnauthorized means the server can be reached and understood the request, but requires
	// the user to present appropriate authorization credentials (identified by the WWW-Authenticate header)
	// in order for the action to be completed. If the user has specified credentials on the request, the
	// server considers them insufficient.
	StatusCodeUnauthorized   = http.StatusUnauthorized
	StatusReasonUnauthorized = "Unauthorized"

	// StatusReasonForbidden means the server can be reached and understood the request, but refuses
	// to take any further action.  It is the result of the server being configured to deny access for some reason
	// to the requested resource by the client.
	StatusCodeForbidden   = http.StatusForbidden
	StatusReasonForbidden = "Forbidden"

	// StatusReasonNotFound means one or more resources required for this operation
	// could not be found.
	StatusCodeNotFound          = http.StatusNotFound
	StatusReasonNotFound string = "NotFound"

	// StatusReasonAlreadyExists means the resource you are creating already exists.
	StatusCodeAlreadyExists   = http.StatusConflict
	StatusReasonAlreadyExists = "AlreadyExists"

	// StatusReasonConflict means the requested operation cannot be completed
	// due to a conflict in the operation. The client may need to alter the
	// request. Each resource may define custom details that indicate the
	// nature of the conflict.
	StatusCodeConflict   = http.StatusConflict
	StatusReasonConflict = "Conflict"

	// StatusReasonGone means the item is no longer available at the server and no
	// forwarding address is known.
	StatusCodeGone   = http.StatusGone
	StatusReasonGone = "Gone"

	// StatusReasonInvalid means the requested create or update operation cannot be
	// completed due to invalid data provided as part of the request. The client may
	// need to alter the request. When set, the client may use the StatusDetails
	// message field as a summary of the issues encountered.
	StatusCodeInvalid   = http.StatusBadRequest
	StatusReasonInvalid = "Invalid"

	// StatusReasonServerTimeout means the server can be reached and understood the request,
	// but cannot complete the action in a reasonable time. The client should retry the request.
	// This is may be due to temporary server load or a transient communication issue with
	// another server. Status code 500 is used because the HTTP spec provides no suitable
	// server-requested client retry and the 5xx class represents actionable errors.
	StatusCodeServerTimeout   = http.StatusInternalServerError
	StatusReasonServerTimeout = "ServerTimeout"

	// StatusReasonTimeout means that the request could not be completed within the given time.
	// Clients can get this response only when they specified a timeout param in the request,
	// or if the server cannot complete the operation within a reasonable amount of time.
	// The request might succeed with an increased value of timeout param. The client *should*
	// wait at least the number of seconds specified by the retryAfterSeconds field.
	StatusCodeTimeout   = http.StatusRequestTimeout
	StatusReasonTimeout = "Timeout"

	// StatusReasonBadRequest means that the request itself was invalid, because the request
	// doesn't make any sense, for example deleting a read-only object.  This is different than
	// StatusReasonInvalid above which indicates that the API call could possibly succeed, but the
	// data was invalid.  API calls that return BadRequest can never succeed.
	StatusCodeBadRequest   = http.StatusBadRequest
	StatusReasonBadRequest = "BadRequest"

	// StatusReasonMethodNotAllowed means that the action the client attempted to perform on the
	// resource was not supported by the code - for instance, attempting to delete a resource that
	// can only be created. API calls that return MethodNotAllowed can never succeed.
	StatusCodeMethodNotAllowed   = http.StatusMethodNotAllowed
	StatusReasonMethodNotAllowed = "MethodNotAllowed"

	// StatusReasonInternalError indicates that an internal error occurred, it is unexpected
	// and the outcome of the call is unknown.
	StatusCodeInternalError   = http.StatusInternalServerError
	StatusReasonInternalError = "InternalError"

	// StatusReasonExpired indicates that the request is invalid because the content you are requesting
	// has expired and is no longer available. It is typically associated with watches that can't be
	// serviced.
	StatusCodeExpired   = http.StatusGone
	StatusReasonExpired = "Expired"

	// StatusReasonServiceUnavailable means that the request itself was valid,
	// but the requested service is unavailable at this time.
	// Retrying the request after some time might succeed.
	StatusCodeServiceUnavailable   = http.StatusServiceUnavailable
	StatusReasonServiceUnavailable = "ServiceUnavailable"

	StatusCodeServerBusy   = http.StatusTooManyRequests
	StatusReasonServerBusy = "TooManyRequests"

	// StatusReasonNotImplemented means that particular functionality is not yet implemented.
	StatusCodeNotImplemented   = http.StatusNotImplemented
	StatusReasonNotImplemented = "NotImplemented"
)
