package concur

import (
	"fmt"
	"sync/atomic"
)

const (
	minCapacityLimit = 1
	maxCapacityLimit = 10000
)

type CapacityLimiter struct {
	limit   int32
	current int32
}

// Add increments counter by 1 and returns true if add succeeded, otherwise false.
func (cl *CapacityLimiter) Add() bool {
	for {
		v := atomic.LoadInt32(&cl.current)
		if v >= cl.limit {
			return false
		}
		if atomic.CompareAndSwapInt32(&cl.current, v, v+1) {
			return true
		}
	}
}

// Done decrements counter thus allowing requests to be processed.
func (cl *CapacityLimiter) Done() {
	atomic.AddInt32(&cl.current, -1)
}

// NewCapacityLimiter creates new limiter with provided maximum limit of in flight requests.
func NewCapacityLimiter(limit int) *CapacityLimiter {
	if limit < minCapacityLimit {
		panic(fmt.Sprintf("limit(%d) < minCapacityLimit(%d)", limit, minCapacityLimit))
	}
	if limit > maxCapacityLimit {
		panic(fmt.Sprintf("limit(%d) > maxCapacityLimit(%d)", limit, maxCapacityLimit))
	}
	return &CapacityLimiter{
		limit: int32(limit),
	}
}
