package service

import (
	"net/http"

	"a.yandex-team.ru/infra/nanny2/pkg/api"
	pb "a.yandex-team.ru/yp/go/proto/hq"
	"github.com/go-chi/chi/v5"
	"github.com/golang/protobuf/proto"
	"golang.org/x/net/context"
)

type federatedService struct {
	cStorage *Storage
}

type FederatedService interface {
	Mount(mux *chi.Mux)
	FindClusters(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status)
	GetCluster(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status)
}

func (srv *federatedService) FindClusters(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	return &pb.FindClustersResponse{Value: srv.cStorage.List()}, nil
}

func (srv *federatedService) GetCluster(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.GetClusterRequest)
	if len(req.Name) == 0 {
		return api.FailBadRequest("No name specified")
	}
	cluster := srv.cStorage.Get(req.Name)
	if cluster == nil {
		status := &pb.Status{
			Status: api.StatusFailure,
			Code:   api.StatusCodeNotFound,
			Reason: api.StatusReasonNotFound,
		}
		return nil, status
	}
	return &pb.GetClusterResponse{Value: cluster}, nil
}

func (srv *federatedService) Mount(m *chi.Mux) {
	rpc := api.NewRPCService("FederatedService", "/rpc/federated/")
	rpc.Route("FindClusters").
		To(srv.FindClusters).Accepts(pb.FindClustersRequest{})
	rpc.Route("GetCluster").
		To(srv.GetCluster).Accepts(pb.GetClusterRequest{})
	rpc.Mount(m)
}

func NewFederatedService(cStorage *Storage) FederatedService {
	return &federatedService{cStorage: cStorage}
}
