package hq

import (
	"fmt"
	"net/http"
	"os"
	"sync"
	"time"

	"a.yandex-team.ru/infra/nanny2/pkg/ypclient"

	"go.etcd.io/etcd/clientv3"
	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/nanny2/pkg/hq/cache"
	"a.yandex-team.ru/infra/nanny2/pkg/hq/housekeeping"
	"a.yandex-team.ru/infra/nanny2/pkg/hq/service"
	"a.yandex-team.ru/infra/nanny2/pkg/hq/storable"
	"a.yandex-team.ru/infra/nanny2/pkg/log"
	"a.yandex-team.ru/infra/nanny2/pkg/storage"
	"a.yandex-team.ru/infra/nanny2/pkg/storage/etcd3"
)

const (
	instancePrefix = "/instances/"
)

var (
	responseWriteTimeout = 60 * time.Second
	requestReadTimeout   = 10 * time.Second
	hostName             string
)

type Config struct {
	Listener             string
	EtcdEndpoints        []string
	OfflinePeriod        time.Duration
	OfflineStreak        int
	NoAuth               bool
	StaffToken           string
	StorageCodec         string
	YpClusters           []string
	FetchYpPodsBatchSize int32
}

func init() {
	h, err := os.Hostname()
	if err != nil {
		panic(err)
	}
	hostName = h
}

func Main(cfg *Config) error {
	// Setup etcd client
	etcdCfg := clientv3.Config{
		Endpoints: cfg.EtcdEndpoints,
	}
	c, err := clientv3.New(etcdCfg)
	if err != nil {
		log.Fatalf("Failed to create etcd client: %s", err.Error())
	}
	codec, err := storage.NewCodec(cfg.StorageCodec)
	if err != nil {
		return err
	}
	ypClient, err := ypclient.NewYpClient(cfg.YpClusters, cfg.FetchYpPodsBatchSize)
	if err != nil {
		return err
	}
	// Setup storage for API objects
	iStore := etcd3.NewStore(c, storable.InstanceCreator, instancePrefix, codec)
	ctx := context.Background()
	cancelContext, cancelFunc := context.WithCancel(ctx)
	// Run in-memory cache
	init := make(chan bool)
	idx := cache.NewIndex()
	d := cache.NewInstanceCache(iStore, idx)
	go d.Run(ctx, init)
	log.Info("Waiting until in-memory index populated...")
	<-init
	log.Info("Index built, will continue.")
	h := housekeeping.NewHouseKeeper(iStore,
		housekeeping.Config{
			OfflinePeriod: cfg.OfflinePeriod,
			OfflineStreak: cfg.OfflineStreak,
		})
	// Construct identifier for this instance
	id := fmt.Sprintf("%s@%s", hostName, cfg.Listener)
	go func() {
		housekeeping.LockAndRun(cancelContext, c, "/_leaders/housekeeper", id, h.Run)
	}()
	// RunFindServiceInstancesCallsCounterDumper
	var wg sync.WaitGroup
	dumpToFile, exists := os.LookupEnv("HQ_DUMP_FIND_INSTANCES_CALLS_COUNTER_TO_FILE")
	if exists {
		dumpInterval, exists := os.LookupEnv("HQ_DUMP_FIND_INSTANCES_CALLS_COUNTER_INTERVAL")
		if !exists {
			dumpInterval = "10m"
		}
		dumpIntervalDur, err := time.ParseDuration(dumpInterval)
		if err != nil {
			return err
		}
		wg.Add(1)
		go service.RunFindServiceInstancesCallsCounterDumper(cancelContext, dumpToFile, dumpIntervalDur, &wg)
	}
	// Setup HTTP server
	router := service.NewRouter(iStore, idx, ypClient)
	httpSrv := http.Server{
		Addr:         cfg.Listener,
		Handler:      router,
		ReadTimeout:  requestReadTimeout,
		WriteTimeout: responseWriteTimeout,
	}
	// Run
	log.Infof("Hi, I am HQ. And I am going to serve on %s", cfg.Listener)
	err = httpSrv.ListenAndServe()
	cancelFunc()
	wg.Wait()
	return err
}
