package cache

import (
	"fmt"
	"sync"

	"a.yandex-team.ru/infra/nanny2/pkg/storage"
	proto "a.yandex-team.ru/yp/go/proto/hq"
)

type InstanceStats struct {
	ReadyCount     int32
	InstalledCount int32
	TotalCount     int32
}

type RevisionStats map[string]InstanceStats

// A set of all object ids (e.g. instances) having particular attribute value.
// E.g. all instances with service id equal to "production_nanny".
type attrSet map[string]struct{}

// Index holding key - attribute value, value - set of instance ids having this attribute value
// If number of items in set reaches zero - we remove item from index.
type attrIndex map[string]attrSet

func (ai attrIndex) Update(value, id string) {
	s, ok := ai[value]
	if !ok {
		s = make(attrSet, 100)
		ai[value] = s
	}
	s[id] = struct{}{}
}

func (ai attrIndex) Delete(value, id string) {
	s, ok := ai[value]
	if ok {
		delete(s, id)
		if len(s) == 0 {
			delete(ai, value)
		}
	}
}

type HashmapIndex struct {
	mu        sync.RWMutex               // Guards all accesses to data structures
	byService attrIndex                  // Instance ids indexed by service id
	index     map[string]*proto.Instance // All instances indexed by id
	version   string                     // Last seen version
}

func NewIndex() *HashmapIndex {
	return &HashmapIndex{
		byService: make(attrIndex),
		index:     make(map[string]*proto.Instance),
		version:   "0",
	}
}

// Replace removes all items from index and replaces them with items from l.
func (idx *HashmapIndex) Replace(l []storage.Storable) {
	idx.mu.Lock()
	if len(idx.index) > 0 {
		idx.clear()
	}
	for _, raw := range l {
		m := raw.(*proto.Instance)
		idx.update(m)
	}
	idx.mu.Unlock()
}

func (idx *HashmapIndex) update(m *proto.Instance) {
	id := m.Meta.Id
	_, ok := idx.index[id]
	// If item is in hashmap index, then there is no need to update attr index
	// because serviceId cannot change
	if !ok {
		idx.byService.Update(m.Meta.ServiceId, id)
	}
	idx.index[id] = m
	idx.version = m.Meta.Version
}

// Update replaces/adds instance into index
func (idx *HashmapIndex) Update(m *proto.Instance) {
	idx.mu.Lock()
	idx.update(m)
	idx.mu.Unlock()
}

// Delete removes item from index and updates last seen version.
func (idx *HashmapIndex) Delete(name, revision string) {
	m, ok := idx.index[name]
	// If item is not in hashmap index, then there is no need to remove from other caches
	// because it should not be there.
	if !ok {
		idx.mu.Lock()
		idx.version = revision
		idx.mu.Unlock()
		return
	}
	idx.mu.Lock()
	idx.byService.Delete(m.Meta.ServiceId, m.Meta.Id)
	delete(idx.index, name)
	idx.version = revision
	idx.mu.Unlock()
}

// clear initializes all internals without holding (clients must do it themselves).
func (idx *HashmapIndex) clear() {
	for k := range idx.index {
		delete(idx.index, k)
	}
	idx.index = make(map[string]*proto.Instance, 100)
	idx.byService = make(attrIndex, 100)
	idx.version = "0"
}

// Clear removes all items from index
func (idx *HashmapIndex) Clear() {
	idx.mu.Lock()
	idx.clear()
	idx.mu.Unlock()
}

// LastIndex returns last seen version of storage
func (idx *HashmapIndex) LastIndex() string {
	return idx.version
}

// InstanceCount returns total number of stored objects
func (idx *HashmapIndex) InstanceCount() int {
	return len(idx.index)
}

// FindByService returns a list of instances filtered by provided predicates
func (idx *HashmapIndex) FindByService(serviceID string, readyOnly bool) ([]*proto.Instance, string) {
	r := make([]*proto.Instance, 0, 100)
	if len(serviceID) > 0 {
		idx.mu.RLock()
		s, ok := idx.byService[serviceID]
		// If we have set with instance ids for requested service
		// just iterate through whole set.
		if ok {
			for id := range s {
				m, ok := idx.index[string(id)]
				if !ok {
					idx.mu.RUnlock()
					panic("Failed to find " + id)
				}
				if readyOnly {
					if m.Status.Ready.Status == "True" {
						r = append(r, m)
					}
				} else {
					r = append(r, m)
				}
			}
		}
	} else {
		// No service id, need to iterate over all instances.
		idx.mu.RLock()
		for _, m := range idx.index {
			if readyOnly {
				if m.Status.Ready.Status == "True" {
					r = append(r, m)
				}
			} else {
				r = append(r, m)
			}
		}
	}
	idx.mu.RUnlock()
	return r, idx.version
}

// GetStatsByService returns statistics about instances by given service identifier.
func (idx *HashmapIndex) GetStatsByService(serviceID string) InstanceStats {
	stats := InstanceStats{}
	if len(serviceID) > 0 {
		idx.mu.RLock()
		s, ok := idx.byService[serviceID]
		// If we have set with instance ids for requested service
		// just iterate through whole set.
		if ok {
			for id := range s {
				m, ok := idx.index[string(id)]
				if !ok {
					idx.mu.RUnlock()
					panic("Failed to find " + id)
				}
				stats.TotalCount += 1
				if m.Status.Ready.Status == "True" {
					stats.ReadyCount += 1
				}
			}
		}
	} else {
		// No service id, need to iterate over all instances.
		idx.mu.RLock()
		stats.TotalCount = int32(len(idx.index))
		for _, m := range idx.index {
			if m.Status.Ready.Status == "True" {
				stats.ReadyCount += 1
			}
		}
	}
	idx.mu.RUnlock()
	return stats
}

// ListAttributeValues returns a list of available attribute values among all instances
func (idx *HashmapIndex) ListAttributeValues(attr string) ([]string, error) {
	if attr == "service_id" {
		idx.mu.RLock()
		l := make([]string, 0, len(idx.byService))
		for k := range idx.byService {
			l = append(l, k)
		}
		idx.mu.RUnlock()
		return l, nil
	} else {
		return nil, fmt.Errorf("unsupported attribute value=%s", attr)
	}
}

func (idx *HashmapIndex) GetRevisionStats(serviceID string) RevisionStats {
	idx.mu.RLock()
	stats := make(RevisionStats)
	s, ok := idx.byService[serviceID]
	if ok {
		for id := range s {
			m, ok := idx.index[string(id)]
			if !ok {
				idx.mu.RUnlock()
				panic("Failed to find " + id)
			}
			for _, rev := range m.Status.Revision {
				// We cannot take an address of struct,
				// so copy value here and assign back
				x := stats[rev.Id]
				x.TotalCount += 1
				if rev.Ready.Status == "True" {
					x.ReadyCount += 1
				}
				if rev.Installed.Status == "True" {
					x.InstalledCount += 1
				}
				stats[rev.Id] = x
			}
		}
	}
	idx.mu.RUnlock()
	return stats
}
