package cache

import (
	proto "a.yandex-team.ru/yp/go/proto/hq"
	"testing"
)

func TestIndex_GetStatsByService_EmptyCase(t *testing.T) {
	d := NewIndex()
	stats := d.GetStatsByService("")
	if stats.TotalCount != 0 || stats.ReadyCount != 0 {
		t.Fatalf("Wrong empty stats: %+v", stats)
	}
	defer d.Clear()
	// Add instances and check again
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "ws35-035@1041",
			ServiceId: "production_testing",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "True",
			},
		},
	})
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "sas1-0123@1041",
			ServiceId: "production_testing",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "False",
			},
		},
	})
	stats = d.GetStatsByService("")
	if stats.TotalCount != 2 || stats.ReadyCount != 1 {
		t.Fatalf("Wrong stats: %+v", stats)
	}
}

func TestIndex_GetStatsByService_WithServiceId(t *testing.T) {
	d := NewIndex()
	stats := d.GetStatsByService("production_testing")
	if stats.TotalCount != 0 || stats.ReadyCount != 0 {
		t.Fatalf("Wrong empty stats: %+v", stats)
	}
	// Add instances and check again
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "ws35-035@1041",
			ServiceId: "production_testing",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "True",
			},
		},
	})
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "sas1-0123@1041",
			ServiceId: "production_testing",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "False",
			},
		},
	})
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "sas1-9876@1041",
			ServiceId: "production_prestable",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "False",
			},
		},
	})
	stats = d.GetStatsByService("production_testing")
	if stats.TotalCount != 2 || stats.ReadyCount != 1 {
		t.Fatalf("Wrong stats: %+v", stats)
	}
	// Remove instance from another service
	d.Delete("sas1-9876@1041", "3")
	stats = d.GetStatsByService("production_testing")
	if stats.TotalCount != 2 || stats.ReadyCount != 1 {
		t.Fatalf("Wrong stats: %+v", stats)
	}
	// Remove instance from requested service
	d.Delete("sas1-0123@1041", "4")
	stats = d.GetStatsByService("production_testing")
	if stats.TotalCount != 1 || stats.ReadyCount != 1 {
		t.Fatalf("Wrong stats: %+v", stats)
	}
}

func TestHashmapIndex_GetRevisionStats(t *testing.T) {
	d := NewIndex()
	stats := d.GetRevisionStats("production_testing")
	if len(stats) != 0 {
		t.Fatal("Non empty stats!")
	}
	// Add instances and check again
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "ws35-035@1041",
			ServiceId: "production_testing",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "True",
			},
			Revision: []*proto.RevisionStatus{
				{
					Id: "1",
					Ready: &proto.Condition{
						Status: "True",
					},
					Installed: &proto.Condition{
						Status: "True",
					},
				},
			},
		},
	})
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "sas1-0123@1041",
			ServiceId: "production_testing",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "False",
			},
			Revision: []*proto.RevisionStatus{
				{
					Id: "1",
					Ready: &proto.Condition{
						Status: "False",
					},
					Installed: &proto.Condition{
						Status: "True",
					},
				},
			},
		},
	})
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "sas1-9876@1041",
			ServiceId: "production_testing",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "False",
			},
			Revision: []*proto.RevisionStatus{
				{
					Id: "2",
					Ready: &proto.Condition{
						Status: "False",
					},
					Installed: &proto.Condition{
						Status: "True",
					},
				},
			},
		},
	})
	stats = d.GetRevisionStats("production_testing")
	if l := len(stats); l != 2 {
		t.Fatalf("Invalid stats len! expected 2, got %d", l)
	}
	st, ok := stats["1"]
	if !ok {
		t.Fatal("Rev stats for '1' not found")
	}
	if st.TotalCount != 2 || st.ReadyCount != 1 {
		t.Fatalf("Wrong stats: %+v", st)
	}
	st, ok = stats["2"]
	if st.TotalCount != 1 || st.ReadyCount != 0 {
		t.Fatalf("Wrong stats: %+v", st)
	}
}

func TestIndex_FindByService(t *testing.T) {
	d := NewIndex()
	l, version := d.FindByService("production_testing", false)
	if len(l) != 0 {
		t.Fatalf("Non empty instance list: %+v", l)
	}
	if version != "0" {
		t.Fatalf("Wrong version: %v != '0'", version)
	}
	// Add instances and check again
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "ws35-035@1041",
			ServiceId: "production_testing",
			Version:   "2",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "True",
			},
		},
	})
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "sas1-0123@1041",
			ServiceId: "production_testing",
			Version:   "8",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "False",
			},
		},
	})
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "sas1-9876@1041",
			ServiceId: "production_prestable",
			Version:   "10",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "False",
			},
		},
	})
	curV := "10"
	l, version = d.FindByService("production_testing", true)
	if len(l) != 1 {
		t.Fatalf("Wrong result: %+v", l)
	}
	if need, id := "ws35-035@1041", l[0].Meta.Id; id != need {
		t.Fatalf("Wrong ready instance: need=%v, got=%v", need, id)
	}
	if version != curV {
		t.Fatalf("Wrong version: %v != '%v'", version, curV)
	}
	curV = "11"
	// Remove instance from another service
	d.Delete("sas1-9876@1041", curV)
	l, version = d.FindByService("production_testing", false)
	if version != curV {
		t.Fatalf("Wrong version: %v != %v", version, curV)
	}
	if len(l) != 2 {
		t.Fatalf("Wrong result: %+v", l)
	}
	curV = "100000"
	// Remove instance from requested service
	d.Delete("sas1-0123@1041", curV)
	l, version = d.FindByService("production_testing", false)
	if version != curV {
		t.Fatalf("Wrong version: %v != %v", version, curV)
	}
	if len(l) != 1 {
		t.Fatalf("Wrong result: %+v", l)
	}
}

func TestIndex_ListAttributeValues_Unsupported(t *testing.T) {
	d := NewIndex()
	defer d.Clear()
	_, err := d.ListAttributeValues("something wrong")
	if err == nil {
		t.Error("No error on invalid attribute")
	}
}

func TestIndex_ListAttributeValues(t *testing.T) {
	d := NewIndex()
	defer d.Clear()
	l, err := d.ListAttributeValues("service_id")
	if err != nil {
		t.Error(err.Error())
	}
	if len(l) != 0 {
		t.Error("Non empty result, unexpected")
	}
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "sas1-0123@1041",
			ServiceId: "production_testing",
			Version:   "8",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "False",
			},
		},
	})
	l, err = d.ListAttributeValues("service_id")
	if err != nil {
		t.Error(err.Error())
	}
	if len(l) != 1 {
		t.Error("Invalid result")
	}
	if l[0] != "production_testing" {
		t.Errorf("Unexpected value: got '%s', expected '%s'", l[0], "production_testing")
	}
	// Add another instance with separate id and service
	d.Update(&proto.Instance{
		Meta: &proto.InstanceMeta{
			Id:        "man1-0123@1041",
			ServiceId: "production_testing1",
			Version:   "8",
		},
		Status: &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status: "False",
			},
		},
	})
	l, err = d.ListAttributeValues("service_id")
	if err != nil {
		t.Error(err.Error())
	}
	if len(l) != 2 {
		t.Error("Invalid result")
	}
}
