package cache

import (
	"a.yandex-team.ru/infra/nanny2/pkg/log"
	"a.yandex-team.ru/infra/nanny2/pkg/storage"
	proto "a.yandex-team.ru/yp/go/proto/hq"
	"github.com/prometheus/client_golang/prometheus"
	"golang.org/x/net/context"
	"strconv"
	"time"
)

type CacheManager struct {
	idx              *HashmapIndex
	s                storage.Interface
	observedRevision prometheus.Gauge
	instanceGauge    prometheus.Gauge
}

func revisionStringToGauge(r string) float64 {
	v, err := strconv.ParseInt(r, 10, 64)
	if err != nil {
		panic(err)
	}
	return float64(v)
}

func (d *CacheManager) Run(ctx context.Context, init chan bool) {
	for ctx.Err() == nil {
		log.Info("Starting discovery watch")
		d.loop(ctx, init)
		log.Info("Discovery watch finished, will restart")
	}
	log.Errorf("Discovery update stopped: %s", ctx.Err().Error())
	d.idx.Clear()
}

func (d *CacheManager) Index() *HashmapIndex {
	return d.idx
}

func (d *CacheManager) Clear() {
	d.idx.Clear()
}

func (d *CacheManager) updateMetrics() {
	d.observedRevision.Set(revisionStringToGauge(d.idx.LastIndex()))
	d.instanceGauge.Set(float64(d.idx.InstanceCount()))
}

func (d *CacheManager) init(ctx context.Context) error {
	log.Info("Listing all instances...")
	l, err := d.s.List(ctx)
	if err != nil {
		return err
	}
	log.Infof("Listed %d instances.", len(l))
	d.idx.Replace(l)
	log.Info("Done updating index.")
	d.updateMetrics()
	return nil
}

func (d *CacheManager) loop(ctx context.Context, init chan bool) {
	err := d.init(ctx)
	if err != nil {
		log.Error(err.Error())
		return
	}
	log.Info("Signalling, that init done.")
	select {
	case init <- true:
		break
	default:
		break
	}
	w, err := d.s.WatchList(ctx, d.idx.LastIndex())
	if err != nil {
		log.Error(err.Error())
		return
	}
	defer w.Stop()
	for {
		select {
		case event, ok := <-w.ResultChan():
			if !ok {
				log.Info("Channel closed. Stopping.")
				return
			}
			switch event.Type {
			case storage.Added, storage.Modified:
				m := event.Object.(*proto.Instance)
				d.idx.Update(m)
				d.updateMetrics()
			case storage.Deleted:
				i := event.Object.(*proto.Instance)
				d.idx.Delete(i.Meta.Id, event.Revision)
				d.updateMetrics()
			case storage.Error:
				log.Errorf("Watch returned error: %s", event.Status.Message)
				log.Error("Will retry watching")
				time.Sleep(5 * time.Second)
				return
			}
		case <-ctx.Done():
			log.Info("Context done. Stopping loop...")
			return
		}
	}
}

func NewInstanceCache(s storage.Interface, idx *HashmapIndex) *CacheManager {
	revGauge := prometheus.NewGauge(prometheus.GaugeOpts{
		Namespace: "hq",
		Subsystem: "discoverer",
		Name:      "observed_revision",
		Help:      "Last observed revision from storage",
	})
	prometheus.MustRegister(revGauge)
	instanceGauge := prometheus.NewGauge(prometheus.GaugeOpts{
		Namespace: "hq",
		Subsystem: "discoverer",
		Name:      "instance_count",
		Help:      "Number of instance in memory index",
	})
	prometheus.MustRegister(instanceGauge)
	return &CacheManager{
		idx:              idx,
		s:                s,
		observedRevision: revGauge,
		instanceGauge:    instanceGauge,
	}
}
