package housekeeping

import (
	"a.yandex-team.ru/infra/nanny2/pkg/log"
	"a.yandex-team.ru/infra/nanny2/pkg/storage"
	pb "a.yandex-team.ru/yp/go/proto/hq"
	"github.com/golang/protobuf/ptypes/timestamp"
	"golang.org/x/net/context"
	"time"
)

type instanceOfflineFunc func(name string) error

type Config struct {
	OfflinePeriod time.Duration
	OfflineStreak int
}

type HouseKeeper struct {
	storage   storage.Interface
	config    Config
	offliner  *Offliner
	lastIndex string
}

func NewHouseKeeper(s storage.Interface, config Config) *HouseKeeper {
	keeper := &HouseKeeper{
		storage: s,
		config:  config,
	}
	keeper.offliner = NewOffliner(config.OfflinePeriod, config.OfflineStreak, keeper.offlineInstance)
	return keeper
}

func (h *HouseKeeper) removeInstance(name string) {
	err := h.storage.Delete(context.Background(), name, nil)
	if err != nil && storage.IsNotFound(err) {
		log.Errorf("Failed to remove %s: %s", name, err.Error())
		return
	}
}

func (h *HouseKeeper) offlineInstance(name string) error {
	m := &pb.Instance{}
	conflict := &pb.Instance{} // Filled if there was a conflict during update
	err := h.storage.Get(context.TODO(), name, m)
	if err != nil {
		log.Errorf("Failed to move node '%s' to offline: %s", name, err.Error())
		return err
	}
	for {
		if m.Status == nil {
			m.Status = &pb.InstanceStatus{}
		}
		if m.Status.Ready == nil {
			m.Status.Ready = &pb.Condition{}
		}
		now := &timestamp.Timestamp{Seconds: time.Now().Unix()}
		cond := &pb.Condition{
			Status:             readyFalse,
			LastTransitionTime: now,
			Reason:             reasonSilent,
			Message:            messageSilent,
		}
		m.Status.Ready = cond
		// Move all revisions to offline
		for _, rev := range m.Status.Revision {
			if rev.Ready.Status == "True" {
				rev.Ready = cond
				for _, c := range rev.Container {
					if c.Ready.Status == "True" {
						c.Ready = cond
					}
				}
			}
		}
		err = h.storage.UpdateIfMatch(context.TODO(), name, m, m.Meta.Version, conflict)
		if err != nil {
			if storage.IsTestFailed(err) {
				// Swap conflict with value and retry
				conflict, m = m, conflict
				conflict.Reset()
				continue
			}
			log.Errorf("Failed to expire %s: %s", name, err.Error())
		} else {
			log.Debugf("Set status 'Ready' to False: %s", name)
		}
		break
	}
	return nil
}

func (h *HouseKeeper) onInstanceChange(m *pb.Instance) {
	h.offliner.OnInstanceChange(m)
}

func (h *HouseKeeper) onInstanceRemove(name string) {
	log.Debugf("Removing %s", name)
	h.offliner.OnInstanceRemove(name)
}

func (h *HouseKeeper) loop(ctx context.Context) {
	w, err := h.storage.WatchList(ctx, "0")
	if err != nil {
		log.Error(err.Error())
		return
	}
	offlineTicker := time.NewTicker(5 * time.Second)

	defer w.Stop()
	defer offlineTicker.Stop()
	defer h.offliner.Clear() // Don't forget to clear data
	for {
		select {
		case event, ok := <-w.ResultChan():
			if !ok {
				log.Info("Channel closed. Stopping.")
				return
			}
			switch event.Type {
			case storage.Added, storage.Modified:
				m := event.Object.(*pb.Instance)
				h.onInstanceChange(m)
			case storage.Deleted:
				i := event.Object.(*pb.Instance)
				h.onInstanceRemove(i.Meta.Id)
			case storage.Error:
				log.Errorf("Watch returned error %s", event.Status)
				log.Error("Will retry...")
			}
		case <-ctx.Done():
			log.Info("Context done. Stopping loop...")
			return
		case <-offlineTicker.C:
			h.offliner.Run()
		}
	}
}

func (h *HouseKeeper) Run(ctx context.Context) {
	log.Infof("Housekeeper started with offline timeout %s and offline streak %d", h.config.OfflinePeriod, h.config.OfflineStreak)
	for ctx.Err() == nil {
		log.Info("Starting loop...")
		h.loop(ctx)
		log.Info("Loop finished...")
	}
	log.Info("Context done. Stopping...")
}
