/*
Proof of concept hand made leader election.
*/
package housekeeping

import (
	"time"

	"go.etcd.io/etcd/clientv3"
	"go.etcd.io/etcd/clientv3/concurrency"
	"golang.org/x/net/context"

	"a.yandex-team.ru/infra/nanny2/pkg/log"
)

type CancellableFunc func(context.Context)

func notFound(key string) clientv3.Cmp {
	return clientv3.Compare(clientv3.ModRevision(key), "=", 0)
}

func LockAndRun(ctx context.Context, c *clientv3.Client, name, id string, runnable CancellableFunc) {
	log.Infof("Started locking %s", name)
	for {
	session:
		cancelCtx, cancelFunc := context.WithCancel(ctx)
		log.Info("Creating session...")
		s, err := concurrency.NewSession(c)
		if err == nil {
			log.Infof("Created session with lease ID: %d", s.Lease())
			for {
				log.Debug("Iterating")
				txnResp, err := c.KV.Txn(ctx).If(
					notFound(name),
				).Then(
					clientv3.OpPut(name, id, clientv3.WithLease(s.Lease())),
				).Commit()
				if err != nil {
					if ctx.Err() != nil {
						// We have been cancelled
						return
					}
					// We might have disconnected and need to start a new session
					log.Errorf("Failed to create lock file: %s", err.Error())
					s.Orphan()
					time.Sleep(5 * time.Second) // Avoid busy looping
					goto session
				} else if !txnResp.Succeeded {
					log.Debug("Transaction failed, will retry...")
					time.Sleep(5 * time.Second)
				} else {
					break
				}
			}
			// Now we have created file with lease, assuming lock is held
			log.Infof("Assuming control over '%s', starting runnable", name)
			go runnable(cancelCtx)
			select {
			case <-ctx.Done(): // If initial context is Done - stop all activities
				log.Info("Lock and run cancelled, stop runnable...")
				cancelFunc()
				return
			case <-s.Done(): // If session is Done, just start all over again
				log.Info("Session is done, stop runnable...")
				cancelFunc()
				break
			}
			log.Info("Stopped runnable, will retry")
			continue
		} else if ctx.Err() != nil {
			log.Infof("Context cancelled %s", ctx.Err().Error())
		} else {
			log.Errorf("Failed to establish session: %s", err.Error())
			time.Sleep(5 * time.Second)
		}
	}
}
