/*
 * Watches for instances that did not report for too long and sets there ready state to False.
 */
package housekeeping

import (
	"a.yandex-team.ru/infra/nanny2/pkg/log"
	proto "a.yandex-team.ru/yp/go/proto/hq"
	"time"
)

var (
	readyFalse    = "False"
	reasonSilent  = "InstanceSilent"
	messageSilent = "Heartbeat lost"
)

type Offliner struct {
	items           *ExpirationQueue
	lastIndex       string
	offlineInstance instanceOfflineFunc
	timeout         time.Duration
	offlineStreak   int
}

func NewOffliner(timeout time.Duration, offlineStreak int, r instanceOfflineFunc) *Offliner {
	return &Offliner{
		offlineInstance: r,
		offlineStreak:   offlineStreak,
		timeout:         timeout,
		items:           NewQueue(),
	}
}

func (o *Offliner) OnInstanceChange(m *proto.Instance) {
	if m.Status == nil || m.Status.Ready == nil || m.Status.Ready.Status != "True" {
		o.items.RemoveByID(m.Meta.Id)
	} else {
		v := expireValue{ID: m.Meta.Id}
		o.items.UpdateValue(v, int(m.Status.LastHeartbeatTime.Seconds))
	}
}

func (o *Offliner) OnInstanceRemove(name string) {
	o.items.RemoveByID(name)
}

func (o *Offliner) Run() {
	now := time.Now()
	i := 0
	for ; i < o.offlineStreak; i++ {
		item, ok := o.items.Peak()
		if !ok {
			break
		}
		t := time.Unix(int64(item.priority), 0)
		if now.Sub(t) < o.timeout {
			break
		}
		err := o.offlineInstance(item.value.ID)
		if err == nil {
			o.items.DeleteMin()
		} else {
			break
		}

	}
	if i != 0 {
		log.Infof("Moved %d instances to offline", i)
	}
}

func (o *Offliner) Clear() {
	o.items.Clear()
}
