package housekeeping

import (
	"container/heap"
)

// Value being stored in priority queue. It contains only subset of fields needed to
// perform housekeeping, namely:
//  * ID - instance name
type expireValue struct {
	ID string
}

type expireItem struct {
	value    expireValue
	priority int
	index    int
}

type ExpirationQueue struct {
	// We need priority queue to check in O(1) if there is an item to expire
	pq []expireItem
	// We need a map from instance id to index in pq in order to quickly locate item and update
	// it's priority when we receive heartbeat
	posIndex map[string]int
}

func NewQueue() *ExpirationQueue {
	return &ExpirationQueue{
		pq:       make([]expireItem, 0, 100),
		posIndex: make(map[string]int, 100),
	}
}

func (eq ExpirationQueue) Len() int {
	return len(eq.pq)
}

func (eq ExpirationQueue) Less(i, j int) bool {
	return eq.pq[i].priority < eq.pq[j].priority
}

func (eq ExpirationQueue) Swap(i, j int) {
	eq.pq[i], eq.pq[j] = eq.pq[j], eq.pq[i]
	eq.pq[i].index = i
	eq.pq[j].index = j
	// Update name to index mapping too
	eq.posIndex[eq.pq[i].value.ID] = i
	eq.posIndex[eq.pq[j].value.ID] = j
}

// Implements heap.Interface.Push
func (eq *ExpirationQueue) Push(x interface{}) {
	n := len(eq.pq)
	item := x.(expireItem)
	item.index = n
	eq.pq = append(eq.pq, item)
	eq.posIndex[item.value.ID] = n
}

// Implements heap.Interface.Pop
func (eq *ExpirationQueue) Pop() interface{} {
	n := len(eq.pq)
	item := eq.pq[n-1]
	item.index = -1 // for safety
	eq.pq = eq.pq[0 : n-1]
	delete(eq.posIndex, item.value.ID)
	return item
}

// Remove by setting priority to -1 and popping.
func (eq *ExpirationQueue) RemoveByID(name string) {
	i, ok := eq.posIndex[name]
	if !ok {
		return
	}
	heap.Remove(eq, i)
}

func (eq *ExpirationQueue) UpdateValue(value expireValue, priority int) {
	i, ok := eq.posIndex[value.ID]
	if !ok {
		item := expireItem{
			value:    value,
			priority: priority,
			index:    -1,
		}
		heap.Push(eq, item)
	} else {
		// Update priority in place (we use copies) and fix heap (bubbling item up or down)
		eq.pq[i].priority = priority
		heap.Fix(eq, i)
	}
}

// Peak returns value with minimal priority
func (eq *ExpirationQueue) Peak() (expireItem, bool) {
	if len(eq.pq) == 0 {
		return expireItem{}, false
	}
	return eq.pq[0], true
}

// DeleteMin removes minimal item from queue. Can't use Pop as a name because it is used by heap.Pop
func (eq *ExpirationQueue) DeleteMin() {
	heap.Pop(eq)
}

// Clear removes all elements from queue
func (eq *ExpirationQueue) Clear() {
	eq.pq = eq.pq[:0]
	eq.posIndex = make(map[string]int, 100)
}
