/*
Helpers to avoid copy-paste.
*/
package service

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/infra/nanny2/pkg/api"
	"a.yandex-team.ru/infra/nanny2/pkg/log"
	"a.yandex-team.ru/infra/nanny2/pkg/ypclient"
	pb "a.yandex-team.ru/yp/go/proto/hq"
	"github.com/golang/protobuf/proto"
)

var statusNotImplemented = &pb.Status{
	Status:  api.StatusFailure,
	Code:    api.StatusCodeNotImplemented,
	Reason:  api.StatusReasonNotImplemented,
	Message: "This functionality is not yet implemented",
}

var (
	readyFalse     = "False"
	reasonCreated  = "InstanceCreated"
	messageCreated = "Instance created"
)

// FailNotImplemented returns API response tuple to signal that particular functionality is not implemented.
func FailNotImplemented() (proto.Message, *pb.Status) {
	return nil, statusNotImplemented
}

func hasInstancesWithSpecInYp(instances []*pb.Instance) bool {
	for _, i := range instances {
		if i.Meta.StoragePolicy != pb.StoragePolicy_SPEC_HQ_STATUS_HQ {
			return true
		}
	}
	return false
}

func makeIDToInstanceIndex(instances []*pb.Instance, skipGenCfgInstances bool) map[string]*pb.Instance {
	result := make(map[string]*pb.Instance, len(instances))
	for _, instance := range instances {
		if skipGenCfgInstances && strings.Contains(instance.Meta.Id, ":") {
			// skip gencfg instances (like "man1-8080.search.yandex.net:8083@prestable_nanny")
			continue
		}
		instanceID := strings.SplitN(instance.Meta.Id, "@", 2)[0]
		result[instanceID] = instance
	}
	return result
}

func makeIDToRevisionIndex(revisions []*pb.InstanceRevision) map[string]*pb.InstanceRevision {
	result := make(map[string]*pb.InstanceRevision, len(revisions))
	for _, rev := range revisions {
		result[rev.Id] = rev
	}
	return result
}

func makeIDToStatusIndex(statuses []*pb.RevisionStatus) map[string]*pb.RevisionStatus {
	result := make(map[string]*pb.RevisionStatus, len(statuses))
	for _, st := range statuses {
		result[st.Id] = st
	}
	return result
}

func findInstanceWithPod(idToInstanceIndex map[string]*pb.Instance, ypPod *ypclient.YpPod) *pb.Instance {
	if val, ok := idToInstanceIndex[*ypPod.Pod.Status.Dns.PersistentFqdn]; ok {
		return val
	}
	if val, ok := idToInstanceIndex[ypPod.Pod.Meta.Id]; ok {
		return val
	}
	return nil
}

func makeInstanceIDForYpPod(ypPod *ypclient.YpPod, serviceID string) string {
	instanceID := ""
	if ypPod.Cluster == "iva" || ypPod.Cluster == "myt" {
		instanceID = *ypPod.Pod.Status.Dns.PersistentFqdn
	} else {
		instanceID = ypPod.Pod.Meta.Id
	}
	return fmt.Sprintf("%s@%s", instanceID, serviceID)
}

func mergeInstanceWithYpPod(instance *pb.Instance, ypPod *ypclient.YpPod, createdCondition *pb.Condition) {
	if instance.Spec == nil {
		instance.Spec = &pb.InstanceSpec{
			Revision: make([]*pb.InstanceRevision, 0),
		}
	}
	if instance.Spec.Allocation == nil {
		instance.Spec.Allocation = &pb.ResourceAllocation{}
	}

	if instance.Spec.Allocation.Port == nil || len(instance.Spec.Allocation.Port) == 0 {
		instance.Spec.Allocation.Port = []*pb.Port{
			{Name: "default", Port: 80, Protocol: "TCP"},
		}
	}

	instance.Spec.NodeName = *ypPod.Pod.Spec.NodeId
	instance.Spec.Hostname = *ypPod.Pod.Status.Dns.PersistentFqdn

	// merge spec.revision
	idToRevisionIndex := makeIDToRevisionIndex(instance.Spec.Revision)
	for _, hostConfigurationInstance := range ypPod.Pod.Spec.Iss.Instances {
		if hostConfigurationInstance.InstanceRevision == nil {
			continue
		}
		if _, exists := idToRevisionIndex[hostConfigurationInstance.InstanceRevision.Id]; !exists {
			instance.Spec.Revision = append(instance.Spec.Revision, hostConfigurationInstance.InstanceRevision)
		}
	}

	// merge status.revision
	idToStatusIndex := makeIDToStatusIndex(instance.Status.Revision)
	for _, revision := range instance.Spec.Revision {
		if _, exists := idToStatusIndex[revision.Id]; !exists {
			rs := &pb.RevisionStatus{
				Id:        revision.Id,
				Installed: createdCondition,
				Ready:     createdCondition,
			}
			instance.Status.Revision = append(instance.Status.Revision, rs)
		}
	}
}

type Counter struct {
	mu sync.Mutex
	v  sync.Map
}

func (c *Counter) Get(key string) uint64 {
	v := uint64(0)
	value, ok := c.v.Load(key)
	if ok {
		v, _ = value.(uint64)
	}
	return v
}

func (c *Counter) Increment(key string) {
	c.mu.Lock()
	defer c.mu.Unlock()
	v := c.Get(key)
	c.v.Store(key, v+1)
}

func (c *Counter) Dump() map[string]uint64 {
	o := make(map[string]uint64)
	c.v.Range(func(key interface{}, value interface{}) bool {
		k, _ := key.(string)
		v, _ := value.(uint64)
		o[k] = v
		return true
	})
	return o
}

func (c *Counter) Clean() {
	c.mu.Lock()
	defer c.mu.Unlock()
	c.v = sync.Map{}
}

func NewCounter() *Counter {
	return &Counter{sync.Mutex{}, sync.Map{}}
}

var FindServiceInstancesCallsCounter = NewCounter()
var HeaderRtcClient = "x-rtc-client"

func IncrFindInstanceServiceCallsCount(serviceID string, header http.Header) bool {
	if serviceID == "" {
		return false
	}
	rtcClient := header.Get(HeaderRtcClient)

	if rtcClient != "" {
		return false
	}

	FindServiceInstancesCallsCounter.Increment(serviceID)
	return true
}

func DumpFindServiceInstancesCallsCounter(toFile string) error {
	f, err := os.OpenFile(toFile, os.O_RDWR|os.O_CREATE|os.O_TRUNC, 0755)
	if err != nil {
		return err
	}
	dump := FindServiceInstancesCallsCounter.Dump()
	enc := json.NewEncoder(f)
	enc.SetIndent("", "  ")

	if err := enc.Encode(dump); err != nil {
		return err
	}
	return f.Close()
}

func RunFindServiceInstancesCallsCounterDumper(ctx context.Context, toFile string, d time.Duration, wg *sync.WaitGroup) {
	defer wg.Done()
	shouldStop := false
	for {
		select {
		case <-time.After(d):
			// Do nothing.
		case <-ctx.Done():
			shouldStop = true
		}
		err := DumpFindServiceInstancesCallsCounter(toFile)
		if err != nil {
			log.Errorf("Failed dump FindServiceInstancesCallsCounter to file: %s", err.Error())
		}
		if shouldStop {
			return
		}
	}
}
