package service

import (
	"context"
	"io/ioutil"
	"net/http"
	"os"
	"sync"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
)

func TestCounter_Increment(t *testing.T) {
	c := NewCounter()
	c.Increment("someService")
	c.Increment("someService")
	c.Increment("someService")
	assert.Equal(t, c.Get("someService"), uint64(3))
}

func TestCounter_DumpAndClear(t *testing.T) {
	c := NewCounter()
	c.Increment("someService1")
	c.Increment("someService2")
	c.Increment("someService1")

	dump := c.Dump()
	dumpExpected := make(map[string]uint64)
	dumpExpected["someService1"] = uint64(2)
	dumpExpected["someService2"] = uint64(1)
	assert.Equal(t, dump, dumpExpected)
}

func TestIncrFindInstanceServiceCallsCount(t *testing.T) {
	header := http.Header{"X-Rtc-Client": {"any"}}
	emptyHeader := http.Header{}
	assert.False(t, IncrFindInstanceServiceCallsCount("testService", header))
	assert.True(t, IncrFindInstanceServiceCallsCount("testService", emptyHeader))
}

func TestDupmFindServiceInstancesCallsCounter(t *testing.T) {
	FindServiceInstancesCallsCounter.Clean()
	emptyHeader := http.Header{}
	IncrFindInstanceServiceCallsCount("testService1", emptyHeader)
	IncrFindInstanceServiceCallsCount("testService2", emptyHeader)
	IncrFindInstanceServiceCallsCount("testService3", emptyHeader)
	IncrFindInstanceServiceCallsCount("testService3", emptyHeader)
	IncrFindInstanceServiceCallsCount("testService3", emptyHeader)
	tempFile, err := ioutil.TempFile(os.TempDir(), "dump_test.*.txt")
	assert.NoError(t, err)
	toFile := tempFile.Name()
	err = DumpFindServiceInstancesCallsCounter(toFile)
	assert.NoError(t, err)
	res, err := os.ReadFile(toFile)
	assert.NoError(t, err)
	expectedContent := `{
	  "testService1": 1,
	  "testService2": 1,
	  "testService3": 3
	}`
	assert.JSONEq(t, expectedContent, string(res))

	assert.True(t, IncrFindInstanceServiceCallsCount("testService2", emptyHeader))
	err = DumpFindServiceInstancesCallsCounter(toFile)
	assert.NoError(t, err)
	res, err = os.ReadFile(toFile)
	assert.NoError(t, err)
	expectedContent = `{
		"testService1": 1,
		"testService2": 2,
		"testService3": 3
	}`
	assert.JSONEq(t, expectedContent, string(res))

}

func TestRunFindServiceInstancesCallsCounterDumper(t *testing.T) {
	FindServiceInstancesCallsCounter.Clean()
	ctx, cancelFunc := context.WithCancel(context.Background())
	tempFile, err := ioutil.TempFile(os.TempDir(), "dumper_test.*.txt")
	assert.NoError(t, err)
	toFile := tempFile.Name()
	d, err := time.ParseDuration("10m")
	assert.NoError(t, err)
	wg := sync.WaitGroup{}
	wg.Add(1)
	go RunFindServiceInstancesCallsCounterDumper(ctx, toFile, d, &wg)
	emptyHeader := http.Header{}
	IncrFindInstanceServiceCallsCount("testService1", emptyHeader)
	IncrFindInstanceServiceCallsCount("testService2", emptyHeader)
	IncrFindInstanceServiceCallsCount("testService3", emptyHeader)
	cancelFunc()
	wg.Wait()

	res, err := os.ReadFile(toFile)
	assert.NoError(t, err)
	expectedContent := `{
		"testService1": 1,
		"testService2": 1,
		"testService3": 1
	  }`
	assert.JSONEq(t, expectedContent, string(res))
}
