package service

import (
	"errors"
	"fmt"
	"net/http"
	"os"
	"strings"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/golang/protobuf/proto"
	"github.com/golang/protobuf/ptypes/timestamp"
	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/infra/nanny2/pkg/api"
	"a.yandex-team.ru/infra/nanny2/pkg/hq/cache"
	"a.yandex-team.ru/infra/nanny2/pkg/hq/validation"
	"a.yandex-team.ru/infra/nanny2/pkg/storage"
	"a.yandex-team.ru/infra/nanny2/pkg/ypclient"
	pb "a.yandex-team.ru/yp/go/proto/hq"
)

const (
	deleteConcurrencyLevel = 8
)

var timeNow = time.Now

type instanceService struct {
	store    storage.Interface
	idx      cache.InstanceIndex
	ypClient ypclient.YpClientInterface
}

type InstanceRPCService interface {
	Mount(mux *chi.Mux) bool
}

var (
	errNoRev = errors.New("NOREV")
)

func updateInstanceHostname(m *pb.Instance, req *pb.AddInstanceRevRequest) {
	// For details see: https://st.yandex-team.ru/SWAT-4126
	if req.Hostname == "" || req.HostnameVersion == nil {
		return
	}

	oldVer := m.Spec.HostnameVersion
	newVer := req.HostnameVersion
	if oldVer == nil {
		m.Spec.Hostname = req.Hostname
		m.Spec.HostnameVersion = req.HostnameVersion
		return
	}
	if newVer.Major == 0 && newVer.Minor == 0 {
		// We must overwrite hostname if version is not given
		m.Spec.Hostname = req.Hostname
		m.Spec.HostnameVersion = req.HostnameVersion
		return
	}
	if oldVer.Major < newVer.Major || oldVer.Major == newVer.Major && oldVer.Minor < newVer.Minor {
		m.Spec.Hostname = req.Hostname
		m.Spec.HostnameVersion = req.HostnameVersion
		return
	}
}

// updateInstanceReadyCondition update instance ready condition after some revision reported new status
func updateInstanceReadyCondition(m *pb.Instance, rev *pb.RevisionStatus) {
	// Reported revision is ready, set instance ready status
	if rev.Ready.Status == "True" {
		m.Status.Ready.Status = "True"
		m.Status.Ready.Reason = ""
		m.Status.Ready.Message = ""
		m.Status.LastHeartbeatTime.Seconds = time.Now().Unix()
	} else {
		for _, r := range m.Status.Revision {
			if r.Id == rev.Id {
				if r.Ready.Status == "True" {
					m.Status.Ready.Status = "False"
					m.Status.Ready.Reason = rev.Ready.Reason
					m.Status.Ready.Message = rev.Ready.Message
					m.Status.LastHeartbeatTime.Seconds = time.Now().Unix()
				}
				break
			}
		}
	}
}

func updateInstanceAfterRevStatusReport(m *pb.Instance, update *pb.RevisionStatus) {
	updateInstanceReadyCondition(m, update)
	// We need two separate loops:
	// * if reported status ready is True - need to set others to False
	// * otherwise - just find revision and update it
	// We ignore status report if there is no corresponding revision in instance status
	nowSec := time.Now().Unix()
	if update.Ready.Status == "True" {
		for i, rev := range m.Status.Revision {
			if rev.Id == update.Id {
				m.Status.Revision[i] = update
			} else if rev.Ready.Status == "True" {
				// Reported revision is ready
				// this revision cannot be active simultaneously
				rev.Ready.Status = "False"
				rev.Ready.LastTransitionTime.Seconds = nowSec
				// Update container statuses too
				for _, c := range rev.Container {
					if c.Ready.Status == "True" {
						c.Ready.Status = "False"
						c.Ready.LastTransitionTime.Seconds = nowSec
					}
				}
			}
		}
	} else {
		for i, rev := range m.Status.Revision {
			if rev.Id == update.Id {
				m.Status.Revision[i] = update
				break
			}
		}
	}
}

func addRevisionIfNotExists(m *pb.Instance, update *pb.RevisionStatus, addIfNotExists bool) bool {
	exists := false
	for _, rev := range m.Status.Revision {
		if rev.Id == update.Id {
			exists = true
			break
		}
	}
	if !exists && addIfNotExists {
		m.Status.Revision = append(m.Status.Revision, update)
		exists = true
	}
	return exists
}

// prepareInstanceUpdate properly sets fields in after
func prepareInstanceUpdate(before, after *pb.Instance) {
	// We do not allow changing meta data for now
	after.Meta = before.Meta
	after.Status = before.Status
}

func (srv *instanceService) GetInstance(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.GetInstanceRequest)
	err := validation.ValidateGetInstance(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	m := &pb.Instance{}
	err = srv.store.Get(ctx, req.Id, m)
	if err != nil {
		status := &pb.Status{Status: api.StatusFailure}
		if storage.IsNotFound(err) {
			status.Reason = api.StatusReasonNotFound
			status.Code = http.StatusNotFound

		} else {
			status.Reason = api.StatusReasonInternalError
			status.Code = http.StatusInternalServerError
		}
		return nil, status
	} else {
		return &pb.GetInstanceResponse{Instance: m}, nil
	}
}

func (srv *instanceService) GetInstanceWithProxyToYp(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.GetInstanceRequest)
	err := validation.ValidateGetInstance(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	m := &pb.Instance{}
	err = srv.store.Get(ctx, req.Id, m)
	if err != nil {
		status := &pb.Status{Status: api.StatusFailure}
		if storage.IsNotFound(err) {
			status.Reason = api.StatusReasonNotFound
			status.Code = http.StatusNotFound

		} else {
			status.Reason = api.StatusReasonInternalError
			status.Code = http.StatusInternalServerError
		}
		return nil, status
	}
	if m.Meta.StoragePolicy == pb.StoragePolicy_SPEC_HQ_STATUS_HQ {
		return &pb.GetInstanceResponse{Instance: m}, nil
	}

	ypClusters := srv.ypClient.GetClusters()
	instanceID := strings.SplitN(m.Meta.Id, "@", 2)[0]
	if len(ypClusters) > 1 {
		for _, cluster := range ypClusters {
			if strings.HasSuffix(instanceID, fmt.Sprintf("%s.yp-c.yandex.net", cluster)) {
				ypClusters = []string{cluster}
				break
			}
		}
	}
	podID := instanceID
	if strings.Contains(podID, ".") {
		podID = strings.SplitN(podID, ".", 2)[0]
	}
	ypPodCandidate, err := srv.ypClient.FindPod(ctx, podID, ypClusters...)
	if err != nil {
		return nil, &pb.Status{
			Status:  api.StatusFailure,
			Reason:  api.StatusReasonInternalError,
			Code:    http.StatusInternalServerError,
			Message: err.Error(),
		}
	}
	if ypPodCandidate == nil {
		return nil, &pb.Status{
			Status:  api.StatusFailure,
			Reason:  api.StatusReasonInternalError,
			Code:    http.StatusInternalServerError,
			Message: "Can't merge instance with yp pod, pod is not found",
		}
	}
	currentTimestamp := &timestamp.Timestamp{Seconds: timeNow().Unix()}
	createdCondition := &pb.Condition{
		Status:             readyFalse,
		Reason:             "Created",
		Message:            "Just created",
		LastTransitionTime: currentTimestamp,
	}
	mergeInstanceWithYpPod(m, ypPodCandidate, createdCondition)
	return &pb.GetInstanceResponse{Instance: m}, nil
}

func (srv *instanceService) FindInstances(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.FindInstancesRequest)
	serviceID := ""
	if req.Filter != nil && len(req.Filter.ServiceId) != 0 {
		serviceID = req.Filter.ServiceId
	}
	IncrFindInstanceServiceCallsCount(serviceID, header)
	readyOnly := false
	if req.Filter != nil {
		readyOnly = req.Filter.ReadyOnly
	}
	l, lastIndex := srv.idx.FindByService(serviceID, readyOnly)
	// Return as is if no mask specified
	if req.FieldMask == nil || len(req.FieldMask.Paths) == 0 {
		resp := &pb.FindInstancesResponse{
			Instance:  l,
			LastIndex: lastIndex,
		}
		return resp, nil
	}
	// Poor man's field mask application
	paths := req.FieldMask.Paths
	// Need to create a copy of instance list before filtering
	maskedList := make([]*pb.Instance, len(l))
	for i, m := range l {
		// Create new instance to copy only masked attributes
		spec := &pb.InstanceSpec{
			NodeName:   m.Spec.NodeName,
			Hostname:   m.Spec.Hostname,
			Allocation: m.Spec.Allocation,
		}
		v := &pb.Instance{
			Meta: &pb.InstanceMeta{
				Id:         m.Meta.Id,
				ServiceId:  m.Meta.ServiceId,
				Generation: m.Meta.Generation,
				Version:    m.Meta.Version,
			},
			Spec:   spec,
			Status: &pb.InstanceStatus{},
		}
		// Apply mask
		for _, path := range paths {
			switch path {
			case "status.ready":
				v.Status.Ready = m.Status.Ready
			case "status.revision":
				v.Status.Revision = m.Status.Revision
			case "status.last_heartbeat_time":
				v.Status.LastHeartbeatTime.Seconds = m.Status.LastHeartbeatTime.Seconds
			case "spec.revision.id":
				spec.Revision = make([]*pb.InstanceRevision, len(m.Spec.Revision))
				for i, rev := range m.Spec.Revision {
					spec.Revision[i] = &pb.InstanceRevision{
						Id:        rev.Id,
						Tags:      rev.Tags,
						ShardName: rev.ShardName,
						Hostname:  rev.Hostname,
					}
				}
			}
		}
		// Set element
		maskedList[i] = v
	}
	resp := &pb.FindInstancesResponse{
		Instance:  maskedList,
		LastIndex: lastIndex,
	}
	return resp, nil
}

func (srv *instanceService) FindInstancesWithProxyToYp(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.FindInstancesRequest)
	serviceID := ""
	if req.Filter != nil && len(req.Filter.ServiceId) != 0 {
		serviceID = req.Filter.ServiceId
	} else {
		return nil, &pb.Status{
			Status:  api.StatusFailure,
			Message: ".Filter.ServiceId is required",
			Reason:  api.StatusReasonBadRequest,
			Code:    http.StatusBadRequest,
		}
	}
	IncrFindInstanceServiceCallsCount(serviceID, header)

	readyOnly := false
	if req.Filter != nil {
		readyOnly = req.Filter.ReadyOnly
	}
	instances, lastIndex := srv.idx.FindByService(serviceID, readyOnly)

	// Alemate sets special env variable HQ_REPORT_VERSION = 2 when pushing spec only to YP.
	// In this case instancectl reports special storage policy SPEC_YP_STATUS_HQ based on which
	// we can say that at least some of instance specs stored only in YP, and cannot be found in HQ,
	// so we have to get specs from YP
	// We cannot determine this case before the first instancectl report storage policy = SPEC_YP_STATUS_HQ
	// of at least one of service instances . But we think that it's very rare corner case and we can ignore it.
	if hasInstancesWithSpecInYp(instances) {
		idToInstanceIndex := makeIDToInstanceIndex(instances, true)
		podSetID := strings.ReplaceAll(serviceID, "_", "-")
		ypPods, err := srv.ypClient.ListPods(ctx, podSetID)
		if err != nil {
			return nil, &pb.Status{
				Status:  api.StatusFailure,
				Message: err.Error(),
				Reason:  api.StatusReasonInternalError,
				Code:    http.StatusInternalServerError,
			}
		}
		for _, ypPod := range ypPods {
			if len(ypPod.Pod.Spec.Iss.Instances) == 0 {
				continue
			}
			instance := findInstanceWithPod(idToInstanceIndex, ypPod)
			currentTimestamp := &timestamp.Timestamp{Seconds: timeNow().Unix()}
			createdCondition := &pb.Condition{
				Status:             readyFalse,
				Reason:             "Created",
				Message:            "Just created",
				LastTransitionTime: currentTimestamp,
			}
			if instance == nil {
				// We found pod spec in YP but there is no corresponding status in HQ,
				// it's valid situation, we must add it to response, because instance may exist
				// in YP but it has not reported its status yet
				instance = &pb.Instance{
					Meta: &pb.InstanceMeta{
						Id:            makeInstanceIDForYpPod(ypPod, serviceID),
						ServiceId:     serviceID,
						StoragePolicy: pb.StoragePolicy_SPEC_YP_STATUS_HQ,
					},
					Status: &pb.InstanceStatus{
						Ready:             createdCondition,
						LastHeartbeatTime: currentTimestamp,
						Revision:          make([]*pb.RevisionStatus, 0),
					},
				}
				instances = append(instances, instance)
			}
			mergeInstanceWithYpPod(instance, ypPod, createdCondition)
		}
		j := 0
		for i := range instances {
			revisionEmpty := instances[i].Spec == nil ||
				instances[i].Spec.Revision == nil ||
				len(instances[i].Spec.Revision) == 0
			if !revisionEmpty {
				instances[j] = instances[i]
				j++
			}
		}
		instances = instances[:j]
	}

	if req.FieldMask == nil || len(req.FieldMask.Paths) == 0 {
		resp := &pb.FindInstancesResponse{
			// Return as is if no mask specified
			Instance:  instances,
			LastIndex: lastIndex,
		}
		return resp, nil
	}
	// Poor man's field mask application
	paths := req.FieldMask.Paths
	// Need to create a copy of instance list before filtering
	maskedList := make([]*pb.Instance, len(instances))
	for i, m := range instances {
		// Create new instance to copy only masked attributes
		spec := &pb.InstanceSpec{
			NodeName:   m.Spec.NodeName,
			Hostname:   m.Spec.Hostname,
			Allocation: m.Spec.Allocation,
		}
		v := &pb.Instance{
			Meta: &pb.InstanceMeta{
				Id:         m.Meta.Id,
				ServiceId:  m.Meta.ServiceId,
				Generation: m.Meta.Generation,
				Version:    m.Meta.Version,
			},
			Spec:   spec,
			Status: &pb.InstanceStatus{},
		}
		// Apply mask
		for _, path := range paths {
			switch path {
			case "status.ready":
				v.Status.Ready = m.Status.Ready
			case "status.revision":
				v.Status.Revision = m.Status.Revision
			case "status.last_heartbeat_time":
				v.Status.LastHeartbeatTime.Seconds = m.Status.LastHeartbeatTime.Seconds
			case "spec.revision.id":
				spec.Revision = make([]*pb.InstanceRevision, len(m.Spec.Revision))
				for i, rev := range m.Spec.Revision {
					spec.Revision[i] = &pb.InstanceRevision{
						Id:        rev.Id,
						Tags:      rev.Tags,
						ShardName: rev.ShardName,
						Hostname:  rev.Hostname,
					}
				}
			}
		}
		// Set element
		maskedList[i] = v
	}
	resp := &pb.FindInstancesResponse{
		Instance:  maskedList,
		LastIndex: lastIndex,
	}
	return resp, nil
}

func (srv *instanceService) deleteRevision(ctx context.Context, instanceID, rev string) error {
	wasLast := false
	err := srv.store.GuaranteedUpdate(ctx, instanceID, func(obj storage.Storable) error {
		m := obj.(*pb.Instance)
		// Remove from spec
		j := 0
		exists := false
		specEmpty := m.Spec == nil || m.Spec.Revision == nil
		if !specEmpty {
			for i := range m.Spec.Revision {
				if m.Spec.Revision[i].Id != rev {
					m.Spec.Revision[j] = m.Spec.Revision[i]
					j++
				} else {
					exists = true
				}
			}
			m.Spec.Revision = m.Spec.Revision[:j]
			if len(m.Spec.Revision) == 0 {
				specEmpty = true
			}
		}
		// Remove from status
		j = 0
		for i := range m.Status.Revision {
			if m.Status.Revision[i].Id != rev {
				m.Status.Revision[j] = m.Status.Revision[i]
				j++
			} else {
				exists = true
			}
		}
		if !exists {
			// Return special error to avoid updating instance in storage
			return errNoRev
		}
		m.Status.Revision = m.Status.Revision[:j]
		if specEmpty {
			if m.Meta.StoragePolicy == pb.StoragePolicy_SPEC_HQ_STATUS_HQ {
				// Remove instance only if .Spec.Revision became empty (default behavior)
				wasLast = true
			} else if len(m.Status.Revision) == 0 {
				// With StoragePolicy SPEC_YP_STATUS_HQ .Spec.Revision maybe already empty
				// remove instance only if .Status.Revision became empty
				wasLast = true
			}
		}
		return nil
	})
	if err != nil && (err == errNoRev || storage.IsNotFound(err)) {
		return nil
	}

	// Need to remove instance itself if it was last revision
	if wasLast {
		err = srv.store.Delete(ctx, instanceID, nil)
		if err != nil && storage.IsNotFound(err) {
			return nil
		}
	}
	return err
}

// splitInstances returns a slice of jobs to be run in parallel
func splitInstances(s []*pb.Instance, parts int) [][]*pb.Instance {
	a := len(s) / parts
	if a == 0 {
		return [][]*pb.Instance{s}
	}
	rv := make([][]*pb.Instance, 0)
	i := 0
	for ; i < parts-1; i++ {
		rv = append(rv, s[a*i:a*(i+1)])
	}
	rv = append(rv, s[a*i:])
	return rv
}

func (srv *instanceService) DeleteServiceRev(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.DeleteServiceRevRequest)
	if req.ServiceId == "" {
		return api.FailBadRequest("No 'service_id' provided")
	}
	if req.Rev == "" {
		return api.FailBadRequest("Not 'rev' provided")
	}
	// Use in memory index to find service instances
	instances, _ := srv.idx.FindByService(req.ServiceId, false)
	if len(instances) == 0 {
		return &pb.DeleteServiceRevResponse{}, nil
	}
	jobs := splitInstances(instances, deleteConcurrencyLevel)
	g, gCtx := errgroup.WithContext(ctx)
	for _, job := range jobs {
		job := job
		g.Go(func() error {
			for _, i := range job {
				if i.Meta.StoragePolicy == pb.StoragePolicy_SPEC_HQ_STATUS_HQ && i.Spec == nil {
					// Sanity check against bad data (should not happen)
					continue
				}
				if err := srv.deleteRevision(gCtx, i.Meta.Id, req.Rev); err != nil {
					return err
				}
			}
			return nil
		})
	}
	if err := g.Wait(); err != nil {
		return api.FailInternalError(err.Error())
	}
	return &pb.DeleteServiceRevResponse{}, nil
}

func (srv *instanceService) UpdateInstance(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.UpdateInstanceRequest)
	err := validation.ValidateUpdateRequest(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	before := &pb.Instance{}
	after := &pb.Instance{
		Meta: req.Meta,
		Spec: req.Spec,
	}
	err = srv.store.Get(ctx, req.Meta.Id, before)
	if err != nil {
		if storage.IsNotFound(err) {
			// Validate and create as is
			if err = validation.ValidateInstanceCreation(after); err != nil {
				return api.FailBadRequest(err.Error())
			}
			err = srv.store.Create(ctx, after.Meta.Id, after, after)
			if err != nil {
				return api.FailInternalError(err.Error())
			}
			return after, nil
		}
		return api.FailInternalError(err.Error())
	}
	// We have existing instance, need to update
	prepareInstanceUpdate(before, after)
	err = srv.store.UpdateIfMatch(ctx, after.Meta.Id, after, req.Meta.Version, before)
	if err != nil {
		status := &pb.Status{Status: api.StatusFailure}
		if storage.IsTestFailed(err) {
			status.Code = api.StatusCodeConflict
			status.Reason = api.StatusReasonConflict
			status.Message = err.Error()
		} else if storage.IsNotFound(err) {
			status.Code = api.StatusCodeNotFound
			status.Reason = api.StatusReasonNotFound
			status.Message = "Instance " + after.Meta.Id + " not found"
		} else {
			status.Code = api.StatusCodeInternalError
			status.Reason = api.StatusReasonInternalError
			status.Message = err.Error()
		}
		return nil, status
	}
	return &pb.UpdateInstanceResponse{Instance: after}, nil
}

func (srv *instanceService) UpdateInstanceAllocation(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.UpdateInstanceAllocationRequest)
	err := validation.ValidateUpdateInstanceAllocationRequest(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	err = srv.store.GuaranteedUpdate(ctx, req.Id, func(obj storage.Storable) error {
		m := obj.(*pb.Instance)
		if m.Spec != nil {
			m.Spec.NodeName = req.NodeName
		}
		return nil
	})
	if err != nil {
		if storage.IsNotFound(err) {
			// Ignore not found
			return &pb.UpdateInstanceAllocationResponse{}, nil
		} else {
			status := &pb.Status{Status: api.StatusFailure}
			status.Code = api.StatusCodeInternalError
			status.Reason = api.StatusReasonInternalError
			status.Message = err.Error()
			return nil, status
		}
	}
	return &pb.UpdateInstanceAllocationResponse{}, nil
}

func (srv *instanceService) AddInstanceRev(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.AddInstanceRevRequest)
	err := validation.ValidateAddInstanceRev(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	err = srv.store.GuaranteedUpdate(ctx, req.Id, func(obj storage.Storable) error {
		m := obj.(*pb.Instance)
		// For case when export spec to hq enabled again
		if m.Spec == nil {
			m.Spec = &pb.InstanceSpec{
				Revision: make([]*pb.InstanceRevision, 0, 1),
			}
		}
		updateInstanceHostname(m, req)
		for _, rev := range m.Spec.Revision {
			if rev.Id == req.Rev.Id {
				// It might be better to compare existing and requested revisions content
				// and response OK or CONFLICT depending on comparison result
				return storage.NewResourceContentConflictsError(m.Meta.Id, "Revision exists already")
			}
		}
		m.Spec.Revision = append(m.Spec.Revision, req.Rev)
		// Add revision to status too
		c := &pb.Condition{
			Status:             readyFalse,
			Reason:             reasonCreated,
			Message:            messageCreated,
			LastTransitionTime: &timestamp.Timestamp{Seconds: time.Now().Unix()},
		}
		m.Status.Revision = append(m.Status.Revision, &pb.RevisionStatus{
			Id:        req.Rev.Id,
			Ready:     c,
			Installed: c,
		})
		return nil
	})
	if err != nil {
		status := &pb.Status{Status: api.StatusFailure}
		if storage.IsResourceContentConflict(err) {
			status.Code = api.StatusCodeConflict
			status.Reason = api.StatusReasonConflict
			status.Message = "Revision exists already"
		} else if storage.IsNotFound(err) {
			status.Code = api.StatusCodeNotFound
			status.Reason = api.StatusReasonNotFound
			status.Message = "Instance not found"
		} else {
			status.Code = api.StatusCodeInternalError
			status.Reason = api.StatusReasonInternalError
			status.Message = err.Error()
		}
		return nil, status
	}
	return &pb.AddInstanceRevResponse{}, nil
}

func (srv *instanceService) CreateInstance(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.CreateInstanceRequest)
	err := validation.ValidateCreateRequest(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	t := &timestamp.Timestamp{Seconds: time.Now().Unix()}
	m := &pb.Instance{
		Meta: req.Meta,
		Spec: req.Spec,
		Status: &pb.InstanceStatus{
			Ready: &pb.Condition{
				Status:             readyFalse,
				Reason:             reasonCreated,
				Message:            messageCreated,
				LastTransitionTime: t,
			},
			LastHeartbeatTime: t,
			Revision:          make([]*pb.RevisionStatus, 0, 1),
		},
	}
	c := &pb.Condition{
		Status:             readyFalse,
		Reason:             "Created",
		Message:            "Just created",
		LastTransitionTime: t,
	}
	// Fill in statuses for revisions in spec
	for _, r := range m.Spec.Revision {
		rs := &pb.RevisionStatus{
			Id:        r.Id,
			Installed: c,
			Ready:     c,
		}
		m.Status.Revision = append(m.Status.Revision, rs)
	}
	err = srv.store.Create(ctx, m.Meta.Id, m, nil)
	if err != nil {
		status := &pb.Status{Status: api.StatusFailure}
		if storage.IsNodeExist(err) {
			status.Reason = api.StatusReasonAlreadyExists
			status.Code = api.StatusCodeAlreadyExists
			status.Message = "Instance already exists"
		} else {
			status.Reason = api.StatusReasonServiceUnavailable
			status.Code = api.StatusCodeInternalError
			status.Message = err.Error()
		}
		return nil, status
	}
	return &pb.CreateInstanceResponse{Instance: m}, nil
}

func (srv *instanceService) DeleteInstance(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.DeleteInstanceRequest)
	if req.Name == "" {
		return api.FailBadRequest("No instance name specified")
	}
	m := &pb.Instance{}
	err := srv.store.Delete(ctx, req.Name, m)
	if err != nil {
		status := &pb.Status{Status: api.StatusFailure}
		if storage.IsNotFound(err) {
			status.Reason = api.StatusReasonNotFound
			status.Code = http.StatusNotFound
		} else {
			status.Reason = api.StatusReasonInternalError
			status.Code = http.StatusInternalServerError
		}
		return nil, status
	} else {
		return &pb.DeleteInstanceResponse{Instance: m}, nil
	}
}

func (srv *instanceService) GetInstanceStats(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.GetInstanceStatsRequest)
	serviceID := ""
	if req.Filter != nil && len(req.Filter.ServiceId) != 0 {
		serviceID = req.Filter.ServiceId
	}
	stats := srv.idx.GetStatsByService(serviceID)
	return &pb.GetInstanceStatsResponse{
		ReadyCount: stats.ReadyCount,
		TotalCount: stats.TotalCount,
	}, nil

}

func (srv *instanceService) ReportInstanceRevStatus(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.ReportInstanceRevStatusRequest)
	err := validation.ValidateRevStatusReport(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	err = srv.store.GuaranteedUpdate(ctx, req.InstanceId, func(obj storage.Storable) error {
		m := obj.(*pb.Instance)
		m.Meta.StoragePolicy = pb.StoragePolicy_SPEC_HQ_STATUS_HQ
		updateInstanceAfterRevStatusReport(m, req.Status)
		return nil
	})
	if err != nil {
		status := &pb.Status{Status: api.StatusFailure}
		if storage.IsNotFound(err) {
			status.Reason = api.StatusReasonNotFound
			status.Code = http.StatusNotFound
			status.Message = "Instance not found"
		} else {
			status.Reason = api.StatusReasonInternalError
			status.Code = http.StatusInternalServerError
			status.Message = err.Error()
		}
		return nil, status
	}
	return &pb.ReportInstanceRevStatusResponse{}, nil
}

func (srv *instanceService) ReportInstanceRevStatusV2(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.ReportInstanceRevStatusV2Request)
	createInstanceIfNotExists := false
	for _, e := range req.Policies {
		if e == pb.ReportInstanceRevStatusPolicy_CREATE_INSTANCE_IF_NOT_EXISTS {
			createInstanceIfNotExists = true
			break
		}
	}
	err := validation.ValidateRevStatusReportV2(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	err = srv.store.GuaranteedUpdate(ctx, req.InstanceId, func(obj storage.Storable) error {
		m := obj.(*pb.Instance)
		m.Meta.StoragePolicy = pb.StoragePolicy_SPEC_YP_STATUS_HQ
		revStatusExists := addRevisionIfNotExists(m, req.Status, createInstanceIfNotExists)
		if !revStatusExists {
			return nil
		}
		updateInstanceAfterRevStatusReport(m, req.Status)
		return nil
	})
	if err == nil {
		return &pb.ReportInstanceRevStatusResponse{}, nil
	}

	status := &pb.Status{Status: api.StatusFailure}
	if storage.IsNotFound(err) && createInstanceIfNotExists {
		m := &pb.Instance{
			Meta: &pb.InstanceMeta{
				Id:        req.InstanceId,
				ServiceId: req.ServiceId,
			},
			Status: &pb.InstanceStatus{
				Ready:             &pb.Condition{},
				Revision:          []*pb.RevisionStatus{req.Status},
				LastHeartbeatTime: &timestamp.Timestamp{Seconds: time.Now().Unix()},
			},
		}

		if req.Status.Ready.Status == "True" {
			m.Status.Ready.Status = "True"
			m.Status.Ready.Reason = ""
			m.Status.Ready.Message = ""
		} else {
			m.Status.Ready.Status = "False"
			m.Status.Ready.Reason = req.Status.Ready.Reason
			m.Status.Ready.Message = req.Status.Ready.Message
		}

		err = srv.store.Create(ctx, m.Meta.Id, m, nil)
		if err == nil {
			return &pb.ReportInstanceRevStatusResponse{}, nil
		}

		if storage.IsNodeExist(err) {
			status.Reason = api.StatusReasonConflict
			status.Code = api.StatusCodeConflict
			status.Message = "Instance already exists"
		} else {
			status.Reason = api.StatusReasonServiceUnavailable
			status.Code = api.StatusCodeInternalError
			status.Message = err.Error()
		}

	} else if storage.IsNotFound(err) && !createInstanceIfNotExists {
		status.Reason = api.StatusReasonNotFound
		status.Code = http.StatusNotFound
		status.Message = "Instance not found"
	} else {
		status.Reason = api.StatusReasonInternalError
		status.Code = http.StatusInternalServerError
		status.Message = err.Error()
	}
	return nil, status
}

func (srv *instanceService) GetInstanceRev(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.GetInstanceRevRequest)
	err := validation.ValidateGetInstanceRev(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	m := &pb.Instance{}
	err = srv.store.Get(ctx, req.Id, m)
	if err != nil {
		status := &pb.Status{Status: api.StatusFailure}
		if storage.IsNotFound(err) {
			status.Reason = api.StatusReasonNotFound
			status.Code = http.StatusNotFound
		} else {
			status.Reason = api.StatusReasonInternalError
			status.Code = http.StatusInternalServerError
		}
		return nil, status
	}
	// Try to find revision in instance spec
	for _, rev := range m.Spec.Revision {
		if rev.Id == req.Rev {
			return &pb.GetInstanceRevResponse{Revision: rev}, nil
		}
	}
	return api.FailNotFound(req.Rev + " not found")
}

func (srv *instanceService) ListAttributeValues(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.ListAttributeValuesRequest)
	err := validation.ValidateListAttributeValues(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	l, err := srv.idx.ListAttributeValues(req.Attr)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	return &pb.ListAttributeValuesResponse{Values: l}, nil
}

func (srv *instanceService) GetRevisionStats(ctx context.Context, in interface{}, header http.Header) (proto.Message, *pb.Status) {
	req := in.(*pb.GetRevisionStatsRequest)
	err := validation.ValidateGetRevisionStats(req)
	if err != nil {
		return api.FailBadRequest(err.Error())
	}
	stats := srv.idx.GetRevisionStats(req.Filter.ServiceId)
	resp := &pb.GetRevisionStatsResponse{
		Stats: make([]*pb.RevisionStats, 0, len(stats)),
	}
	for rev, st := range stats {
		resp.Stats = append(resp.Stats, &pb.RevisionStats{
			Revision:       rev,
			ReadyCount:     st.ReadyCount,
			InstalledCount: st.InstalledCount,
			TotalCount:     st.TotalCount,
		})
	}
	return resp, nil
}

func (srv *instanceService) Mount(mux *chi.Mux) bool {
	rpc := api.NewRPCService("InstanceService", "/rpc/instances/")

	_, ProxyToYp := os.LookupEnv("ENABLE_PROXY_TO_YP")

	GetInstanceFunc := srv.GetInstance
	FindInstancesFunc := srv.FindInstances
	FindInstancesTimeout := 5 * time.Second
	FindInstancesLimits := 10

	if ProxyToYp {
		GetInstanceFunc = srv.GetInstanceWithProxyToYp
		FindInstancesFunc = srv.FindInstancesWithProxyToYp
		FindInstancesTimeout = 15 * time.Second
		FindInstancesLimits = 20
	}

	// login validation disables if env variable is not set
	nannyRobotLogin := os.Getenv("NANNY_ROBOT_LOGIN")

	rpc.Route("GetInstance").
		WithLimit(100).
		WithTimeout(5 * time.Second).
		To(GetInstanceFunc).
		Accepts(pb.GetInstanceRequest{})

	rpc.Route("FindInstances").
		WithLimit(FindInstancesLimits).
		WithTimeout(FindInstancesTimeout).
		To(FindInstancesFunc).
		Accepts(pb.FindInstancesRequest{})

	rpc.Route("GetInstanceWithProxyToYp").
		WithLimit(100).
		WithTimeout(5 * time.Second).
		To(srv.GetInstanceWithProxyToYp).
		Accepts(pb.GetInstanceRequest{})

	rpc.Route("FindInstancesWithProxyToYp").
		WithLimit(20).
		WithTimeout(15 * time.Second).
		To(srv.FindInstancesWithProxyToYp).
		Accepts(pb.FindInstancesRequest{})

	rpc.Route("UpdateInstance").
		WithLimit(100).
		WithTimeout(5 * time.Second).
		To(srv.UpdateInstance).
		Accepts(pb.UpdateInstanceRequest{}).
		DemandsLogin(nannyRobotLogin)
	rpc.Route("UpdateInstanceAllocation").
		WithLimit(100).
		WithTimeout(5 * time.Second).
		To(srv.UpdateInstanceAllocation).
		Accepts(pb.UpdateInstanceAllocationRequest{}).
		DemandsLogin(nannyRobotLogin)
	rpc.Route("CreateInstance").
		WithLimit(100).
		WithTimeout(5 * time.Second).
		To(srv.CreateInstance).
		Accepts(pb.CreateInstanceRequest{}).
		DemandsLogin(nannyRobotLogin)
	rpc.Route("DeleteInstance").
		WithLimit(100).
		WithTimeout(5 * time.Second).
		To(srv.DeleteInstance).
		Accepts(pb.DeleteInstanceRequest{}).
		DemandsLogin(nannyRobotLogin)
	rpc.Route("GetInstanceStats").
		WithLimit(30).
		WithTimeout(5 * time.Second).
		To(srv.GetInstanceStats).
		Accepts(pb.GetInstanceStatsRequest{})
	rpc.Route("ReportInstanceRevStatus").
		WithLimit(160).
		WithTimeout(5 * time.Second).
		To(srv.ReportInstanceRevStatus).
		Accepts(pb.ReportInstanceRevStatusRequest{})
	rpc.Route("ReportInstanceRevStatusV2").
		WithLimit(160).
		WithTimeout(5 * time.Second).
		To(srv.ReportInstanceRevStatusV2).
		Accepts(pb.ReportInstanceRevStatusV2Request{})
	rpc.Route("GetInstanceRev").
		WithLimit(100).
		WithTimeout(5 * time.Second).
		To(srv.GetInstanceRev).
		Accepts(pb.GetInstanceRevRequest{})
	rpc.Route("AddInstanceRev").
		WithLimit(100).
		WithTimeout(5 * time.Second).
		To(srv.AddInstanceRev).
		Accepts(pb.AddInstanceRevRequest{}).
		DemandsLogin(nannyRobotLogin)
	rpc.Route("DeleteServiceRev").
		WithLimit(30).
		WithTimeout(300 * time.Second).
		To(srv.DeleteServiceRev).
		Accepts(pb.DeleteServiceRevRequest{}).
		DemandsLogin(nannyRobotLogin)
	rpc.Route("ListAttributeValues").
		WithLimit(100).
		WithTimeout(10 * time.Second).
		To(srv.ListAttributeValues).
		Accepts(pb.ListAttributeValuesRequest{})
	rpc.Route("GetRevisionStats").
		WithLimit(100).
		WithTimeout(5 * time.Second).
		To(srv.GetRevisionStats).
		Accepts(pb.GetRevisionStatsRequest{})
	rpc.Mount(mux)

	return ProxyToYp
}

func NewInstanceRPCService(iStore storage.Interface, idx cache.InstanceIndex,
	ypClient ypclient.YpClientInterface) InstanceRPCService {
	return &instanceService{store: iStore, idx: idx, ypClient: ypClient}
}
