package service

import (
	"context"
	"net/http"
	"os"
	"strconv"
	"strings"
	"testing"

	"a.yandex-team.ru/infra/nanny2/pkg/ypclient"
	"a.yandex-team.ru/yp/go/proto/clusterapi"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"github.com/go-chi/chi/v5"

	"github.com/golang/protobuf/proto"
	"github.com/golang/protobuf/ptypes/timestamp"
	"github.com/stretchr/testify/assert"

	"a.yandex-team.ru/infra/nanny2/pkg/api"
	"a.yandex-team.ru/infra/nanny2/pkg/hq/cache"
	"a.yandex-team.ru/infra/nanny2/pkg/storage"
	pb "a.yandex-team.ru/yp/go/proto/hq"
)

func OnGetNotFound(ctx context.Context, key string, obj storage.Storable) error {
	return storage.NewKeyNotFoundError(key, 0)
}

func OnGuaranteedUpdateNotFound(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
	return storage.NewKeyNotFoundError(key, 0)
}

func OnGetStorageError(ctx context.Context, key string, obj storage.Storable) error {
	return storage.NewUnreachableError(key, 0)
}

func OnGuaranteedUpdateStorageError(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
	return storage.NewUnreachableError(key, 0)
}

func testStatusCode(t *testing.T, status *pb.Status, code int32) {
	if status == nil {
		t.Fatal("Status not set")
	}
	if status.Code != code {
		t.Fatalf("Wrong status code. Need %d, actual status: %+v", code, status)
	}
}

func TestInstanceService_ReportInstanceRevStatus_BadCases(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	in := &pb.ReportInstanceRevStatusRequest{}
	// Test that validation works
	_, status := srv.ReportInstanceRevStatus(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeBadRequest)
	// Make request valid
	in = &pb.ReportInstanceRevStatusRequest{
		InstanceId: "ws35-035:3030@production_nanny",
		Status: &pb.RevisionStatus{
			Id: "1",
			Ready: &pb.Condition{
				Status:             "True",
				LastTransitionTime: &timestamp.Timestamp{},
			},
		},
	}
	// Test NotFound case
	store.OnGuaranteedUpdate = OnGuaranteedUpdateNotFound
	_, status = srv.ReportInstanceRevStatus(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeNotFound)
	// Test internal error during update
	store.OnGuaranteedUpdate = OnGuaranteedUpdateStorageError
	_, status = srv.ReportInstanceRevStatus(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeInternalError)
}

// Test that previously unknown revision reported it's status as ready
func TestInstanceService_ReportInstanceRevStatus_NewActiveRevisionReported(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{},
		Spec: &pb.InstanceSpec{
			Revision: []*pb.InstanceRevision{
				{
					Id: "1",
				},
			},
		},
		Status: &pb.InstanceStatus{
			Ready: &pb.Condition{
				Status:             "False",
				LastTransitionTime: &timestamp.Timestamp{},
			},
			LastHeartbeatTime: &timestamp.Timestamp{},
			Revision: []*pb.RevisionStatus{
				{
					Id: "1",
				},
			},
		},
	}
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		err := tryUpdate(m)
		if err != nil {
			t.Fatal(err)
		}
		return nil
	}
	in := &pb.ReportInstanceRevStatusRequest{
		InstanceId: "ws35-035:3030@production_nanny",
		Status: &pb.RevisionStatus{
			Id: "1",
			Ready: &pb.Condition{
				Status:             "True",
				LastTransitionTime: &timestamp.Timestamp{},
			},
		},
	}
	resource, status := srv.ReportInstanceRevStatus(context.Background(), in, http.Header{})
	if status != nil {
		t.Fatalf("Status must be nil, got %+v", status)
	}
	if resp, ok := resource.(*pb.ReportInstanceRevStatusResponse); !ok {
		t.Fatalf("Invalid response type: %+v", resp)
	}
	// Check that last heartbeat time adjusted
	if m.Status.LastHeartbeatTime.Seconds == 0 {
		t.Fatalf("Last heartbeat time not set: %+v", m.Status.LastHeartbeatTime)
	}
	// Check that revision ready status adjusted instance status
	if m.Status.Ready.Status != "True" {
		t.Fatal("status.ready is not True")
	}
	// Check that we correctly added revision status
	if l := len(m.Status.Revision); l != 1 {
		t.Fatalf("Wrong revisions count: %d", l)
	}
	rev := m.Status.Revision[0]
	if !proto.Equal(rev, in.Status) {
		t.Fatalf("%+v != %+v", rev, in.Status)
	}
}

// Test that not ready revision reported that it is ready
func TestInstanceService_ReportInstanceRevStatus_ExistingActiveRevisionReported(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{},
		Spec: &pb.InstanceSpec{},
		Status: &pb.InstanceStatus{
			Ready: &pb.Condition{
				Status:             "False",
				LastTransitionTime: &timestamp.Timestamp{},
			},
			LastHeartbeatTime: &timestamp.Timestamp{},
			Revision: []*pb.RevisionStatus{
				{
					Id: "1",
					Ready: &pb.Condition{
						Status:             "False",
						LastTransitionTime: &timestamp.Timestamp{},
					},
				},
			},
		},
	}
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		err := tryUpdate(m)
		if err != nil {
			t.Fatal(err)
		}
		return nil
	}

	in := &pb.ReportInstanceRevStatusRequest{
		InstanceId: "ws35-035:3030@production_nanny",
		Status: &pb.RevisionStatus{
			Id: "1",
			Ready: &pb.Condition{
				Status:             "True",
				LastTransitionTime: &timestamp.Timestamp{},
			},
		},
	}
	resource, status := srv.ReportInstanceRevStatus(context.Background(), in, http.Header{})
	if status != nil {
		t.Fatalf("Status must be nil, got %+v", status)
	}
	if resp, ok := resource.(*pb.ReportInstanceRevStatusResponse); !ok {
		t.Fatalf("Invalid response type: %+v", resp)
	}
	// Check that last heartbeat time adjusted
	if m.Status.LastHeartbeatTime.Seconds == 0 {
		t.Fatalf("Last heartbeat time not set: %+v", m.Status.LastHeartbeatTime)
	}
	// Check that revision ready status adjusted instance status
	if m.Status.Ready.Status != "True" {
		t.Fatal("status.ready is not True")
	}
	// Check that we correctly set revision status
	if l := len(m.Status.Revision); l != 1 {
		t.Fatalf("Wrong revisions count: %d", l)
	}
	rev := m.Status.Revision[0]
	if !proto.Equal(rev, in.Status) {
		t.Fatalf("%+v != %+v", rev, in.Status)
	}
}

// Test that ready revision reported that it is not ready
func TestInstanceService_ReportInstanceRevStatus_NotReadyRevisionReported(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	secondRev := &pb.RevisionStatus{
		Id: "2",
		Ready: &pb.Condition{
			Status:             "False",
			LastTransitionTime: &timestamp.Timestamp{},
		},
	}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{},
		Spec: &pb.InstanceSpec{},
		Status: &pb.InstanceStatus{
			Ready: &pb.Condition{
				Status:             "True",
				LastTransitionTime: &timestamp.Timestamp{},
			},
			LastHeartbeatTime: &timestamp.Timestamp{},
			Revision: []*pb.RevisionStatus{
				{
					Id: "1",
					Ready: &pb.Condition{
						Status:             "True",
						LastTransitionTime: &timestamp.Timestamp{},
					},
				},
				secondRev,
			},
		},
	}

	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		err := tryUpdate(m)
		if err != nil {
			t.Fatal(err)
		}
		return nil
	}
	in := &pb.ReportInstanceRevStatusRequest{
		InstanceId: "ws35-035:3030@production_nanny",
		Status: &pb.RevisionStatus{
			Id: "1",
			Ready: &pb.Condition{
				Status:             "False",
				LastTransitionTime: &timestamp.Timestamp{},
			},
		},
	}
	resource, status := srv.ReportInstanceRevStatus(context.Background(), in, http.Header{})
	if status != nil {
		t.Fatalf("Status must be nil, got %+v", status)
	}
	if resp, ok := resource.(*pb.ReportInstanceRevStatusResponse); !ok {
		t.Fatalf("Invalid response type: %+v", resp)
	}
	// Check that last heartbeat time adjusted
	if m.Status.LastHeartbeatTime.Seconds == 0 {
		t.Fatalf("Last heartbeat time not set: %+v", m.Status.LastHeartbeatTime)
	}
	// Check that revision ready status adjusted instance status
	if m.Status.Ready.Status != "False" {
		t.Fatal("status.ready is not False")
	}
	// Check that we correctly set revision status
	if l := len(m.Status.Revision); l != 2 {
		t.Fatalf("Wrong revisions count: %d", l)
	}
	for _, rev := range m.Status.Revision {
		if rev.Id == in.Status.Id {
			if !proto.Equal(rev, in.Status) {
				t.Fatalf("%+v != %+v", rev, in.Status)
			}
		} else {
			if !proto.Equal(rev, secondRev) {
				t.Fatalf("%+v != %+v", rev, secondRev)
			}
		}
	}
}

func TestInstanceService_ReportInstanceRevStatusV2_BadCases(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	in := &pb.ReportInstanceRevStatusV2Request{}
	// Test that validation works
	_, status := srv.ReportInstanceRevStatusV2(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeBadRequest)
	// Test service_id is empty
	in = &pb.ReportInstanceRevStatusV2Request{
		InstanceId: "ws35-035:3030@production_nanny",
		Status: &pb.RevisionStatus{
			Id: "1",
			Ready: &pb.Condition{
				Status:             "True",
				LastTransitionTime: &timestamp.Timestamp{},
			},
		},
	}
	_, status = srv.ReportInstanceRevStatusV2(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeBadRequest)

	// Make request valid
	in = &pb.ReportInstanceRevStatusV2Request{
		InstanceId: "ws35-035:3030@production_nanny",
		ServiceId:  "production_nanny",
		Status: &pb.RevisionStatus{
			Id: "1",
			Ready: &pb.Condition{
				Status:             "True",
				LastTransitionTime: &timestamp.Timestamp{},
			},
		},
	}
	// Test NotFound case
	store.OnGuaranteedUpdate = OnGuaranteedUpdateNotFound
	_, status = srv.ReportInstanceRevStatusV2(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeNotFound)
	// Test internal error during update
	store.OnGuaranteedUpdate = OnGuaranteedUpdateStorageError
	_, status = srv.ReportInstanceRevStatusV2(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeInternalError)

	// Test internal error during create instance
	in = &pb.ReportInstanceRevStatusV2Request{
		InstanceId: "ws35-035:3030@production_nanny",
		ServiceId:  "production_nanny",
		Status: &pb.RevisionStatus{
			Id: "1",
			Ready: &pb.Condition{
				Status:             "True",
				LastTransitionTime: &timestamp.Timestamp{},
			},
		},
		Policies: []pb.ReportInstanceRevStatusPolicy{pb.ReportInstanceRevStatusPolicy_CREATE_INSTANCE_IF_NOT_EXISTS},
	}
	store.OnGuaranteedUpdate = OnGuaranteedUpdateNotFound
	store.OnCreate = func(ctx context.Context, key string, obj storage.Storable, out storage.Storable) error {
		return storage.NewUnreachableError(key, 0)
	}
	_, status = srv.ReportInstanceRevStatusV2(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeInternalError)

	// Test conflict error during create instance
	store.OnCreate = func(ctx context.Context, key string, obj storage.Storable, out storage.Storable) error {
		return storage.NewKeyExistsError(key, 0)
	}
	_, status = srv.ReportInstanceRevStatusV2(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeConflict)
}

func TestInstanceService_GetInstanceRev(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	in := &pb.GetInstanceRevRequest{}
	// Test invalid request
	_, status := srv.GetInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeBadRequest)
	// Test unknown instance
	in.Id = "ws35-035:8081@production_nanny"
	in.Rev = "1"
	store.OnGet = OnGetNotFound
	_, status = srv.GetInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeNotFound)
	// Test storage error
	store.OnGet = OnGetStorageError
	_, status = srv.GetInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeInternalError)
	// Test revision not found
	store.OnGet = func(ctx context.Context, key string, objPtr storage.Storable) error {
		m := objPtr.(*pb.Instance)
		m.Reset()
		// Just don't fill any revisions
		m.Spec = &pb.InstanceSpec{}
		return nil
	}
	_, status = srv.GetInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeNotFound)
	// Test good case
	store.OnGet = func(ctx context.Context, key string, objPtr storage.Storable) error {
		m := objPtr.(*pb.Instance)
		m.Reset()
		// Just don't fill any revisions
		m.Spec = &pb.InstanceSpec{
			Revision: []*pb.InstanceRevision{
				{
					Id: "1",
				},
			},
		}
		return nil
	}
	resp, status := srv.GetInstanceRev(context.Background(), in, http.Header{})
	if status != nil {
		t.Fatal(status)
	}
	m, ok := resp.(*pb.GetInstanceRevResponse)
	if !ok {
		t.Fatalf("Invalid response type: %+v", resp)
	}
	if m.Revision.Id != "1" {
		t.Fatalf("Wrong response content: %+v", m)
	}
}

func TestInstanceService_CreateInstance(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	// Create instance with 1 revision, check that instance created and status is set properly
	in := &pb.CreateInstanceRequest{
		Meta: &pb.InstanceMeta{
			Id:        "production_nanny:1082@ws25-500.search.yandex.net",
			ServiceId: "production_nanny",
		},
		Spec: &pb.InstanceSpec{
			Revision: []*pb.InstanceRevision{
				{
					Id: "production_nanny-123245843570",
				},
			},
		},
	}
	store.OnGet = OnGetNotFound
	store.OnCreate = func(ctx context.Context, key string, obj storage.Storable, out storage.Storable) error {
		if key != in.Meta.Id {
			t.Fatal("Wrong key ", key)
		}
		m := obj.(*pb.Instance)
		if m.Status == nil {
			t.Fatal("Status should be set, got nil")
		}
		if m.Status.Ready.Status != "False" {
			t.Fatalf("status.ready.status must be False, got %s", m.Status.Ready.Status)
		}
		if len(m.Status.Revision) != 1 {
			t.Fatalf("status.revision must have 1 element, got %d", len(m.Status.Revision))
		}
		rs := m.Status.Revision[0]
		if rs.Id != m.Spec.Revision[0].Id {
			t.Fatalf("status.revision[0] id is wrong, %s != %s", rs.Id, m.Spec.Revision[0].Id)
		}
		if rs.Ready.Status != "False" {
			t.Fatalf("status.revision[0].ready.status must be False, got %s", rs.Ready.Status)
		}
		if rs.Installed.Status != "False" {
			t.Fatalf("status.revision[0].installed.status must be False, got %s", rs.Installed.Status)
		}
		return nil
	}
	resp, status := srv.CreateInstance(context.TODO(), in, http.Header{})
	if status != nil {
		t.Fatalf("Status is not nil: %+v", status)
	}
	if _, ok := resp.(*pb.CreateInstanceResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}
}

func Test_splitInstances(t *testing.T) {
	ids := make([]string, 0, 100)
	for i := 0; i < cap(ids); i++ {
		ids = append(ids, strconv.Itoa(i))
	}
	instances := make([]*pb.Instance, len(ids), cap(ids))
	for i, id := range ids {
		m := &pb.Instance{
			Meta: &pb.InstanceMeta{Id: id},
		}
		instances[i] = m
	}
	// Check case with 10 instances split in 4 parts
	jobs := splitInstances(instances[:10], 4)
	if l := len(jobs); l != 4 {
		t.Fatalf("Not enough jobs, got %d, need %d", l, 4)
	}
	i := 0

	// Check that all jobs are present and in order (implementation detail, but helps in tests)
	for _, job := range jobs {
		for _, m := range job {
			if m.Meta.Id != ids[i] {
				t.Fatal(jobs, instances[:10])
			}
			i++
		}
	}
	if i != 10 {
		t.Fatal(i)
	}
	// Check case with 100 instances split in 8 parts
	jobs = splitInstances(instances, 8)
	if l := len(jobs); l != 8 {
		t.Fatalf("Not enough jobs, got %d, need %d", l, 8)
	}
	i = 0
	// Check that all jobs are present and in order (implementation detail, but helps in tests)
	for _, job := range jobs {
		for _, m := range job {
			if m.Meta.Id != ids[i] {
				t.Fatal(jobs, instances)
			}
			i++
		}
	}
	if i != 100 {
		t.Fatal(i)
	}
}

func TestInstanceService_DeleteServiceRev(t *testing.T) {
	rev := "prodution_testing-1"
	store := storage.FakeStore{}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{
			Id:        "ws35-035:1041@production_testing",
			ServiceId: "production_testing",
		},
		Spec: &pb.InstanceSpec{
			Revision: []*pb.InstanceRevision{
				{
					Id: rev,
				},
			},
		},
		Status: &pb.InstanceStatus{},
	}
	srv := instanceService{
		store: &store,
		idx: &cache.FakeInstanceIndex{
			OnFindByService: func(serviceId string, readyOnly bool) ([]*pb.Instance, string) {
				// Return instance with one revision
				return []*pb.Instance{m}, "1"
			},
		},
	}
	in := &pb.DeleteServiceRevRequest{
		Rev:       rev,
		ServiceId: "production_testing",
	}
	updateCalled := false
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		updateCalled = true
		err := tryUpdate(m)
		if err != nil {
			t.Fatal(err)
		}
		if len(m.Spec.Revision) != 0 {
			t.Fatalf("tryUpdate did not remove revision: %+v", m.Spec)
		}
		return nil
	}
	deleteCalled := false
	store.OnDelete = func(ctx context.Context, key string, out storage.Storable) error {
		deleteCalled = true
		if key != m.Meta.Id {
			t.Fatalf("Wrong key passed to Delete: %s", key)
		}
		return nil
	}
	// Check response
	resp, status := srv.DeleteServiceRev(context.Background(), in, http.Header{})
	if !deleteCalled {
		t.Fatal("Did not called delete for instance")
	}
	if !updateCalled {
		t.Fatal("Did not called update for instance")
	}
	if status != nil {
		t.Fatalf("Unexpected failure status: %+v", status)
	}
	if _, ok := resp.(*pb.DeleteServiceRevResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}
}

func TestInstanceService_DeleteServiceRev_WithYpSpecStorage(t *testing.T) {
	rev := "prodution_testing-1"
	store := storage.FakeStore{}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{
			Id:            "ws35-035:1041@production_testing",
			ServiceId:     "production_testing",
			StoragePolicy: pb.StoragePolicy_SPEC_YP_STATUS_HQ,
		},
		Status: &pb.InstanceStatus{
			Revision: []*pb.RevisionStatus{
				{Id: rev},
			},
		},
	}
	srv := instanceService{
		store: &store,
		idx: &cache.FakeInstanceIndex{
			OnFindByService: func(serviceId string, readyOnly bool) ([]*pb.Instance, string) {
				// Return instance with one revision
				return []*pb.Instance{m}, "1"
			},
		},
	}
	in := &pb.DeleteServiceRevRequest{
		Rev:       rev,
		ServiceId: "production_testing",
	}
	updateCalled := false
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		updateCalled = true
		err := tryUpdate(m)
		if err != nil {
			t.Fatal(err)
		}
		if len(m.Status.Revision) != 0 {
			t.Fatalf("tryUpdate did not remove revision: %+v", m.Status)
		}
		return nil
	}
	deleteCalled := false
	store.OnDelete = func(ctx context.Context, key string, out storage.Storable) error {
		deleteCalled = true
		if key != m.Meta.Id {
			t.Fatalf("Wrong key passed to Delete: %s", key)
		}
		return nil
	}
	// Check response
	resp, status := srv.DeleteServiceRev(context.Background(), in, http.Header{})
	if !deleteCalled {
		t.Fatal("Did not called delete for instance")
	}
	if !updateCalled {
		t.Fatal("Did not called update for instance")
	}
	if status != nil {
		t.Fatalf("Unexpected failure status: %+v", status)
	}
	if _, ok := resp.(*pb.DeleteServiceRevResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}
}

func TestInstanceService_DeleteServiceRev_WithPartialYpSpecStorage(t *testing.T) {
	rev := "prodution_testing-1"
	store := storage.FakeStore{}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{
			Id:            "ws35-035:1041@production_testing",
			ServiceId:     "production_testing",
			StoragePolicy: pb.StoragePolicy_SPEC_YP_STATUS_HQ,
		},
		Spec: &pb.InstanceSpec{
			Revision: []*pb.InstanceRevision{
				{
					Id: rev,
				},
			},
		},
		Status: &pb.InstanceStatus{
			Revision: []*pb.RevisionStatus{
				{Id: rev},
			},
		},
	}
	srv := instanceService{
		store: &store,
		idx: &cache.FakeInstanceIndex{
			OnFindByService: func(serviceId string, readyOnly bool) ([]*pb.Instance, string) {
				// Return instance with one revision
				return []*pb.Instance{m}, "1"
			},
		},
	}
	in := &pb.DeleteServiceRevRequest{
		Rev:       rev,
		ServiceId: "production_testing",
	}
	updateCalled := false
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		updateCalled = true
		err := tryUpdate(m)
		if err != nil {
			t.Fatal(err)
		}
		if len(m.Status.Revision) != 0 {
			t.Fatalf("tryUpdate did not remove revision: %+v", m.Status)
		}
		if len(m.Spec.Revision) != 0 {
			t.Fatalf("tryUpdate did not remove revision: %+v", m.Spec)
		}
		return nil
	}
	deleteCalled := false
	store.OnDelete = func(ctx context.Context, key string, out storage.Storable) error {
		deleteCalled = true
		if key != m.Meta.Id {
			t.Fatalf("Wrong key passed to Delete: %s", key)
		}
		return nil
	}
	// Check response
	resp, status := srv.DeleteServiceRev(context.Background(), in, http.Header{})
	if !deleteCalled {
		t.Fatal("Did not called delete for instance")
	}
	if !updateCalled {
		t.Fatal("Did not called update for instance")
	}
	if status != nil {
		t.Fatalf("Unexpected failure status: %+v", status)
	}
	if _, ok := resp.(*pb.DeleteServiceRevResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}
}

func TestInstanceService_AddInstanceRev_InvalidReq(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	// Test no instance id specified
	in := &pb.AddInstanceRevRequest{
		Id: "",
	}
	_, status := srv.AddInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeBadRequest)
	// Test rev == nil
	in.Id = "production_test_addrev:1031"
	_, status = srv.AddInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeBadRequest)
	// Test no rev id set in revision object itself
	in.Rev = &pb.InstanceRevision{}
	_, status = srv.AddInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeBadRequest)
}

func TestInstanceService_AddInstanceRev_StorageErrors(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	in := &pb.AddInstanceRevRequest{
		Id: "production_test_addrev:1031",
		Rev: &pb.InstanceRevision{
			Id:        "production_test-1213123",
			ShardName: strings.Repeat("x", 500),
		},
	}
	// Test shard_name validation
	_, status := srv.AddInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeBadRequest)
	// Test instance not found
	in.Rev.ShardName = "shard-123"
	store.OnGuaranteedUpdate = OnGuaranteedUpdateNotFound
	_, status = srv.AddInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeNotFound)
	// Test storage internal error
	store.OnGuaranteedUpdate = OnGuaranteedUpdateStorageError
	_, status = srv.AddInstanceRev(context.Background(), in, http.Header{})
	testStatusCode(t, status, api.StatusCodeInternalError)
}

func TestInstanceService_AddInstanceRev_GoodCase(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	instanceID := "production_test_addrev:1031"
	in := &pb.AddInstanceRevRequest{
		Id: instanceID,
		Rev: &pb.InstanceRevision{
			Id:        "production_test-1213123",
			ShardName: "shard-123",
		},
	}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{
			Id:        instanceID,
			ServiceId: "production_test_addrev",
		},
		Spec: &pb.InstanceSpec{
			Revision: []*pb.InstanceRevision{
				{
					Id: "production_test_addrev-1",
				},
			},
		},
		Status: &pb.InstanceStatus{
			Ready: &pb.Condition{},
			Revision: []*pb.RevisionStatus{
				{
					Id:        "production_test_addrev-1",
					Ready:     &pb.Condition{},
					Installed: &pb.Condition{},
				},
			},
		},
	}
	// Test update success
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		if key != instanceID {
			t.Fatalf("Wrong key for GuaranteedUpdate: %s != %s", key, instanceID)
		}
		if err := tryUpdate(m); err != nil {
			t.Fatalf("Update func failed with: %s", err.Error())
		}
		return nil
	}
	// Check response
	resp, status := srv.AddInstanceRev(context.Background(), in, http.Header{})
	if status != nil {
		t.Fatalf("Unexpected failure status: %+v", status)
	}
	if _, ok := resp.(*pb.AddInstanceRevResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}
	// Check that we did actually added revision and set initial status for it
	if l := len(m.Spec.Revision); l != 2 {
		t.Fatalf("Did not add revision to spec: %d", l)
	}
	if r := m.Spec.Revision[1]; !proto.Equal(r, in.Rev) {
		t.Fatalf("Provided and saved revision don't match: %#v", r)
	}
	if l := len(m.Status.Revision); l != 2 {
		t.Fatalf("Did not add revision to status: %d", l)
	}
	rs := m.Status.Revision[1]
	if rs.Id != in.Rev.Id {
		t.Fatalf("Wrong revision id in status: %s != %s", rs.Id, in.Rev.Id)
	}
	if rs.Ready.Status != "False" {
		t.Fatalf("Wrong revision ready status: %s", rs.Ready.Status)
	}
	if rs.Installed.Status != "False" {
		t.Fatalf("Wrong revision installed status: %s", rs.Ready.Status)
	}
}

func TestInstanceService_AddInstanceRev_SetNewHostnameVersion(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	instanceID := "production_test_addrev:1031"
	in := &pb.AddInstanceRevRequest{
		Id: instanceID,
		Rev: &pb.InstanceRevision{
			Id:        "production_test-1213123",
			ShardName: "shard-123",
		},
		Hostname: "new.fake.hostname",
		HostnameVersion: &pb.HostnameVersion{
			Major: 10,
			Minor: 9,
		},
	}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{
			Id:        instanceID,
			ServiceId: "production_test_addrev",
		},
		Spec: &pb.InstanceSpec{
			Revision: []*pb.InstanceRevision{
				{
					Id: "production_test_addrev-1",
				},
			},
			HostnameVersion: &pb.HostnameVersion{
				Major: 10,
				Minor: 8,
			},
		},
		Status: &pb.InstanceStatus{
			Ready: &pb.Condition{},
			Revision: []*pb.RevisionStatus{
				{
					Id:        "production_test_addrev-1",
					Ready:     &pb.Condition{},
					Installed: &pb.Condition{},
				},
			},
		},
	}
	// Test update success
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		if key != instanceID {
			t.Fatalf("Wrong key for GuaranteedUpdate: %s != %s", key, instanceID)
		}
		if err := tryUpdate(m); err != nil {
			t.Fatalf("Update func failed with: %s", err.Error())
		}
		return nil
	}
	// Check response
	resp, status := srv.AddInstanceRev(context.Background(), in, http.Header{})
	if status != nil {
		t.Fatalf("Unexpected failure status: %+v", status)
	}
	if _, ok := resp.(*pb.AddInstanceRevResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}
	if h := m.Spec.Hostname; h != in.Hostname {
		t.Fatalf("Provided and saved hostname don't match: %#v", h)
	}
	if v := m.Spec.HostnameVersion; !proto.Equal(v, in.HostnameVersion) {
		t.Fatalf("Provided and saved hostname version don't match: %#v", v)
	}
}

func TestInstanceService_AddInstanceRev_SetNewHostnameVersionTrunk(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	instanceID := "production_test_addrev:1031"
	in := &pb.AddInstanceRevRequest{
		Id: instanceID,
		Rev: &pb.InstanceRevision{
			Id:        "production_test-1213123",
			ShardName: "shard-123",
		},
		Hostname: "new.fake.hostname",
		HostnameVersion: &pb.HostnameVersion{
			Major: 0,
			Minor: 0,
		},
	}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{
			Id:        instanceID,
			ServiceId: "production_test_addrev",
		},
		Spec: &pb.InstanceSpec{
			Revision: []*pb.InstanceRevision{
				{
					Id: "production_test_addrev-1",
				},
			},
			HostnameVersion: &pb.HostnameVersion{
				Major: 10,
				Minor: 8,
			},
		},
		Status: &pb.InstanceStatus{
			Ready: &pb.Condition{},
			Revision: []*pb.RevisionStatus{
				{
					Id:        "production_test_addrev-1",
					Ready:     &pb.Condition{},
					Installed: &pb.Condition{},
				},
			},
		},
	}
	// Test update success
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		if key != instanceID {
			t.Fatalf("Wrong key for GuaranteedUpdate: %s != %s", key, instanceID)
		}
		if err := tryUpdate(m); err != nil {
			t.Fatalf("Update func failed with: %s", err.Error())
		}
		return nil
	}
	// Check response
	resp, status := srv.AddInstanceRev(context.Background(), in, http.Header{})
	if status != nil {
		t.Fatalf("Unexpected failure status: %+v", status)
	}
	if _, ok := resp.(*pb.AddInstanceRevResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}
	if h := m.Spec.Hostname; h != in.Hostname {
		t.Fatalf("Provided and saved hostname don't match: %#v", h)
	}
	if v := m.Spec.HostnameVersion; !proto.Equal(v, in.HostnameVersion) {
		t.Fatalf("Provided and saved hostname version don't match: %#v", v)
	}
}

func TestInstanceService_AddInstanceRev_IgnoreOldHostname(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	instanceID := "production_test_addrev:1031"
	in := &pb.AddInstanceRevRequest{
		Id: instanceID,
		Rev: &pb.InstanceRevision{
			Id:        "production_test-1213123",
			ShardName: "shard-123",
		},
		Hostname: "new.fake.hostname",
		HostnameVersion: &pb.HostnameVersion{
			Major: 10,
			Minor: 8,
		},
	}
	oldHostname := "existing.fake.hostname"
	oldHostnameVersion := &pb.HostnameVersion{
		Major: 10,
		Minor: 9,
	}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{
			Id:        instanceID,
			ServiceId: "production_test_addrev",
		},
		Spec: &pb.InstanceSpec{
			Revision: []*pb.InstanceRevision{
				{
					Id: "production_test_addrev-1",
				},
			},
			Hostname:        oldHostname,
			HostnameVersion: oldHostnameVersion,
		},
		Status: &pb.InstanceStatus{
			Ready: &pb.Condition{},
			Revision: []*pb.RevisionStatus{
				{
					Id:        "production_test_addrev-1",
					Ready:     &pb.Condition{},
					Installed: &pb.Condition{},
				},
			},
		},
	}
	// Test update success
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		if key != instanceID {
			t.Fatalf("Wrong key for GuaranteedUpdate: %s != %s", key, instanceID)
		}
		if err := tryUpdate(m); err != nil {
			t.Fatalf("Update func failed with: %s", err.Error())
		}
		return nil
	}
	// Check response
	resp, status := srv.AddInstanceRev(context.Background(), in, http.Header{})
	if status != nil {
		t.Fatalf("Unexpected failure status: %+v", status)
	}
	if _, ok := resp.(*pb.AddInstanceRevResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}
	if h := m.Spec.Hostname; h != oldHostname {
		t.Fatalf("Hostname has been updated with old version: %#v", h)
	}
	if v := m.Spec.HostnameVersion; !proto.Equal(v, oldHostnameVersion) {
		t.Fatalf("Hostname version has been updated with old version: %#v", v)
	}
}

func TestInstanceService_UpdateInstanceAllocation_GoodCase(t *testing.T) {
	store := storage.FakeStore{}
	srv := instanceService{
		store: &store,
		idx:   &cache.FakeInstanceIndex{},
	}
	instanceID := "production_test:1031"
	in := &pb.UpdateInstanceAllocationRequest{
		Id:       instanceID,
		NodeName: "new.fake.node_name",
	}
	m := &pb.Instance{
		Meta: &pb.InstanceMeta{
			Id:        instanceID,
			ServiceId: "production_test",
		},
		Spec: &pb.InstanceSpec{
			NodeName: "old.fake.node_name",
		},
	}
	// Test update success
	store.OnGuaranteedUpdate = func(ctx context.Context, key string, tryUpdate storage.UpdateFunc) error {
		if key != instanceID {
			t.Fatalf("Wrong key for GuaranteedUpdate: %s != %s", key, instanceID)
		}
		if err := tryUpdate(m); err != nil {
			t.Fatalf("Update func failed with: %s", err.Error())
		}
		return nil
	}
	// Check response
	resp, status := srv.UpdateInstanceAllocation(context.Background(), in, http.Header{})
	if status != nil {
		t.Fatalf("Unexpected failure status: %+v", status)
	}
	if _, ok := resp.(*pb.UpdateInstanceAllocationResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}
	if h := m.Spec.NodeName; h != in.NodeName {
		t.Fatalf("Provided and saved node name don't match: %#v", h)
	}
}

func TestInstanceService_FindInstancesWithProxyToYp(t *testing.T) {
	idxCache := &cache.FakeInstanceIndex{}
	ypClient := &ypclient.FakeYpClient{}
	srv := instanceService{
		idx:      idxCache,
		ypClient: ypClient,
	}
	idxCache.OnFindByService = func(serviceID string, readyOnly bool) ([]*pb.Instance, string) {
		return []*pb.Instance{
			{
				Meta: &pb.InstanceMeta{
					Id:            "test-1@service_id",
					StoragePolicy: pb.StoragePolicy_SPEC_YP_STATUS_HQ,
				},
				Status: &pb.InstanceStatus{
					Revision: make([]*pb.RevisionStatus, 0),
				},
			},
			{
				Meta: &pb.InstanceMeta{
					Id:            "test-3.myt.yp-c.yandex.net@service_id",
					StoragePolicy: pb.StoragePolicy_SPEC_YP_STATUS_HQ,
				},
				Status: &pb.InstanceStatus{
					Revision: make([]*pb.RevisionStatus, 0),
				},
			},
			{
				Meta: &pb.InstanceMeta{
					Id:            "removed-by-nil-spec@service_id",
					StoragePolicy: pb.StoragePolicy_SPEC_YP_STATUS_HQ,
				},
				Status: &pb.InstanceStatus{
					Revision: make([]*pb.RevisionStatus, 0),
				},
			},
			{
				Meta: &pb.InstanceMeta{
					Id:            "removed-by-spec-revisions-empty@service_id",
					StoragePolicy: pb.StoragePolicy_SPEC_YP_STATUS_HQ,
				},
				Spec: &pb.InstanceSpec{
					Revision: make([]*pb.InstanceRevision, 0)},
				Status: &pb.InstanceStatus{
					Revision: make([]*pb.RevisionStatus, 0),
				},
			},
		}, ""
	}
	nodeID := "node-id"
	ypClient.OnListPods = func(ctx context.Context, podSetId string) (pods []*ypclient.YpPod, err error) {
		fqdnTest1 := "test-1.sas.yp-c.yandex.net"
		fqdnTest2 := "test-2.iva.yp-c.yandex.net"
		fqdnTest3 := "test-3.myt.yp-c.yandex.net"
		return []*ypclient.YpPod{
			{
				Pod: ypapi.TPod{
					Meta: &ypapi.TPodMeta{
						Id: "test-1",
					},
					Spec: &ypapi.TPodSpec{
						NodeId: &nodeID,
						Iss: &clusterapi.HostConfiguration{
							Instances: []*clusterapi.HostConfigurationInstance{
								{InstanceRevision: &pb.InstanceRevision{Id: "service_id-rev-1"}},
							},
						},
					},
					Status: &ypapi.TPodStatus{
						Dns: &ypapi.TPodStatus_TDns{
							PersistentFqdn: &fqdnTest1,
						},
					},
				},
				Cluster: "sas",
			},
			{
				Pod: ypapi.TPod{
					Meta: &ypapi.TPodMeta{
						Id: "test-2",
					},
					Spec: &ypapi.TPodSpec{
						NodeId: &nodeID,
						Iss: &clusterapi.HostConfiguration{
							Instances: []*clusterapi.HostConfigurationInstance{
								{InstanceRevision: &pb.InstanceRevision{Id: "service_id-rev-1"}},
							},
						},
					},
					Status: &ypapi.TPodStatus{
						Dns: &ypapi.TPodStatus_TDns{
							PersistentFqdn: &fqdnTest2,
						},
					},
				},
				Cluster: "iva",
			},
			{
				Pod: ypapi.TPod{
					Meta: &ypapi.TPodMeta{
						Id: "test-3",
					},
					Spec: &ypapi.TPodSpec{
						NodeId: &nodeID,
						Iss: &clusterapi.HostConfiguration{
							Instances: []*clusterapi.HostConfigurationInstance{
								{InstanceRevision: &pb.InstanceRevision{Id: "service_id-rev-1"}},
								{InstanceRevision: &pb.InstanceRevision{Id: "service_id-rev-2"}},
							},
						},
					},
					Status: &ypapi.TPodStatus{
						Dns: &ypapi.TPodStatus_TDns{
							PersistentFqdn: &fqdnTest3,
						},
					},
				},
				Cluster: "myt",
			},
		}, nil
	}

	request00 := &pb.FindInstancesRequest{
		Filter: nil,
		FieldMask: &pb.FieldMask{
			Paths: []string{"status.ready", "status.revision"},
		},
	}
	_, status00 := srv.FindInstancesWithProxyToYp(context.Background(), request00, http.Header{})
	assert.NotNil(t, status00)
	assert.Equal(t, int32(400), status00.Code)
	assert.Equal(t, status00.Message, ".Filter.ServiceId is required")

	request01 := &pb.FindInstancesRequest{
		Filter: &pb.InstanceFilter{ServiceId: "service_id"},
		FieldMask: &pb.FieldMask{
			Paths: []string{"status.ready", "status.revision"},
		},
	}
	resp01, status01 := srv.FindInstancesWithProxyToYp(context.Background(), request01, http.Header{})
	assert.Nil(t, status01)
	assert.NotNil(t, resp01)

	request := &pb.FindInstancesRequest{
		Filter:    &pb.InstanceFilter{ServiceId: "service_id"},
		FieldMask: nil,
	}
	resp, status := srv.FindInstancesWithProxyToYp(context.Background(), request, http.Header{})
	if status != nil {
		t.Fatalf("Unexpected failure status: %+v", status)
	}

	if _, ok := resp.(*pb.FindInstancesResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}

	r := resp.(*pb.FindInstancesResponse)

	assert.Equal(t, len(r.Instance), 3)
	instancesMap := makeIDToInstanceIndex(r.Instance, false)
	instance, ok := instancesMap["test-1"]
	assert.Equal(t, nodeID, instance.Spec.NodeName)
	assert.Equal(t, ok, true, "test-1 instance does not exists")
	assert.Len(t, instance.Spec.Revision, 1)
	assert.Len(t, instance.Spec.Allocation.Port, 1)

	instance2, ok := instancesMap["test-2.iva.yp-c.yandex.net"]
	assert.Equal(t, ok, true, "test-2.iva.yp-c.yandex.net instance does not exists")
	assert.Len(t, instance2.Spec.Revision, 1)
	assert.Len(t, instance2.Status.Revision, 1)

	instance3, ok := instancesMap["test-3.myt.yp-c.yandex.net"]
	assert.Equal(t, ok, true, "test-3.myt.yp-c.yandex.net instance does not exists")
	assert.Len(t, instance3.Spec.Revision, 2, "Spec Revisions should be merged")
	assert.Len(t, instance3.Status.Revision, 2, "RevisionsStatuses should be initialised")
}

func TestInstanceService_GetInstanceWithProxyToYp(t *testing.T) {
	store := &storage.FakeStore{}
	idxCache := &cache.FakeInstanceIndex{}
	ypClient := &ypclient.FakeYpClient{}
	srv := instanceService{
		store:    store,
		idx:      idxCache,
		ypClient: ypClient,
	}
	store.OnGet = func(ctx context.Context, key string, objPtr storage.Storable) error {
		m := objPtr.(*pb.Instance)
		m.Reset()
		m.Meta = &pb.InstanceMeta{
			Id:            "test-1@service_id",
			StoragePolicy: pb.StoragePolicy_SPEC_YP_STATUS_HQ,
		}
		m.Status = &pb.InstanceStatus{
			Revision: make([]*pb.RevisionStatus, 0),
		}
		return nil
	}
	ypClient.OnFindPod = func(ctx context.Context, podID string, clusters ...string) (pod *ypclient.YpPod, err error) {
		assert.Equal(t, "test-1", podID)
		nodeID := "node-id"
		fqdnTest1 := "test-1.sas.yp-c.yandex.net"
		return &ypclient.YpPod{
			Pod: ypapi.TPod{
				Meta: &ypapi.TPodMeta{
					Id: "test-1",
				},
				Spec: &ypapi.TPodSpec{
					NodeId: &nodeID,
					Iss: &clusterapi.HostConfiguration{
						Instances: []*clusterapi.HostConfigurationInstance{
							{InstanceRevision: &pb.InstanceRevision{Id: "service_id-rev-1"}},
						},
					},
				},
				Status: &ypapi.TPodStatus{
					Dns: &ypapi.TPodStatus_TDns{
						PersistentFqdn: &fqdnTest1,
					},
				},
			},
			Cluster: "sas",
		}, nil
	}

	ypClient.OnGetClusters = func() []string {
		return []string{"sas"}
	}

	request := &pb.GetInstanceRequest{
		Id: "test-1@service_id",
	}
	resp, status := srv.GetInstanceWithProxyToYp(context.Background(), request, http.Header{})
	assert.Nil(t, status)

	if _, ok := resp.(*pb.GetInstanceResponse); !ok {
		t.Fatalf("Wrong response type: %+v", resp)
	}

	r := resp.(*pb.GetInstanceResponse)
	assert.Len(t, r.Instance.Spec.Revision, 1)
	assert.Len(t, r.Instance.Status.Revision, 1)
	assert.Len(t, r.Instance.Spec.Allocation.Port, 1)

}

func TestInstanceService_Mount(t *testing.T) {
	srv := instanceService{}
	mux := chi.NewRouter()

	enabled := srv.Mount(mux)
	assert.Equal(t, false, enabled)

	err := os.Setenv("ENABLE_PROXY_TO_YP", "1")
	if err != nil {
		t.Fatalf("Can't set env: %s", err)
	}
	mux2 := chi.NewRouter()
	enabledNow := srv.Mount(mux2)
	assert.Equal(t, true, enabledNow)
}
