package validation

import (
	"errors"
	"time"

	"github.com/golang/protobuf/ptypes/timestamp"

	proto "a.yandex-team.ru/yp/go/proto/hq"
)

var (
	errBadMeta          = errors.New("bad meta field")
	errNoID             = errors.New("no id specified")
	errNoNodeName       = errors.New("no node_name specified")
	errNoServiceID      = errors.New("no meta.service_id specified")
	errNoVersion        = errors.New("no meta.version specified")
	errBadStatus        = errors.New("bad status field")
	errChangedServiceID = errors.New("changing meta.service_id is not allowed")
)

func ValidateUpdateRequest(m *proto.UpdateInstanceRequest) error {
	if m.Meta == nil {
		return errBadMeta
	}
	if m.Meta.Id == "" {
		return errNoID
	}
	if m.Meta.Version == "" {
		return errNoVersion
	}
	return nil
}

func ValidateUpdateInstanceAllocationRequest(m *proto.UpdateInstanceAllocationRequest) error {
	if m.Id == "" {
		return errNoID
	}
	if m.NodeName == "" {
		return errNoNodeName
	}
	return nil
}

func ValidateCreateRequest(m *proto.CreateInstanceRequest) error {
	if m.Meta == nil {
		return errBadMeta
	}
	if m.Meta.Id == "" {
		return errNoID
	}
	if m.Meta.ServiceId == "" {
		return errNoServiceID
	}
	return nil
}

func ValidateInstanceCreation(m *proto.Instance) error {
	nowSec := time.Now().Unix()
	if m.Status == nil {
		m.Status = &proto.InstanceStatus{
			Ready: &proto.Condition{
				Status:  "False",
				Reason:  "Created",
				Message: "Instance created",
				LastTransitionTime: &timestamp.Timestamp{
					Seconds: nowSec,
				},
			},
		}
	}
	if m.Status.LastHeartbeatTime == nil {
		m.Status.LastHeartbeatTime = &timestamp.Timestamp{
			Seconds: nowSec,
		}
	}
	return nil
}

func ValidateRevStatusReport(m *proto.ReportInstanceRevStatusRequest) error {
	if m.InstanceId == "" {
		return errNoID
	}
	if m.Status == nil {
		return errBadStatus
	}
	if m.Status.Id == "" {
		return errNoID
	}
	if m.Status.Ready == nil {
		return errors.New("status.ready must be set")
	}
	if m.Status.Ready.Status != "True" && m.Status.Ready.Status != "False" {
		return errors.New("status.ready.status must be one of: True, False")
	}
	if m.Status.Ready.LastTransitionTime == nil {
		return errors.New("status.ready.last_transition_time must be set")
	}
	return nil
}

func ValidateRevStatusReportV2(m *proto.ReportInstanceRevStatusV2Request) error {
	if m.InstanceId == "" {
		return errNoID
	}
	if m.ServiceId == "" {
		return errNoServiceID
	}
	if m.Status == nil {
		return errBadStatus
	}
	if m.Status.Id == "" {
		return errNoID
	}
	if m.Status.Ready == nil {
		return errors.New("status.ready must be set")
	}
	if m.Status.Ready.Status != "True" && m.Status.Ready.Status != "False" {
		return errors.New("status.ready.status must be one of: True, False")
	}
	if m.Status.Ready.LastTransitionTime == nil {
		return errors.New("status.ready.last_transition_time must be set")
	}
	return nil
}

func ValidateGetInstance(m *proto.GetInstanceRequest) error {
	if m.Id == "" {
		return errNoID
	}
	return nil
}

func ValidateGetInstanceRev(m *proto.GetInstanceRevRequest) error {
	if m.Id == "" {
		return errNoID
	}
	if m.Rev == "" {
		return errors.New("no rev specified")
	}
	return nil
}

func ValidateListAttributeValues(m *proto.ListAttributeValuesRequest) error {
	if m.Attr == "" {
		return errors.New("no attr specified")
	}
	if len(m.Attr) > 1024 {
		return errors.New("attr value is too long")
	}
	return nil
}

func ValidateGetRevisionStats(m *proto.GetRevisionStatsRequest) error {
	if m.Filter == nil {
		return errors.New("no filter specified")
	}
	if m.Filter.ServiceId == "" {
		return errors.New("no filter.service_id specified")
	}
	if len(m.Filter.ServiceId) > 1024 {
		return errors.New("filter.service_id value is too long")
	}
	return nil
}

func ValidateAddInstanceRev(m *proto.AddInstanceRevRequest) error {
	if m.Id == "" {
		return errNoID
	}
	if m.Rev == nil {
		return errors.New("no rev specified")
	}
	if m.Rev.Id == "" {
		return errors.New("no rev id specified")
	}
	if len(m.Rev.ShardName) > 255 {
		return errors.New("shard_name is too long (max 255 symbols allowed)")
	}
	return nil
}
