package hqsyncer

import (
	"context"
	"time"

	"a.yandex-team.ru/infra/nanny2/pkg/log"
	"a.yandex-team.ru/infra/nanny2/pkg/rpc"
	"a.yandex-team.ru/infra/nanny2/pkg/ypclient"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/ytlock"
)

type Config struct {
	SleepAfterIterationSeconds int               `yaml:"sleep_after_iteration_seconds"`
	YpClusters                 []string          `yaml:"yp_clusters"`
	FetchYpPodsBatchSize       int32             `yaml:"fetch_yp_pods_batch_size"`
	StrippedHQClusters         []string          `yaml:"stripped_hq_clusters"`
	StrippedHQServices         []string          `yaml:"stripped_hq_services"`
	LockeNodeYtPath            ypath.Path        `yaml:"locke_node_yt_path"` // string
	RPCConfig                  rpc.ClientConfig  `yaml:"rpc"`
	CustomYpClusterSuffixes    map[string]string `yaml:"custom_yp_cluster_suffixes"`
	MaxPodsToUpdateOnIteration int               `yaml:"max_pods_to_update_on_iteration"`
}

func Main(ctx context.Context, cfg *Config, dryRun bool) error {
	yc, err := ythttp.NewClient(&yt.Config{Proxy: "locke"})
	if err != nil {
		return err
	}
	lock := ytlock.NewLock(yc, cfg.LockeNodeYtPath)
	lost, err := lock.Acquire(ctx)
	for err != nil {
		log.Infof("Failed to acquire lock: %s", err)
		time.Sleep(time.Duration(cfg.SleepAfterIterationSeconds) * time.Second)
		lost, err = lock.Acquire(ctx)
	}
	go func() {
		<-lost // Waiting if transaction aborted
		log.Fatal("Transaction aborted")
	}()

	ypClient, err := ypclient.NewYpClient(cfg.YpClusters, cfg.FetchYpPodsBatchSize)
	if err != nil {
		return err
	}
	hqClient := NewHQClient(cfg.StrippedHQClusters, cfg.StrippedHQServices, cfg.YpClusters, cfg.RPCConfig, cfg.CustomYpClusterSuffixes)

	for {
		err = SyncPods(ctx, ypClient, hqClient, cfg.MaxPodsToUpdateOnIteration, dryRun)
		if err != nil {
			log.Errorf("Failed to process: %s", err.Error())
		}
		log.Infof("Iteration finished, sleep for %d seconds", cfg.SleepAfterIterationSeconds)
		time.Sleep(time.Duration(cfg.SleepAfterIterationSeconds) * time.Second)
	}
}
