package hqsyncer

import (
	"context"

	"a.yandex-team.ru/infra/nanny2/pkg/log"
	"a.yandex-team.ru/infra/nanny2/pkg/ypclient"
	pb "a.yandex-team.ru/yp/go/proto/hq"
	"a.yandex-team.ru/yt/go/yson"
)

func SyncPods(ctx context.Context, ypClient ypclient.YpClientInterface, hqClient *HQClient, MaxPodsToUpdateOnIteration int, dryRun bool) error {
	pods, err := ypClient.SelectHQSyncerPods(ctx)
	if err != nil {
		return err
	}

	updated := 0
	for _, pod := range pods {
		var serviceID string
		err = yson.Unmarshal(pod.Pod.Labels.Attributes[0].Value, &serviceID)
		if err != nil {
			log.Errorf("Failed to parse service ID from [%s]%s label: %s", pod.Cluster, pod.Pod.Meta.Id, err)
			continue
		}
		log.Infof("Processing [%s]%s, service %s, node %s", pod.Cluster, pod.Pod.Meta.Id, serviceID, *pod.Pod.Spec.NodeId)

		if dryRun {
			log.Infof("DryRun is true, continue")
			continue
		}

		log.Infof("Updating [%s]%s instance allocation", pod.Cluster, pod.Pod.Meta.Id)

		reqPb := &pb.UpdateInstanceAllocationRequest{
			Id:       hqClient.GetHQPodID(pod.Cluster, pod.Pod.Meta.Id, serviceID),
			NodeName: *pod.Pod.Spec.NodeId,
		}
		_, err = hqClient.ClientFactory[pod.Cluster].UpdateInstanceAllocation(ctx, reqPb)
		if err != nil {
			log.Errorf("Failed to update [%s]%s instance allocation: %s", pod.Cluster, pod.Pod.Meta.Id, err)
			continue
		}
		updated += 1

		ysonValue, err := yson.Marshal(*pod.Pod.Status.Scheduling.LastUpdated)
		if err != nil {
			log.Errorf("Failed to make yson from [%s]%s Status.Scheduling.LastUpdated: %s", pod.Cluster, pod.Pod.Meta.Id, err)
			continue
		}
		log.Infof("Updating [%s]%s pod label", pod.Cluster, pod.Pod.Meta.Id)
		err = ypClient.SetPodLabel(ctx, pod.Cluster, pod.Pod.Meta.Id, "hq_synced_eviction_time", ysonValue)
		if err != nil {
			log.Errorf("Failed to set [%s]%s pod label: %s", pod.Cluster, pod.Pod.Meta.Id, err)
			continue
		}
		if updated >= MaxPodsToUpdateOnIteration {
			log.Infof("Updated %v pods, early finish the iteration", updated)
			break
		}
	}
	return nil
}
