// Implements a deflate compression handler middleware for Negroni.

package middleware

import (
	"compress/zlib"
	"io/ioutil"
	"net/http"
	"strings"
	"sync"
)

// These compression constants are copied from the compress/gzip package.
const (
	encodingDeflate = "deflate"

	headerAcceptEncoding  = "Accept-Encoding"
	headerContentEncoding = "Content-Encoding"
	headerContentLength   = "Content-Length"
	headerVary            = "Vary"
	headerSecWebSocketKey = "Sec-WebSocket-Key"

	BestCompression    = zlib.BestCompression
	BestSpeed          = zlib.BestSpeed
	DefaultCompression = zlib.DefaultCompression
	NoCompression      = zlib.NoCompression
)

// zlibResponseWriter is the ResponseWriter that negroni.ResponseWriter is
// wrapped in.
type zlibResponseWriter struct {
	w *zlib.Writer
	http.ResponseWriter
}

// Write writes bytes to the gzip.Writer.
func (zrw zlibResponseWriter) Write(b []byte) (int, error) {
	return zrw.w.Write(b)
}

// handler struct contains the ServeHTTP method
type handler struct {
	pool sync.Pool
	next http.Handler
}

// Gzip returns a handler which will handle the Gzip compression in ServeHTTP.
// Valid values for level are identical to those in the compress/gzip package.
func Gzip(next http.Handler) http.Handler {
	h := &handler{
		next: next,
	}
	h.pool.New = func() interface{} {
		z, err := zlib.NewWriterLevel(ioutil.Discard, DefaultCompression)
		if err != nil {
			panic(err)
		}
		return z
	}
	return h
}

// ServeHTTP wraps the http.ResponseWriter with a gzip.Writer.
func (h *handler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	// Skip compression if the client doesn't accept gzip encoding.
	if !strings.Contains(r.Header.Get(headerAcceptEncoding), encodingDeflate) {
		h.next.ServeHTTP(w, r)
		return
	}
	// Skip compression if client attempt WebSocket connection
	if len(r.Header.Get(headerSecWebSocketKey)) > 0 {
		h.next.ServeHTTP(w, r)
		return
	}
	// Skip compression if already compressed
	if w.Header().Get(headerContentEncoding) != "" {
		h.next.ServeHTTP(w, r)
		return
	}

	// Retrieve gzip writer from the pool. Reset it to use the ResponseWriter.
	// This allows us to re-use an already allocated buffer rather than
	// allocating a new buffer for every request.
	// We defer g.pool.Put here so that the gz writer is returned to the
	// pool if any thing after here fails for some reason (functions in
	// next could potentially panic, etc)
	z := h.pool.Get().(*zlib.Writer)
	defer h.pool.Put(z)
	z.Reset(w)

	// Set the appropriate gzip headers.
	headers := w.Header()
	headers.Set(headerContentEncoding, encodingDeflate)
	headers.Set(headerVary, headerAcceptEncoding)

	// Wrap the original http.ResponseWriter with negroni.ResponseWriter
	// and create the gzipResponseWriter.
	grw := zlibResponseWriter{
		z,
		w,
	}

	// Call the next handler supplying the gzipResponseWriter instead of
	// the original.
	h.next.ServeHTTP(grw, r)

	// Delete the content length after we know we have been written to.
	grw.Header().Del(headerContentLength)

	_ = z.Close()
}
