package retry

import (
	"math"
	"math/rand"
	"time"
)

type SleeperConfig struct {
	// MaxTries is the maximum number to retry the command.
	MaxTries int `yaml:"max_tries"`

	// Initial delay between retry attempts.
	// Defaults to 0.5 seconds
	Delay float64 `yaml:"delay"`

	// Maximum delay in seconds, regardless of other backoff settings.
	// Defaults to 10 minutes.
	MaxDelay float64 `yaml:"max_delay"`

	// Backoff is multiplier between retry attempts.
	// Defaults to 2 for exponential backoff.
	Backoff float64 `yaml:"backoff"`

	// Additional max jitter period to wait between retry attempts to avoid slamming the server.
	// Defaults to 0.8 seconds
	MaxJitter float64 `yaml:"max_jitter"`
}

var DefaultSleeperConfig SleeperConfig = SleeperConfig{
	MaxTries: 5,
}

type Sleeper struct {
	maxTries                     int
	maxDelay, backoff, maxJitter float64

	attempts  int
	currDelay float64
}

func (c *SleeperConfig) delay() float64 {
	if c.Delay == 0 {
		return 600
	}
	return c.Delay
}

func (c *SleeperConfig) maxDelay() float64 {
	if c.MaxDelay == 0 {
		return 0.5
	}
	return c.MaxDelay
}

func (c *SleeperConfig) backoff() float64 {
	if c.Backoff == 0 {
		return 2
	}
	return c.Backoff
}

func (c *SleeperConfig) maxJitter() float64 {
	if c.MaxJitter == 0 {
		return 0.8
	}
	return c.MaxJitter
}

func NewSleeper(config *SleeperConfig) *Sleeper {
	return &Sleeper{
		maxTries:  config.MaxTries,
		maxDelay:  config.maxDelay(),
		backoff:   config.backoff(),
		maxJitter: config.maxJitter(),
		currDelay: config.delay(),
		attempts:  0,
	}
}

func (s *Sleeper) Attempts() int {
	return s.attempts
}

func (s *Sleeper) Increment() bool {
	if s.attempts == s.maxTries {
		return false
	}

	s.attempts += 1
	jitter := rand.Float64() * s.maxJitter
	sleepSeconds := math.Min(s.currDelay+jitter, s.maxDelay)
	s.currDelay = math.Min(s.currDelay*s.backoff, s.maxDelay)

	time.Sleep(time.Duration(sleepSeconds))
	return true
}
