package rpc

import (
	"bytes"
	"context"
	"errors"
	"fmt"
	"io/ioutil"
	"net"
	"net/http"
	"path"
	"strings"
	"syscall"
	"time"

	"github.com/golang/protobuf/proto"
	"google.golang.org/grpc"

	"a.yandex-team.ru/infra/nanny2/pkg/retry"
	pb "a.yandex-team.ru/infra/nanny2/proto/rpc"
)

type ClientConfig struct {
	// RPCURL is url prefix for all client methods. Final url for method will be constructed
	// by concatenating RPCURL with method name like "https://awacs.yandex-team.ru/api/" + "ListNamespaces"
	RPCURL string `yaml:"url"`

	// OauthToken is the authorization token will be placed in "Authorization" header. Has the format "AQAD-..."
	OauthToken string `yaml:"token"`

	// Retry429 indicates the need for retrying request if response status code is 429
	Retry429 bool `yaml:"retry_429"`

	// Retry5xx indicates the need for retrying request if response status code in 500-599
	Retry5xx bool `yaml:"retry_5xx"`

	// RetryConnectionErrors indicates the need for retrying request if connection has failed
	RetryConnectionErrors bool `yaml:"retry_connection_errors"`

	// RequestTimeout includes connection time and reading the response body.
	RequestTimeout int `yaml:"request_timeout"`

	// ConnectionTimeout is the maximum amount of time a dial will wait for a connect to complete
	ConnectionTimeout int `yaml:"connection_timeout"`

	// SleeperConfig implement settings for retrying failed requests
	// Defaults to retry.DefaultSleeperConfig
	RetrySleeperConfig retry.SleeperConfig `yaml:"retry_sleeper"`
}

// Client is an implementation of grpc.ClientConnInterface that supports only unary RPCs.
type Client struct {
	rpcURL, oauthToken                        string
	retry429, retry5xx, retryConnectionErrors bool
	retrySleeperConfig                        retry.SleeperConfig

	httpClient http.Client
}

func NewClient(config *ClientConfig) *Client {
	httpClient := http.Client{
		Timeout: time.Duration(config.RequestTimeout) * time.Second,
	}

	if config.ConnectionTimeout != 0 {
		httpClient.Transport = &http.Transport{
			Dial: (&net.Dialer{Timeout: time.Duration(config.ConnectionTimeout) * time.Second}).Dial,
		}
	}

	client := &Client{
		rpcURL:                strings.TrimRight(config.RPCURL, "/"),
		oauthToken:            config.OauthToken,
		retry429:              config.Retry429,
		retry5xx:              config.Retry5xx,
		retryConnectionErrors: config.RetryConnectionErrors,
		httpClient:            httpClient,
	}
	retrySleeperConfig := config.RetrySleeperConfig
	if retrySleeperConfig.MaxTries == 0 {
		retrySleeperConfig = retry.DefaultSleeperConfig
	}
	client.retrySleeperConfig = retrySleeperConfig

	return client
}

func (c *Client) isRetryNeeded(statusCode int) bool {
	if c.retry429 && statusCode == http.StatusTooManyRequests {
		return true
	}
	if c.retry5xx && 500 <= statusCode && statusCode < 600 {
		return true
	}
	return false
}

type ClientError struct {
	Status       string
	Message      string
	Reason       string
	Code         int32
	RetriesCount int
}

func (e *ClientError) Error() string {
	return fmt.Sprintf("bad response status: %d (%s); %s; %s", e.Code, e.Status, e.Reason, e.Message)
}

func (c *Client) Invoke(ctx context.Context, method string, args interface{}, reply interface{}, opts ...grpc.CallOption) error {
	reqPb := args.(proto.Message)
	respPb := reply.(proto.Message)

	url := fmt.Sprintf("%s/%s/", c.rpcURL, path.Base(method))

	data, err := proto.Marshal(reqPb)
	if err != nil {
		return err
	}
	req, err := http.NewRequest("POST", url, bytes.NewBuffer(data))
	if err != nil {
		return err
	}
	req.Header.Add("Accept", "application/x-protobuf")
	req.Header.Add("Content-Type", "application/x-protobuf")
	req.Header.Add("User-Agent", "a.yandex-team.ru/infra/nanny2/pkg/rpc")
	req.Header.Add("Authorization", fmt.Sprintf("OAuth %s", c.oauthToken))

	retrySleeper := retry.NewSleeper(&c.retrySleeperConfig)
	for {
		resp, err := c.httpClient.Do(req)
		if err != nil {
			if errors.Is(err, syscall.ECONNREFUSED) && c.retryConnectionErrors && retrySleeper.Increment() {
				continue
			}
			return fmt.Errorf("failed to post request after %v retries: %v", retrySleeper.Attempts(), err)
		}

		defer resp.Body.Close()
		if resp.StatusCode == http.StatusOK {
			body, err := ioutil.ReadAll(resp.Body)
			if err != nil {
				return err
			}
			err = proto.Unmarshal(body, respPb)
			if err != nil {
				return err
			}
			return nil
		}

		if c.isRetryNeeded(resp.StatusCode) && retrySleeper.Increment() {
			continue
		}

		// There is no tries remained or retry is not needed

		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return err
		}
		if resp.Header.Get("Content-Type") != "application/x-protobuf" {
			if resp.StatusCode == http.StatusGatewayTimeout {
				return fmt.Errorf("invalid response after %v retries: %d: %s", retrySleeper.Attempts(), resp.StatusCode, body)
			}
			return fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode, body)
		}

		statusPb := &pb.Status{}
		err = proto.Unmarshal(body, statusPb)
		if err != nil {
			return fmt.Errorf("invalid response from server: %d: %s", resp.StatusCode, body)
		}
		return &ClientError{
			Code:         statusPb.Code,
			Status:       statusPb.Status,
			Reason:       statusPb.Reason,
			Message:      statusPb.Message,
			RetriesCount: retrySleeper.Attempts(),
		}
	}
}

func (c *Client) NewStream(ctx context.Context, desc *grpc.StreamDesc, method string, opts ...grpc.CallOption) (grpc.ClientStream, error) {
	panic("streaming RPC is not supported in infra/nanny2/pkg/rpc client")
}
