package storage

import (
	"context"
	"github.com/golang/protobuf/proto"
)

type UpdateFunc func(input Storable) error

// Interface offers a common interface for object marshaling/unmarshaling operations and
// hides all the storage-related operations behind it.
type Interface interface {
	// Create adds a new object at a key unless it already exists.
	// If error is returned and out is not nil, out will be
	// set to the read value from database.
	Create(ctx context.Context, key string, obj Storable, out Storable) error

	Update(ctx context.Context, key string, obj Storable) error

	UpdateIfMatch(ctx context.Context, key string, obj Storable, version string, out Storable) error

	// GuaranteedUpdate loads object at a key, calls user update function
	// and saves. Can call update function several times in case of contention.
	GuaranteedUpdate(ctx context.Context, key string, tryUpdate UpdateFunc) error

	// Delete removes the specified key and returns the value that existed at that spot.
	// If key didn't exist, it will return NotFound storage error.
	Delete(ctx context.Context, key string, out Storable) error

	// Watch begins watching the specified key. Events are decoded into API objects,
	// and sent down to returned watch.Interface.
	// resourceVersion may be used to specify what version to begin watching,
	// which should be the current resourceVersion, and no longer rv+1
	// (e.g. reconnecting without missing any updates).
	Watch(ctx context.Context, key string, resourceVersion string) (WatchInterface, error)

	// WatchList begins watching all items in store. Items are decoded into API
	// objects and sent down to returned watch.Interface.
	// resourceVersion may be used to specify what version to begin watching,
	// which should be the current resourceVersion, and no longer rv+1
	// (e.g. reconnecting without missing any updates).
	WatchList(ctx context.Context, resourceVersion string) (WatchInterface, error)

	// Get unmarshals value found at key into objPtr. On a not found error, will either
	// return a zero object of the requested type, or an error, depending on ignoreNotFound.
	// Treats empty responses and nil response nodes exactly like a not found error.
	Get(ctx context.Context, key string, objPtr Storable) error

	// List returns a slice of all objects in store
	List(ctx context.Context) (objList []Storable, err error)

	// Status returns error if underlying storage is unavailable
	Status(ctx context.Context) error
}

// Interface copied from protobuf, not sure what to use
type Storable interface {
	proto.Message
}

type ObjectCreator interface {
	SetVersion(item interface{}, version string)
	Version(item interface{}) string
	New() Storable
}
