package storage

import (
	pb "a.yandex-team.ru/yp/go/proto/hq"
	"sync"
)

// Interface can be implemented by anything that knows how to watch and report changes.
type WatchInterface interface {
	// Stops watching. Will close the channel returned by ResultChan(). Releases
	// any resources used by the watch.
	Stop()

	// Returns a chan which will receive all the events. If an error occurs
	// or Stop() is called, this channel will be closed, in which case the
	// watch should be completely cleaned up.
	ResultChan() <-chan *Event
}

// EventType defines the possible types of events.
type EventType string

const (
	Added    EventType = "ADDED"
	Modified EventType = "MODIFIED"
	Deleted  EventType = "DELETED"
	Error    EventType = "ERROR"
)

// Event represents a single event to a watched resource.
type Event struct {
	Type EventType
	// Object is:
	//  * If Type is Added or Modified: the new state of the object.
	//  * If Type is Deleted: the state of the object immediately before deletion.
	//  * If Type is Error: nil
	Object interface{}
	Status *pb.Status
	// Revision is set for removed object, because in other cases this information
	// can be retrieved from storable object itself.
	Revision string
}

type emptyWatch chan *Event

// NewEmptyWatch returns a watch interface that returns no results and is closed.
// May be used in certain error conditions where no information is available but
// an error is not warranted.
func NewEmptyWatch() WatchInterface {
	ch := make(chan *Event)
	close(ch)
	return emptyWatch(ch)
}

// Stop implements Interface
func (w emptyWatch) Stop() {
}

// ResultChan implements Interface
func (w emptyWatch) ResultChan() <-chan *Event {
	return chan *Event(w)
}

// FakeWatcher lets you test anything that consumes a watch.Interface; threadsafe.
type FakeWatcher struct {
	result  chan Event
	Stopped bool
	sync.Mutex
}

func NewFake() *FakeWatcher {
	return &FakeWatcher{
		result: make(chan Event),
	}
}

// Stop implements Interface.Stop().
func (f *FakeWatcher) Stop() {
	f.Lock()
	defer f.Unlock()
	if !f.Stopped {
		close(f.result)
		f.Stopped = true
	}
}

func (f *FakeWatcher) ResultChan() <-chan Event {
	return f.result
}

// Add sends an add event.
func (f *FakeWatcher) Add(obj Storable) {
	f.result <- Event{Added, obj, nil, ""}
}

// Modify sends a modify event.
func (f *FakeWatcher) Modify(obj Storable) {
	f.result <- Event{Modified, obj, nil, ""}
}

// Delete sends a delete event.
func (f *FakeWatcher) Delete(lastValue Storable) {
	f.result <- Event{Deleted, lastValue, nil, ""}
}

// Error sends an Error event.
func (f *FakeWatcher) Error(errValue Storable) {
	f.result <- Event{Error, errValue, nil, ""}
}

// Action sends an event of the requested type, for table-based testing.
func (f *FakeWatcher) Action(action EventType, obj Storable) {
	f.result <- Event{action, obj, nil, ""}
}
