package ypclient

import (
	"context"
	"fmt"
	"strings"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
	"a.yandex-team.ru/yp/go/yp/yperrors"
	"a.yandex-team.ru/yt/go/proto/core/ytree"
	"a.yandex-team.ru/yt/go/yterrors"
)

type YpClient struct {
	ypClientMap map[string]*yp.Client
	batchSize   int32
	clusters    []string
}

func NewYpClientWithOption(clusters []string, batchSize int32, option yp.Option) (*YpClient, error) {
	ypClientMap := make(map[string]*yp.Client)

	for _, cluster := range clusters {
		client, err := yp.NewClient(cluster, option)
		if err != nil {
			return nil, err
		}
		ypClientMap[strings.ToLower(cluster)] = client
	}
	return &YpClient{
		ypClientMap: ypClientMap,
		batchSize:   batchSize,
		clusters:    clusters,
	}, nil

}

func NewYpClient(clusters []string, batchSize int32) (ypClient *YpClient, err error) {
	return NewYpClientWithOption(clusters, batchSize, yp.WithSystemAuthToken())
}

func NewYpClientToken(clusters []string, batchSize int32, token string) (*YpClient, error) {
	return NewYpClientWithOption(clusters, batchSize, yp.WithAuthToken(token))
}

func (c *YpClient) GetClusters() []string {
	return c.clusters
}

func (c *YpClient) SetPodLabel(ctx context.Context, cluster string, id string, key string, value []byte) error {
	request := yp.UpdateObjectRequest{
		ObjectType: yp.ObjectTypePod,
		ObjectID:   id,
		SetUpdates: []yp.SetObjectUpdate{
			{
				Path:   "/labels/" + key,
				Object: value,
			},
		},
	}
	_, err := c.ypClientMap[cluster].UpdateObject(ctx, request)
	return err
}

func (c *YpClient) SelectHQSyncerPods(ctx context.Context) (pods []*YpPod, err error) {
	ypPods := make([]*YpPod, 0)
	for cluster, client := range c.ypClientMap {
		request := yp.SelectPodsRequest{
			Format: yp.PayloadFormatYson, // Use PayloadFormatYson, because we select only fields
			Filter: fmt.Sprintf("[/labels/deploy_engine]='YP_LITE' and parse_uint64(string([/labels/hq_synced_eviction_time]))!=[/status/scheduling/last_updated]" +
				" and [/spec/node_id]!='' and [/status/scheduling/state]='assigned'"),
			Selectors: []string{"/meta/id", "/spec/node_id", "/status/scheduling/last_updated", "/labels/nanny_service_id"},
			Limit:     c.batchSize,
		}
		for {
			response, err := client.SelectPods(ctx, request)
			if err != nil {
				return nil, err
			}
			for response.Next() {
				ypPod := &YpPod{
					Pod: ypapi.TPod{
						Meta: &ypapi.TPodMeta{},
						Spec: &ypapi.TPodSpec{},
						Status: &ypapi.TPodStatus{
							Scheduling: &ypapi.TPodStatus_TScheduling{},
						},
						Labels: &ytree.TAttributeDictionary{
							Attributes: []*ytree.TAttribute{
								{
									Key:   proto.String("nanny_service_id"),
									Value: []byte(""),
								},
							},
						},
					},
					Cluster: cluster,
				}
				err := response.Fill(&ypPod.Pod.Meta.Id, &ypPod.Pod.Spec.NodeId, &ypPod.Pod.Status.Scheduling.LastUpdated, &ypPod.Pod.Labels.Attributes[0].Value)
				if err != nil {
					return nil, err
				}
				ypPods = append(ypPods, ypPod)
			}
			if int32(response.Count()) < c.batchSize {
				break
			}
			request.ContinuationToken = response.ContinuationToken()
		}
	}
	return ypPods, nil
}

func (c *YpClient) ListPods(ctx context.Context, podSetID string) (pods []*YpPod, err error) {
	ypPods := make([]*YpPod, 0)
	for cluster, client := range c.ypClientMap {
		continuationToken := ""
		for {
			request := yp.SelectPodsRequest{
				Format:    yp.PayloadFormatProto,
				Filter:    fmt.Sprintf("[/meta/pod_set_id] = '%s'", podSetID),
				Selectors: []string{"/meta", "/spec", "/status/dns"},
				Limit:     c.batchSize,
			}
			if len(continuationToken) > 0 {
				request.ContinuationToken = continuationToken
			}
			response, err := client.SelectPods(ctx, request)
			if err != nil {
				return nil, err
			}
			for response.Next() {
				ypPod := &YpPod{
					Pod: ypapi.TPod{
						Meta: &ypapi.TPodMeta{},
						Spec: &ypapi.TPodSpec{},
						Status: &ypapi.TPodStatus{
							Dns: &ypapi.TPodStatus_TDns{},
						},
					},
					Cluster: cluster,
				}
				err := response.Fill(ypPod.Pod.Meta, ypPod.Pod.Spec, ypPod.Pod.Status.Dns)
				if err != nil {
					return nil, err
				}
				ypPods = append(ypPods, ypPod)
			}
			if int32(response.Count()) < c.batchSize {
				break
			}
			continuationToken = response.ContinuationToken()
		}
	}
	return ypPods, nil
}

func (c *YpClient) FindPod(ctx context.Context, podID string, clusters ...string) (pod *YpPod, err error) {
	clustersForSearch := c.clusters
	if len(clusters) > 0 {
		clustersForSearch = clusters
	}
	for _, cluster := range clustersForSearch {
		client, clientExists := c.ypClientMap[cluster]
		if !clientExists {
			return nil, fmt.Errorf("client for cluster: %s not found", cluster)
		}

		req := yp.GetPodRequest{
			ID:        podID,
			Format:    yp.PayloadFormatProto,
			Selectors: []string{"/meta", "/spec", "/status/dns"},
		}
		response, err := client.GetPod(ctx, req)
		if err != nil {
			ytErr, ok := err.(*yterrors.Error)
			if ok && ytErr.Code == yperrors.CodeNoSuchObject {
				continue
			} else {
				return nil, err
			}
		}
		ypPod := &YpPod{
			Pod: ypapi.TPod{
				Meta: &ypapi.TPodMeta{},
				Spec: &ypapi.TPodSpec{},
				Status: &ypapi.TPodStatus{
					Dns: &ypapi.TPodStatus_TDns{},
				},
			},
			Cluster: cluster,
		}
		err = response.Fill(ypPod.Pod.Meta, ypPod.Pod.Spec, ypPod.Pod.Status.Dns)
		if err != nil {
			return nil, err
		}
		return ypPod, nil
	}
	return nil, nil
}

func (c *YpClient) ListServicePods(ctx context.Context, serviceID string) ([]*YpPod, error) {
	ypPods := make([]*YpPod, 0)
	podSetID := strings.ReplaceAll(serviceID, "_", "-")

	for cluster, client := range c.ypClientMap {
		request := yp.SelectPodsRequest{
			Format: yp.PayloadFormatProto,
			Filter: fmt.Sprintf("[/meta/pod_set_id] = '%s'", podSetID),
			Selectors: []string{
				"/meta",
				"/status/eviction",
				"/status/maintenance",
				"/status/scheduling",
				"/status/iss_conf_summaries",
				"/status/agent/iss_summary",
				"/spec/node_id",
				"/spec/iss/instances",
			},
			Limit:           c.batchSize,
			FetchRootObject: true,
		}

		for {
			response, err := client.SelectPods(ctx, request)
			if err != nil {
				return nil, err
			}
			for response.Next() {
				ypPod := &YpPod{
					Pod:     ypapi.TPod{},
					Cluster: cluster,
				}
				err := response.Fill(&ypPod.Pod)
				if err != nil {
					return nil, err
				}
				ypPods = append(ypPods, ypPod)
			}
			if int32(response.Count()) < c.batchSize {
				break
			}
			request.ContinuationToken = response.ContinuationToken()
		}
	}
	return ypPods, nil
}
