import time
import logging
import urllib.parse

from infra.nanny_ci.common import common_pb2
from infra.nanny_ci.approve_location.proto import approve_location_tasklet

from ci.tasklet.common.proto import service_pb2 as ci
from tasklet.services.yav.proto import yav_pb2 as yav

from infra.nanny.nanny_services_rest.nanny_services_rest.client import ServiceRepoClient


log = logging.getLogger(__name__)


class ApproveLocationImpl(approve_location_tasklet.ApproveLocationBase):
    @staticmethod
    def task_link(
        taskgroup_id: str,
        task_id: str = '',
        base_url: str = 'https://nanny.yandex-team.ru',
    ) -> str:
        if task_id:
            return urllib.parse.urljoin(
                base_url,
                f'ui/#/alemate/taskgroups/{taskgroup_id}/tasks/{task_id}/details/',
            )
        else:
            return urllib.parse.urljoin(
                base_url,
                f'ui/#/alemate/taskgroups/{taskgroup_id}/graph/',
            )

    def save_progress(
        self,
        message: str,
        status=ci.TaskletProgress.Status.RUNNING,
        nanny_installation: str = '',
        taskgroup_id: str = '',
        task_id: str = '',
        progress_value: float = 1.,
    ):
        url = self.task_link(
            base_url=nanny_installation or 'https://nanny.yandex-team.ru',
            taskgroup_id=taskgroup_id,
            task_id=task_id,
        )
        module = 'NANNY'

        progress = ci.TaskletProgress()
        progress.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress.id = 'approve_location'
        progress.progress = progress_value
        progress.text = message
        progress.module = module
        progress.url = url
        progress.status = status
        self.ctx.ci.UpdateProgress(progress)

    def get_token(self):
        secret_uid = self.input.context.secret_uid
        secret_key = 'nanny_ci.token'

        spec = yav.YavSecretSpec(uuid=secret_uid, key=secret_key)
        return self.ctx.yav.get_secret(spec, default_key=secret_key).secret

    def find_location(self) -> common_pb2.SnapshotLocation:
        matcher = self.input.location_match
        locations = list(filter(
            lambda loc: (
                (not matcher.nanny_installation or loc.nanny_installation == matcher.nanny_installation)
                and (not matcher.service_id or loc.service_id == matcher.service_id)
                and (not matcher.cluster or loc.cluster == matcher.cluster)
            ),
            self.input.approval_locations,
        ))
        if len(locations) > 1:
            raise Exception(f"Multiple locations match filter: {matcher}")
        elif not locations:
            raise Exception(f"No matching approval location found for filter: {matcher}")

        return locations[0]

    def check_location_finished(
        self,
        nanny_client: ServiceRepoClient,
        loc: common_pb2.SnapshotLocation,
    ) -> bool:
        response = nanny_client._request_with_retries(
            'GET',
            f'/v1/alemate/task_groups/{loc.taskgroup_id}/tasks/{loc.task_id}/',
        )

        log.debug("task %r status: %s", loc.task_id, response)
        progress = response.get('runtimeOptions', {}).get('taskProgress', {}).get('progress', 0) / 100.
        status = response.get('schedulerOptions', {}).get('status', 'RUNNING')
        progress_status = {
            'CANCELLED': ci.TaskletProgress.Status.FAILED,
            'FAILED': ci.TaskletProgress.Status.FAILED,
            'INVALID': ci.TaskletProgress.Status.FAILED,
            'DONE': ci.TaskletProgress.Status.SUCCESSFUL,
        }.get(status, ci.TaskletProgress.Status.RUNNING)

        self.save_progress(
            message=status,
            status=progress_status,
            nanny_installation=loc.nanny_installation,
            taskgroup_id=loc.taskgroup_id,
            task_id=loc.task_id,
            progress_value=progress,
        )
        self.output.state.success = status == 'DONE'
        self.output.state.message = status
        return status in ('CANCELLED', 'INVALID', 'DONE')

    def run(self):
        loc = None

        try:
            loc = self.find_location()
            log.info(
                "[nanny %s] approving: taskgroup=%r task=%r",
                loc.nanny_installation,
                loc.taskgroup_id,
                loc.task_id,
            )

            token = self.get_token()
            nanny_client = ServiceRepoClient(
                loc.nanny_installation,
                token=token,
                timeout=300,
            )

            response = nanny_client._request_with_retries(
                'POST',
                f'/v1/alemate/task_groups/{loc.taskgroup_id}/tasks/{loc.task_id}/commands/',
                data={'type': 'confirm'},
            )

            log.info("Approve succeeded: %s", response)
            self.output.state.success = True
            self.output.state.message = "Approved"

            self.save_progress(
                self.output.state.message,
                nanny_installation=loc.nanny_installation,
                taskgroup_id=loc.taskgroup_id,
                task_id=loc.task_id,
                progress_value=0.
            )

            while not self.check_location_finished(nanny_client, loc):
                time.sleep(5.)

        except Exception as e:
            self.output.state.success = False
            self.output.state.message = f"Approval failed: {e}"

            self.save_progress(
                str(e),
                nanny_installation=loc and loc.nanny_installation or '',
                taskgroup_id=loc and loc.taskgroup_id or '',
                status=ci.TaskletProgress.Status.FAILED,
            )
            raise
