import time
import hashlib
import logging
import urllib.parse

from infra.nanny_ci import util
from infra.nanny_ci.common import common_pb2
from infra.nanny_ci.update_service.proto import update_service_tasklet

from ci.tasklet.common.proto import service_pb2 as ci
from tasklet.services.yav.proto import yav_pb2 as yav

from nanny_tickets import releases_pb2, tickets_pb2, tickets_api_stub, tickets_api_pb2
from nanny_rpc_client import RetryingRpcClient
from infra.nanny.nanny_services_rest.nanny_services_rest.client import ServiceRepoClient

from sandbox.common.rest import Client as SandboxClient
from sandbox.common import auth


log = logging.getLogger(__name__)


class UpdateServiceImpl(update_service_tasklet.NannyUpdateServiceBase):
    steps: list
    sandbox_client: SandboxClient
    nanny_client: RetryingRpcClient
    nanny_repo_client: RetryingRpcClient
    nanny_rest_client: ServiceRepoClient

    @staticmethod
    def service_link(
        base_url: str = 'https://nanny.yandex-team.ru',
        service_id: str = '',
        snapshot_id: str = '',
        deploy_ticket_id: str = '',
        release_id: str = '',
    ) -> str:
        url = ''
        if release_id:
            url = urllib.parse.urljoin(base_url, f'ui/#/r/{release_id}/')
        elif deploy_ticket_id:
            url = urllib.parse.urljoin(base_url, f'ui/#/t/{deploy_ticket_id}/')
        elif service_id:
            if snapshot_id:
                url = urllib.parse.urljoin(base_url, f'ui/#/services/catalog/{service_id}/runtime_attrs_history/{snapshot_id}/')
            else:
                url = urllib.parse.urljoin(base_url, f'ui/#/services/catalog/{service_id}/')
        return url

    def save_progress(
        self,
        progress_ratio: float,
        message: str,
        step_id: str,
        status=ci.TaskletProgress.Status.RUNNING,
        snapshot_id: str = '',
        deploy_ticket_id: str = '',
        release_id: str = '',
    ):
        url = self.service_link(
            base_url=self.input.config.nanny_installation or 'https://nanny.yandex-team.ru',
            service_id=self.input.config.service_id,
            snapshot_id=snapshot_id,
            deploy_ticket_id=deploy_ticket_id,
            release_id=release_id,
        )
        module = 'NANNY' if url else ''

        progress = ci.TaskletProgress()
        progress.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress.id = step_id
        progress.progress = progress_ratio
        progress.text = message
        progress.module = module
        progress.url = url
        progress.status = status
        self.ctx.ci.UpdateProgress(progress)

    def get_token(self):
        secret_uid = self.input.context.secret_uid
        secret_key = 'nanny_ci.token'

        spec = yav.YavSecretSpec(uuid=secret_uid, key=secret_key)
        return self.ctx.yav.get_secret(spec, default_key=secret_key).secret

    def prepare_clients(self):
        token = self.get_token()
        self.sandbox_client = SandboxClient(auth=auth.OAuth(token))
        self.nanny_client = RetryingRpcClient(
            urllib.parse.urljoin(self.input.config.nanny_installation, '/api/tickets'),
            oauth_token=token,
            request_timeout=300,
            retry_5xx=True,
            retry_connection_errors=True,
        )
        self.nanny_repo_client = RetryingRpcClient(
            urllib.parse.urljoin(self.input.config.nanny_installation, '/api/repo'),
            oauth_token=token,
            request_timeout=300,
            retry_5xx=True,
            retry_connection_errors=True,
        )
        self.nanny_rest_client = ServiceRepoClient(
            self.input.config.nanny_installation,
            token=token,
            timeout=300,
        )

    def prepare_steps(self):
        self.steps = [
            (self.validate, "Validating config"),
            (self.create_release, "Creating release"),
            (self.create_ticket, "Creating deploy ticket"),
            (self.commit_ticket, "Committing deploy ticket"),
            (self.prepare_snapshot, "Preparing snapshot if target_status=PREPARED"),
            (self.wait_for_preparation, "Waiting for snapshot prepare if target_status=PREPARED"),
            (self.activate_snapshot, "Activating snapshot if target_status=ACTIVE"),
            (self.collect_locations, "Getting affected locations"),
            (self.wait_for_activate, "Waiting for deploy to finish"),
        ]

    def validate(self):
        if self.input.config.target_status == common_pb2.UNKNOWN:
            raise Exception("Tasklet misconfigured: `target_status` must not be UNKNOWN")

        if len(self.input.config.patches) and self.input.config.docker_image:
            raise Exception("Both `patches` and `docker_image` specified, while they cannot be mixed in one release.")
        elif not len(self.input.config.patches) and not self.input.config.docker_image:
            raise Exception("Neither `patches` nor `docker_image` specified.")

    def create_release(self):
        self.output.release.object_id = create_release(
            self.nanny_client,
            self.sandbox_client,
            self.input,
        )
        log.info("created release %r", self.output.release.object_id)
        self.save_progress(
            progress_ratio=1.,
            message='Release link',
            step_id='release',
            status=ci.TaskletProgress.Status.SUCCESSFUL,
            release_id=self.output.release.object_id,
        )

    def create_ticket(self):
        self.output.ticket.object_id = create_ticket(
            self.nanny_client,
            self.input,
            self.output.release.object_id,
            self.input.config.queue_id or 'CI',
        )
        log.info("created ticket %r", self.output.ticket.object_id)
        self.save_progress(
            progress_ratio=1.,
            message='Ticket link',
            step_id='ticket',
            status=ci.TaskletProgress.Status.SUCCESSFUL,
            deploy_ticket_id=self.output.ticket.object_id,
        )

    def commit_ticket(self):
        self.output.ticket.commit_event_id, self.output.snapshot.snapshot_id = commit_ticket(
            self.nanny_client,
            self.output.ticket.object_id,
        )
        log.info("created snapshot %r", self.output.snapshot.snapshot_id)

        self.save_progress(
            progress_ratio=1.,
            message='Snapshot link',
            step_id='snapshot',
            status=ci.TaskletProgress.Status.SUCCESSFUL,
            snapshot_id=self.output.snapshot.snapshot_id,
        )

        self.output.snapshot.nanny_installation = self.input.config.nanny_installation
        self.output.snapshot.service_id = self.input.config.service_id
        self.output.snapshot.ticket_id = self.output.ticket.object_id
        self.output.ticket.committed = True

    def prepare_snapshot(self):
        if self.input.config.target_status == common_pb2.COMMITTED:
            return

        prepare_recipe = self.input.config.prepare_recipe
        if not prepare_recipe:
            prepare_recipe, _ = resolve_default_recipes(
                self.nanny_rest_client,
                self.input.config.service_id,
            )

        util.set_snapshot_state(
            nanny_client=self.nanny_rest_client,
            service_id=self.output.snapshot.service_id,
            snapshot_id=self.output.snapshot.snapshot_id,
            target_state='PREPARED',
            prepare_recipe=prepare_recipe,
            ticket=self.output.snapshot.ticket_id,
            comment=self.input.config.release_title or f"CI Build: {util.make_build_name(input.context)}",
        )

    def wait_for_preparation(self):
        if self.input.config.target_status == common_pb2.COMMITTED:
            return

        timeout = 24 * 60 * 60
        deployed = util.wait_for_snapshot_status(
            self.nanny_repo_client,
            self.output.snapshot.service_id,
            self.output.snapshot.snapshot_id,
            state='PREPARED',
            timeout=timeout,
        )
        if not deployed:
            raise Exception(f"Wait for deploy failed (not finished in {timeout} seconds)")

    def activate_snapshot(self):
        if self.input.config.target_status != common_pb2.ACTIVE:
            return

        prepare_recipe = self.input.config.prepare_recipe
        activate_recipe, parameters = util.get_activate_recipe(self.input.config)
        if not prepare_recipe or not activate_recipe:
            default_prepare_recipe, default_activate_recipe = resolve_default_recipes(
                self.nanny_rest_client,
                self.input.config.service_id,
            )
            prepare_recipe = prepare_recipe or default_prepare_recipe
            activate_recipe = activate_recipe or default_activate_recipe

        (
            self.output.ticket.activate_event_id,
            self.output.snapshot.snapshot_id,
        ) = util.activate_snapshot(
            self.nanny_client,
            self.output.ticket.object_id,
            prepare_recipe,
            activate_recipe,
            recipe_parameters=parameters,
        )
        self.output.ticket.commit_event_id = self.output.ticket.activate_event_id
        self.output.ticket.activated = True
        log.info("activated snapshot %r", self.output.snapshot.snapshot_id)

    def collect_locations(self):
        if self.input.config.target_status != common_pb2.ACTIVE:
            return

        activate_prefix = util.get_activate_prefix(self.input.config)

        for taskgroup_id, cluster, task_id in util.collect_locations(
            self.nanny_rest_client,
            self.output.snapshot.service_id,
            self.output.snapshot.snapshot_id,
            activate_prefix=activate_prefix,
        ):
            location = self.output.approval_locations.add()
            location.nanny_installation = self.input.config.nanny_installation
            location.service_id = self.input.config.service_id
            location.taskgroup_id = taskgroup_id
            location.cluster = cluster
            location.task_id = task_id

        log.info(
            "collected clusters: %s",
            [location.cluster for location in self.output.approval_locations],
        )

    def wait_for_activate(self):
        if self.input.config.target_status != common_pb2.ACTIVE:
            return

        if len(self.output.approval_locations):
            log.info(
                "Do not waiting for activation: approves required for locations %s",
                [location.cluster for location in self.output.approval_locations],
            )
            return

        timeout = 24 * 60 * 60
        deployed = wait_for_activate(
            self.nanny_client,
            self.output.ticket.object_id,
            timeout=timeout,
        )
        if not deployed:
            raise Exception(f"Wait for deploy failed (not finished in {timeout} seconds)")

    def run(self):
        self.prepare_clients()
        self.prepare_steps()

        for idx, (step, message) in enumerate(self.steps):
            log.info("[%2d] %s", idx, message)
            self.save_progress(progress_ratio=idx / len(self.steps), message=message, step_id='progress')
            try:
                step()
            except Exception as e:
                self.save_progress(
                    progress_ratio=1.,
                    message=f"Release failed: {e}",
                    step_id='progress',
                    status=ci.TaskletProgress.Status.FAILED,
                )
                self.output.state.success = False
                self.output.state.message = f"Step {idx} failed: {e}"
                raise

        self.save_progress(
            progress_ratio=1.,
            message="Done.",
            step_id='progress',
            status=ci.TaskletProgress.Status.SUCCESSFUL,
        )
        self.output.state.success = True
        self.output.state.message = "Success"


def _split_docker_image_name(image: str) -> tuple[str, str, str]:
    # TODO unify with same function in infra/deploy_ci/create_release/impl
    image, *tag_parts = image.rsplit(':', 1)
    if len(tag_parts) != 1:
        raise NotImplementedError("Docker images without exact tag are not supported yet")
    tag = tag_parts[0]

    registry, *image_parts = image.split('/', 1)
    if image_parts and '.' in registry:
        image = image_parts[0]
    else:
        registry = 'registry.yandex.net'

    return registry, image, tag


def make_id(input) -> str:
    flow_launch_id = hashlib.md5(input.context.job_instance_id.flow_launch_id.encode()).hexdigest()[:8]
    job_id = hashlib.md5(input.context.job_instance_id.job_id.encode()).hexdigest()[:8]

    return 'CI-{}-{}'.format(
        flow_launch_id,
        job_id,
    )


def create_release(
    nanny_client: RetryingRpcClient,
    sandbox_client: SandboxClient,
    input,
) -> str:
    release_proto = tickets_api_pb2.CreateReleaseRequest()
    release_proto.id = make_id(input)
    release_proto.meta.author = input.context.flow_triggered_by
    spec = release_proto.spec
    spec.title = input.config.release_title or f"CI Build: {util.make_build_name(input.context)}"
    spec.desc = input.config.release_description or (
        f'Autocheck for {util.make_build_name(input.context)}'
    )
    spec.startrek_ticket_ids.extend(
        issue.id
        for issue in input.context.launch_pull_request_info.issues
    )

    if len(input.config.patches):
        spec.type = releases_pb2.ReleaseSpec.SANDBOX_RELEASE
        spec_release = spec.sandbox_release

        assert len(input.sandbox_resources), "No sandbox resources given to make a release"
        resources = [
            (resource, sandbox_client.resource[resource.id][:])
            for resource in input.sandbox_resources
        ]

        log.info("fetched %d resources infos", len(resources))
        task_infos = {}
        for _, resource_info in resources:
            if resource_info['task']['id'] not in task_infos:
                task_info = task_infos[resource_info['task']['id']] = sandbox_client.task[resource_info['task']['id']][:]
                log.info("fetched task %s %r info", task_info['type'], task_info['id'])

        task_info = next(iter(task_infos.values()))
        spec_release.task_type = task_info['type']
        spec_release.task_id = str(task_info['id'])
        spec_release.release_author = sandbox_client.user.current.read()["login"]
        spec_release.release_type = input.config.release_type or 'testing'
        spec_release.task_author = task_info['author']
        spec_release.title = task_info['type']
        spec_release.desc = task_info['description']

        log.info("release author is %r", spec_release.release_author)

        resource_types = set()

        for resource, resource_info in resources:
            log.info("[resource %r] fetched info: %s", resource_info['id'], resource_info)
            if not resource_info['skynet_id']:
                log.info("[resource %r] skipped: no skynet_id available", resource_info['id'])
                continue

            resource_types.add(str(resource_info['type']))

            task_info = task_infos[resource_info['task']['id']]

            release_resource = spec_release.resources.add()
            release_resource.id = str(resource_info['id'])
            release_resource.type = str(resource_info['type'])
            release_resource.task_id = str(resource_info['task']['id'])
            release_resource.task_type = task_info['type']
            release_resource.description = resource_info['description'] or ''
            release_resource.skynet_id = resource_info['skynet_id']
            release_resource.arch = resource_info['arch'] or ''
            release_resource.file_md5 = resource_info['md5'] or ''

        spec_release.resource_types.extend(resource_types)

    elif input.config.docker_image:
        docker_registry, docker_image, docker_tag = _split_docker_image_name(input.config.docker_image)

        spec.type = releases_pb2.ReleaseSpec.DOCKER_RELEASE
        spec_release = spec.docker_release
        spec_release.image_name = docker_image
        spec_release.image_tag = docker_tag
        spec_release.registry = docker_registry
        spec_release.release_author = input.context.flow_triggered_by
        spec_release.release_type = input.config.release_type or 'testing'

    log.info("will create release with spec: %s", release_proto)
    stub = tickets_api_stub.TicketServiceStub(nanny_client)
    release = stub.create_release(release_proto).value
    log.info("created release value: %s", release)
    return release.id


def create_ticket(
    nanny_client: RetryingRpcClient,
    input,
    release_id: str,
    queue_id: str,
) -> str:
    ticket_proto = tickets_api_pb2.CreateTicketRequest()
    ticket_spec = ticket_proto.spec

    ticket_spec.queue_id = queue_id
    ticket_spec.title = input.config.release_title or f'CI Build: {util.make_build_name(input.context)}'
    ticket_spec.desc = ''  # TODO take PR or commit description
    ticket_spec.release_id = release_id
    ticket_spec.service_deployment.service_id = input.config.service_id

    for patch in input.config.patches:
        new_patch = ticket_spec.patches.add()
        new_patch.CopyFrom(patch)

    log.info("will create ticket with spec: %s", ticket_proto)
    stub = tickets_api_stub.TicketServiceStub(nanny_client)
    ticket = stub.create_ticket(ticket_proto).value
    log.info("created ticket value: %s", ticket)

    return ticket.id


def commit_ticket(
    nanny_client: RetryingRpcClient,
    ticket_id: str,
) -> tuple[str, str]:
    req = tickets_api_pb2.CreateTicketEventRequest()
    req.ticket_id = ticket_id
    req.spec.type = tickets_pb2.EventSpec.COMMIT_RELEASE_REQUEST
    req.spec.commit_release.SetInParent()

    stub = tickets_api_stub.TicketServiceStub(nanny_client)
    response = stub.create_ticket_event(req)
    return response.event.id, response.ticket.spec.service_deployment.snapshot_id


def resolve_default_recipes(nanny_client: ServiceRepoClient, service_id: str) -> tuple[str, str]:
    info_attrs = nanny_client.get_info_attrs(service_id)['content']
    prepare_recipes = info_attrs.get('prepare_recipes', [])
    if not prepare_recipes:
        default_prepare_recipe = ""
    else:
        default_prepare_recipe = next(iter(filter(lambda recipe: recipe['id'] == 'default', prepare_recipes)), prepare_recipes[0])['id']

    activate_recipes = info_attrs.get('recipes', {}).get('content', [])
    if not activate_recipes:
        raise Exception("Service has no activation recipes")

    default_activate_recipe = next(iter(filter(lambda recipe: recipe['id'] == 'default', activate_recipes)), activate_recipes[0])['id']

    return default_prepare_recipe, default_activate_recipe


def wait_for_activate(
    nanny_client: RetryingRpcClient,
    ticket_id: str,
    timeout: int = 24 * 60 * 60
) -> bool:
    deployed = False
    deadline = time.time() + timeout
    stub = tickets_api_stub.TicketServiceStub(nanny_client)

    while not deployed and time.time() <= deadline:
        req = tickets_api_pb2.GetTicketRequest(id=ticket_id)
        status = stub.get_ticket(req).value.status
        # TODO update progress

        status_name = tickets_pb2.TicketStatus.Status.Name(status.status)
        if status.status in (
            tickets_pb2.TicketStatus.CANCELLED,
            tickets_pb2.TicketStatus.DEPLOY_FAILED,
            tickets_pb2.TicketStatus.ROLLED_BACK,
        ):
            raise Exception(f"Deploy failed with status: {status_name}")

        deployed = status.status == tickets_pb2.TicketStatus.DEPLOY_SUCCESS

        if not deployed:
            time.sleep(max(0, min(5, deadline - time.time())))

    return deployed
