#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import logging
import signal
import sys

from infra.netconfig.checks import switch_port_changes
from infra.netconfig.checks import check_ra


def timeout_t(v):
    try:
        v = int(v)
    except Exception:
        raise argparse.ArgumentTypeError('timeout must be integer')
    if v < 0:
        raise argparse.ArgumentTypeError('timeout cannot be negative: {}'.format(v))
    if v > 3600:
        raise argparse.ArgumentTypeError('timeout is too large (max 3600): {}'.format(v))
    return v


def parse_args(argv):
    p = argparse.ArgumentParser(prog='yandex-netconfig-check', add_help=True)
    p.add_argument('-l', '--log-level', default='INFO', choices=['INFO', 'ERROR', 'DEBUG'])
    p.add_argument('-t', '--timeout',
                   type=timeout_t,
                   help='Terminate after timeout (sec). 0 - disable timeout. (default: %(default)s)',
                   default=120)
    return p.parse_args(argv)


def setup_logging(level):
    logging.basicConfig(stream=sys.stdout, level=logging.getLevelName(level))


def set_alarm(timeout):
    if timeout:
        signal.alarm(timeout)


def main():
    args = parse_args(sys.argv[1:])
    setup_logging(args.log_level)
    set_alarm(args.timeout)
    switch_port_changes.run()
    check_ra.run()


if __name__ == "__main__":
    main()
