#!/usr/bin/env python2
# -*- coding: utf-8 -*-

import os
import sys
import logging
import argparse
import socket

from infra.netconfig.lib import master
from infra.netconfig.lib import fixutil
from infra.netconfig.lib import orlyutil
from infra.netconfig.lib import jugglerutil
from infra.netconfig.proto import fixutil_pb2

SERVICE_NAME = 'netconfig_fixall'
ORLY_RULE = 'netconfig-fixall-apply'

LOG_FORMAT = '[%(asctime)s] %(levelname)s: %(message)s'


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '--global-dry-run',
        action='store_true',
        required=False,
        help='Don\'t fix anything.',
    )
    parser.add_argument(
        '--force',
        action='store_true',
        required=False,
        help='Dangerous! Ignore ORLY answer'
    )
    parser.add_argument(
        '--log-file',
        required=False,
        type=str,
        help='Set log file path',
    )
    args = parser.parse_args()

    if args.log_file:
        logging.basicConfig(filename=args.log_file, level=logging.INFO, format=LOG_FORMAT)
    else:
        logging.basicConfig(level=logging.INFO, format=LOG_FORMAT)

    logger = logging.getLogger(__name__)

    interfaces = fixutil.get_interfaces()
    # Filter ifaces if $INTERFACE variable exits
    logger.info('ENV: {}'.format(os.environ))
    env_iface = os.environ.get('IFACE', None)
    logger.info('ENV_IFACE: {}'.format(env_iface))
    if env_iface:
        env_mode = os.environ.get('MODE')
        if env_mode == 'start':
            interfaces = [i for i in interfaces if i.name == env_iface]
        elif env_mode == 'stop':
            fixutil.set_link_down(env_iface)
            sys.exit()

    logger.info('INTERFACES: {}'.format(interfaces))

    target_states = fixutil_pb2.TargetState()

    for iface in interfaces:
        try:
            base_interface_proto = fixutil.prepare_base_phys_interface_proto(iface)
            got_target_states = fixutil.target_states_from_l3segments(iface, base_interface_proto)
            target_states.interfaces.extend(got_target_states.interfaces)
        except master.NetconfigSkip:
            continue
        except fixutil.FixallPrefix6:
            jugglerutil.push_local(
                'CRIT',
                'prefix6 in network data, won\'t fix anything',
                SERVICE_NAME,
                ['yandex-netconfig-fixall'],
            )
            sys.exit(1)

    logger.info('Enforcing sysctls')
    fixutil.enforce_sysctls()  # Originates from RTCNETWORK-264
    results, juggler_mode = fixutil.fixall(target_states, dry_run=True)

    if results:
        str_results = ', '.join(results)
        logger.info('Pushing info to Juggler: {}'.format(str_results))  # eth0:routes, vlan777:link_mtu, vlan777:routes

        jugglerutil.push_local(
            juggler_mode,
            str_results,  # eth0:routes, vlan777:link_mtu, vlan777:routes
            SERVICE_NAME,
            ['yandex-netconfig-fixall'],
        )

        if not args.global_dry_run and fixutil.need_to_fix(results):  # If fixes not disabled globally via arguments - fix all
            if not args.force:
                logger.info('Checking if \'fixall\' operation is allowed in ORLY...')
                ok, err = orlyutil.start_operation(ORLY_RULE, socket.gethostname())
                if not ok:
                    logger.info('Operation is not allowed by {}: {}'.format(ORLY_RULE, err))
                    sys.exit(1)
                results = fixutil.fixall(target_states, dry_run=False)
            else:
                logger.info('FORCE MODE! Don\'t go to ORLY')
                results = fixutil.fixall(target_states, dry_run=False)
        else:
            logger.info('GlobalDryRun argument set. Don\'t fix anything')
    else:
        logger.info('Pushing OK to Juggler')  # eth0:routes, vlan777:link_mtu, vlan777:routes
        jugglerutil.push_local(
            'OK',
            '',
            SERVICE_NAME,
            ['yandex-netconfig-fixall'],
        )


if __name__ == '__main__':
    main()
