#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import logging
import socket
import sys
import json
import netaddr
import re
import os

from infra.netconfig.lib import master
from infra.netconfig.lib import stateutil
from infra.netconfig.lib import orlyutil
from infra.netconfig.lib import lldputil
from infra.netconfig.lib import fixutil
from infra.netconfig.lib import deviceutil
from infra.netconfig.lib import yasmutil
from infra.netconfig.lib import status

from infra.netconfig.proto import fixutil_pb2

from pyroute2 import IPRoute
from google.protobuf.json_format import MessageToJson

ORLY_RULE = 'netconfig-apply-routes'
ORLY_HACKS_RULE = 'netconfig-apply-hacks'


def run_verify(args):
    for iname in master.get_configured_interfaces():
        try:
            stateutil.verify_state(iname, master.get_state(iname))
            logging.info("OK")
        except master.NetconfigError as e:
            logging.error("%s" % e)
            sys.exit(1)


def run_hash_config(args):
    iface_obj = None

    if args.scripting:
        logging.disable(logging.CRITICAL)

    if args.iface:
        iface_objs = master.DebianInterfaces().get_iface_definitions(args.iface)
        if not iface_objs:
            logging.error("Cannot find iface %s" % args.iface)
            sys.exit(1)

        iface_obj = iface_objs[0]

    network_config = master.get_network_info(iface_obj, store=False)
    orig_hash, unhashed = stateutil.calculate_network_config_hash(network_config)
    if unhashed:
        logging.info("Found unhashed fields: %s" % str(unhashed))
        sys.exit(1)

    sys.stdout.write(orig_hash + '\n')


def run_apply_routes(args):
    if not args.orig_hash and not args.force:
        logging.error("Route config hash not specified, set --force to ignore")
        sys.exit(1)

    try:
        diff = stateutil.RoutesDiff()
    except Exception:
        logging.exception("Cannot build diff:")
        sys.exit(1)

    if (
            not diff.new_routes and not diff.deleted_routes and not diff.changed_routes and
            not diff.new_vlans and not diff.deleted_vlans
    ):
        logging.info("No route differencies found, nothing to do")
        sys.exit(0)

    if args.orig_hash != diff.orig_hash and not args.force:
        logging.error("Outdated or incompatible config hash supplied")
        sys.exit(1)

    logging.info("Found discrepancies:")
    logging.info("New routes: %s\n" % json.dumps(diff.new_routes))
    logging.info("Deleted routes: %s\n" % json.dumps(diff.deleted_routes))
    logging.info("Changed routes: %s\n" % json.dumps(diff.changed_routes))
    logging.info("New vlans: %s\n" % json.dumps(diff.new_vlans))
    logging.info("Deleted vlans: %s\n" % json.dumps(diff.deleted_vlans))

    if diff.new_vlans or diff.deleted_vlans:
        logging.error("Vlans does not match, the %s should be restarted or reloaded" % diff.iface_name)
        sys.exit(1)

    if args.orly:
        logging.info('Checking if operation allowed...')
        ok, err = orlyutil.start_operation(ORLY_RULE, socket.gethostname())
        if not ok:
            logging.error('Failed to start operation %s: %s', ORLY_RULE, err)
            sys.exit(1)
    logging.info("Applying routes:")
    try:
        diff.apply()
    except Exception:
        logging.exception("Exception while applying routes:")
        sys.exit(1)


def run_apply_hacks(args):

    hacks = {deviceutil.MlxMt27710RxhashOff(): args.mlx_mt27710_rxhash_off}
    need_hacks = set()

    for hack, requested in hacks.items():
        if args.all or requested:
            result = hack.ensure()
            if not result:
                logging.info("%s: NOT NEEDED" % hack.name())
            else:
                need_hacks.add(hack)

    if len(need_hacks) == 0:
        logging.info('No applicable hacks, exiting')
        return

    if args.orly:
        logging.info('Checking if operation allowed...')
        ok, err = orlyutil.start_operation(ORLY_HACKS_RULE, socket.gethostname())
        if not ok:
            logging.error('Denied to start operation %s: %s', ORLY_HACKS_RULE, err)
            sys.exit(1)

    for hack in need_hacks:
        result = hack.apply()
        if result is not None:
            yasmutil.push_local([("netconfig_%s_thhh" % hack.name(), 1.0 if result else 0.0)])
            if result:
                logging.info("%s: OK" % hack.name())
            else:
                logging.error("%s: FAIL" % hack.name())


def run_pregen_ip(args):
    # Similar logic with master.get_projectid_address()
    project_id = args.project_id
    method = args.method
    mac = None
    ra_prefix = None

    if args.prefix:
        ra_prefix = args.prefix

    if args.iface:
        if not ra_prefix:
            ra_prefix = master.get_ra_prefix(args.iface)
        mac = master.get_mac_address(args.iface)

    if not ra_prefix:
        logging.info('RA prefix is not set, use --iface or --prefix options')
        return

    try:
        ra_prefix = netaddr.IPNetwork(ra_prefix)
    except Exception:
        logging.info('Can not parse RA prefix')
        return

    if ra_prefix.prefixlen < 48:
        raise master.NetconfigError('Got incompatible RA prefix, prefix_length < 48: {}'.format(ra_prefix.prefixlen))

    if method == 'hostname':
        if args.hostname:
            hostname = args.hostname
        else:
            hostname = master.get_my_hostname()

        host_id = master.get_hostname_hostid(hostname)

    elif method == 'mac':
        if args.mac:
            mac = args.mac

        if not mac:
            logging.info('MAC is not specified. Use --iface or --mac options.')
            return

        host_id = master.get_mac_hostid(mac)

    else:
        host_id = master.get_manual_hostid(method)

    ip_part = netaddr.IPAddress('::')

    project_id = int(project_id, 16)
    if project_id > 4294967295:
        logging.info('Project ID is more than 4 bytes')
        return

    ip_part.value = project_id << 32 | host_id
    ip_part.value = ra_prefix.ip.value | ip_part.value

    result = netaddr.IPNetwork(ip_part)
    result.prefixlen = ra_prefix.prefixlen
    logging.info(str(result))
    return result


def interfaces(show_all=False):
    names_re = re.compile(r'^(eth|vlan)[0-9]+$')

    with IPRoute() as ipr:
        links = ipr.get_links()

    if show_all:
        for l in links:
            yield l
    else:
        for l in links:
            if names_re.match(l.get_attr('IFLA_IFNAME')):
                yield l


def run_status(args):
    logging.disable(logging.CRITICAL)
    result = []
    proto_res = fixutil_pb2.TargetState()
    for iface in interfaces(args.show_all):
        try:
            interface_proto = status.make_iface_proto(iface) or None
            if interface_proto:
                if args.proto:
                    proto_res.interfaces.extend([interface_proto])
                else:
                    result.append(json.loads(MessageToJson(interface_proto)))
        except Exception as e:
            logging.debug('{}'.format(e))

    if args.proto:
        print(proto_res.SerializeToString())
    else:
        print(json.dumps(result, indent=4))
    logging.disable(logging.NOTSET)
    sys.exit()


def run_switch_port(args):
    res = {}
    ifaces = [iface for iface in os.listdir('/sys/class/net') if iface.startswith('eth')]
    for i in ifaces:
        if master.is_iface_connected(i) and fixutil.is_physical_interface(i):
            res[i] = lldputil.parse_lldp_packet(i)
    print(res)
    sys.exit()


def parse_args(argv):
    p = argparse.ArgumentParser(prog='yandex-netconfig-utils', add_help=True)
    p.add_argument('-l', '--log-level', default='INFO', choices=['INFO', 'ERROR', 'DEBUG'], required=False)
    sp = p.add_subparsers(title="Actions")

    verify = sp.add_parser(name="verify", description='Verify configured netconfig state')
    verify.set_defaults(handle=run_verify)

    hash_config = sp.add_parser(name="hash_config", description='Obtain current config hash')
    hash_config.set_defaults(handle=run_hash_config)
    hash_config.add_argument('-i', "--iface", help='Use config from interface instead of default', required=False,
                             default=None, type=str)
    hash_config.add_argument('-s', "--scripting", help='Scripting mode, provide only hash value on success',
                             required=False, action="store_true")

    apply_routes = sp.add_parser(name="apply_routes", description='Apply route changes')
    apply_routes.set_defaults(handle=run_apply_routes)
    apply_routes.add_argument("orig_hash",
                              help='Safety measure: do not apply routes if on-host config does not match this one',
                              type=str, nargs='?', default="")
    apply_routes.add_argument('--force', help='Force applying routes without hash matching', action="store_true")
    apply_routes.add_argument('--orly', help='Use ORLY service to allow operations', action="store_true")

    apply_hacks = sp.add_parser(name="apply_hacks", description='Apply specific hw/sw tweaks')
    apply_hacks.set_defaults(handle=run_apply_hacks)
    apply_hacks.add_argument('--orly', help='Use ORLY service to allow operations', action="store_true")
    apply_hacks.add_argument('--all', help='Apply all known hacks', action="store_true")
    apply_hacks.add_argument('--mlx_mt27710_rxhash_off', help='Mellanox mt27710 rxhash off for Search kernel 4.4', action="store_true")

    pregen_ip = sp.add_parser(name='pregen_ip', description='Pregenerate IP for interface')
    pregen_ip.set_defaults(handle=run_pregen_ip)

    pregen_ip.add_argument('--project-id', type=str, help='Project id string, required.', required=True)
    pregen_ip.add_argument('--method', type=str, help='Method to generate IP. It can be \'hostname\', \'mac\', or some string, from which we will count some hash. Required.', required=True)
    pregen_ip.add_argument('-i', '--iface', help='Interface name, like eth0. If used, we will get mac address and RA prefix from it.')
    pregen_ip.add_argument('--prefix', type=str, help='Set RA prefix.')
    pregen_ip.add_argument('--mac', type=str, help='Set mac address.')
    pregen_ip.add_argument('--hostname', type=str, help='Set hostname. Overwrites hostname we got from OS.')

    status = sp.add_parser(name="status", description='Show current network configuration')
    status.set_defaults(handle=run_status)

    status.add_argument('--show-all', action='store_true', help='Include dummy and tunnels.')
    status.add_argument('--proto', action='store_true', help='Use proto format.')

    sw_port = sp.add_parser(name="switch-port", description='Get current switch port via lldp')
    # sw_port.add_argument('-i', '--iface',  help='Interface name, like eth0')
    sw_port.set_defaults(handle=run_switch_port)

    return p.parse_args(argv)


def setup_logging(level):
    logging.basicConfig(stream=sys.stdout, level=logging.getLevelName(level))


def main():
    args = parse_args(sys.argv[1:])
    setup_logging(args.log_level)
    args.handle(args)


if __name__ == "__main__":
    main()
