"""
Former infra/rtc/juggler/bundle/external/check_ra.py check
The diffs are:
 - run with netconfig
 - check mtn prefixes before check RA
"""

import os
import json
import ipaddr
import subprocess

from infra.netconfig.lib import master
from infra.netconfig.lib import jugglerutil

SERVICE_NAME = "check_ra"
exclude_vlans = ['vlan1478', 'vlan711', 'vlan1496']
state_file = '/var/tmp/' + SERVICE_NAME + '.state'


def list_ifaces():
    eth_li = []
    all_eth = os.listdir('/sys/class/net/')
    for eth in all_eth:
        if eth in exclude_vlans:
            continue
        if eth.startswith('L3'):
            continue
        if eth.startswith('tap'):
            continue
        if eth.startswith('veth'):
            continue
        if eth.startswith('ib'):
            continue
        with open('/sys/class/net/' + eth + '/operstate') as f:
            status = f.read()
            status = status.strip()
            if status == 'up':
                eth_li.append(eth)
    return eth_li


def get_ra_counters(ifaces):
    eth_status = {}
    for iface in ifaces:
        eth_status[iface] = 0
        with open('/proc/net/dev_snmp6/' + iface) as f:
            lines = f.readlines()
        for line in lines:
            if not line.startswith('Icmp6InRouterAdvertisements'):
                continue
            line = line.strip()
            name, count = line.split()
            eth_status[iface] = int(count)
            break
    return eth_status


def save_counters(counters):
    try:
        with open(state_file, 'w') as f:
            f.write(json.dumps(counters))
    except:
        pass


def read_counters():
    try:
        with open(state_file) as f:
            content = f.read()
            return json.loads(content)
    except Exception:
        return None
    return None


def diff_counters(current, prev):
    diff = {}
    for name, counter in current.items():
        if name in prev:
            diff[name] = current[name] - prev[name]
        else:
            diff[name] = current[name]
    return diff


def accept_ra(ifaces):
    # /proc/sys/net/ipv6/conf/eth0/accept_ra
    result = {}
    for iface in ifaces:
        try:
            prefix = master.get_ra_prefix(iface)
            ipaddr_prefix = ipaddr.IPNetwork(prefix)
            if (ipaddr_prefix not in master.IPADDR_MTN_BACKBONE_PREFIX and
                    ipaddr_prefix not in master.IPADDR_MTN_FASTBONE_PREFIX):
                with open('/proc/sys/net/ipv6/conf/' + iface + '/accept_ra') as f:
                    result[iface] = int(f.read())
        except Exception:
            pass
    return result


def get_addrfam(name):
    cmd = 'ip -4 addr show dev {}'.format(name)
    output = subprocess.check_output(cmd.split())
    if output == '':
        return 'inet6'
    else:
        return 'inet'


def run():
    eth_li = list_ifaces()
    accept_ra_variables = accept_ra(eth_li)
    current_ra_counters = get_ra_counters(eth_li)
    prev_ra_counters = read_counters()
    save_counters(current_ra_counters)
    if prev_ra_counters is None:
        jugglerutil.push_local(
            'OK',
            '',
            SERVICE_NAME,
            ['OK'],
        )
        return
    else:
        diff = diff_counters(current_ra_counters, prev_ra_counters)
        for name, counter in diff.items():
            if counter == 0:
                jugglerutil.push_local(
                    'CRIT',
                    "have problem with RA on {} ({})".format(name, diff),
                    SERVICE_NAME,
                    ['CRIT'],
                )
                return
            if name in accept_ra_variables and name not in ['vlan688', 'vlan788'] and accept_ra_variables[name] != 2:
                addr_family = get_addrfam(name)
                if addr_family == 'inet':
                    continue
                jugglerutil.push_local(
                    'CRIT',
                    "have problem with ra_accept on {}(set to {})".format(name, accept_ra_variables[name]),
                    SERVICE_NAME,
                    ['CRIT'],
                )
                return

    jugglerutil.push_local('OK', '', SERVICE_NAME, ['OK'],)
