import datetime
import sys
import traceback

from infra.netconfig.lib import jugglerutil
from infra.netconfig.lib import master
from infra.netconfig.lib import hostutil

SERVICE_NAME = "netconfig_execution_state_check"


def check_stage(stage, name, iface, service_name):
    tstr = stage.get("timestamp", None)
    if not tstr:
        jugglerutil.push_local(
            'CRIT',
            "%s timestamp missing for iface %s" % (name, iface),
            service_name,
            ['stage_timestamp'],
        )
        return False

    ts = datetime.datetime.strptime(tstr.split(".")[0], "%Y-%m-%d %H:%M:%S")
    if (datetime.datetime.now() - ts).total_seconds() > hostutil.get_uptime():
        jugglerutil.push_local(
            'CRIT',
            "%s state is outdated for iface %s" % (name, iface),
            service_name,
            ['stage_outdated_state'],
        )
        return False

    status = stage.get("status", None)
    if status == "ERROR":
        msg = stage.get("exception", "")
        tb = stage.get("traceback", "")
        jugglerutil.push_local(
            'CRIT',
            "%s failed for iface %s:\n%s\n%s" % (name, iface, msg, tb),
            service_name,
            ['stage_status_error'],
        )
        return False

    elif status == "INPROGRESS":
        if (datetime.datetime.now() - ts).total_seconds() > 900:
            jugglerutil.push_local(
                'CRIT',
                "%s hangup (lasted over 15 minutes) for iface %s" % (name, iface),
                service_name,
                ['stage_status_hangup'],
            )
            return False

    elif status is None:
        jugglerutil.push_local(
            'CRIT',
            "%s has no status info for iface %s" % (name, iface),
            service_name,
            ['stage_no_status'],
        )

    return True


def get_section(service_name, path=None):
    kwargs = {}
    if path:
        kwargs["path"] = path

    netconfig = master.load_netconfig_state(**kwargs)
    if not netconfig:
        jugglerutil.push_local(
            'CRIT',
            "No state file",
            service_name,
            ['no_state_file'],
        )
        return False

    for section in netconfig:
        if "setup_stages" not in section:
            continue

        return section

    jugglerutil.push_local(
        'CRIT',
        "Empty netconfig state",
        service_name,
        ['empty_state'],
    )
    return False


def check_bootstrap(path=None):
    service_name = 'netconfig_bootstrap'
    section = get_section(service_name, path)
    if not section:
        return False

    iface = section["interface"]
    stages = section["setup_stages"]
    bootstrap = stages.get("bootstrap_net", None)
    if not bootstrap:
        jugglerutil.push_local(
            # Push OK for now (RTCNETWORK-16)
            'OK',
            "Missing bootstrap_net section",
            service_name,
            # ['bootstrap_net_missing'],
            ['OK'],
        )
        return False

    if not check_stage(bootstrap, "bootstrap_net", iface, service_name):
        return False

    jugglerutil.push_local(
        'OK',
        "",
        service_name,
        ['OK'],
    )

    return True


def check_vlans(path=None):
    service_name = 'netconfig_vlans_routes'
    section = get_section(service_name, path)
    if not section:
        return False

    iface = section["interface"]
    stages = section["setup_stages"]
    vlans = stages.get("vlans_routes", None)
    if not vlans:
        jugglerutil.push_local(
            # Push OK for now (RTCNETWORK-16)
            'OK',
            "Missing vlans_routes section",
            service_name,
            # ['vlans_routes_missing'],
            ['OK'],
        )
        return False

    if not check_stage(vlans, "vlans_routes", iface, service_name):
        return False

    jugglerutil.push_local(
        'OK',
        "",
        service_name,
        ['OK'],
    )
    return True


def run(path=None):
    try:
        bootstrap = check_bootstrap(path)
        vlans = check_vlans(path)
        jugglerutil.push_local(
            'OK',
            "",
            SERVICE_NAME,
            ['OK'],
        )
        return bootstrap and vlans
    except Exception:
        jugglerutil.push_local(
            'CRIT',
            "execution_state check failed: %s" % "\n".join(traceback.format_exception(*sys.exc_info())),
            SERVICE_NAME,
            ['check_exception'],
        )
