import json
import logging

from infra.netconfig.lib import stateutil
from infra.netconfig.lib import jugglerutil
from infra.netconfig.lib import yasmutil
from infra.netconfig.lib import master

SERVICE_NAME = "netconfig_host_routes_changes"


# NOC export example:
#
# {
#     "cidr": "2a02:6b8:b000:657:922b:34ff:fec1:d06a",
#     "info": {
#         "backbone_routes": "bb_default",
#         "backbone_vlans": {
#             "688": {
#                 "host64": 1,
#                 "mtn": 1,
#                 "routes": "bb_default"
#             }
#         },
#         "datacenter_id": 697,
#         "datacenter_name": "SAS",
#         "fastbone_vlans": {
#             "761": {
#                 "routes": "fb_Hbf"
#             },
#             "788": {
#                 "host64": 1,
#                 "mtn": 1,
#                 "routes": "fb_Hbf"
#             }
#         },
#         "l2_domain_id": 64,
#         "l2_domain_name": "sas1-2d",
#         "vlan": 604
#     },
#     "ip": "2a02:6b8:b000:657:922b:34ff:fec1:d06a",
#     "project_id": "604",
#     "project_id_host_method": "mac",
#     "routes": {
#         "bb_default": {
#             "2620:10f:d000::/44": {
#                 "gateway": "fe80::1",
#                 "mtu": 8910
#             },
#             "2a02:6b8::/32": {
#                 "gateway": "fe80::1",
#                 "mtu": 8910
#             }
#         },
#         "bb_default4": {
#             "0.0.0.0/0": {
#                 "gateway": "last",
#                 "mtu": 1450
#             },
#             "100.43.64.0/19": {
#                 "gateway": "last",
#                 "mtu": 8910
#             },
#          ....
#            }
#     },
#     "switch_port": "GE1/0/10"
# }
#
# Netconfig state example:
#
# {
#     "backbone": {
#         "688": {
#             "bb_iface": "vlan688",
#             "created": true,
#             "host64": true,
#             "mtn_global": "2a02:6b8:c08:5e09:0:604::1/64",
#             "mtn_local": "fe80::a:9/64",
#             "mtn_net": "2a02:6b8:c08:5e09::/64",
#             "mtn_prefix": "2a02:6b8:c08:5e00::/57",
#             "routes": [
#                 {
#                     "dev": "vlan688",
#                     "gw": "fe80::1",
#                     "mtu": 8910,
#                     "route": "2620:10f:d000::/44",
#                     "table": "688"
#                 },
#                 {
#                     "dev": "vlan688",
#                     "gw": "fe80::1",
#                     "mtu": 8910,
#                     "route": "2a02:6b8::/32",
#                     "table": "688"
#                 },
#                 {
#                     "dev": "vlan688",
#                     "gw": "fe80::1",
#                     "mtu": 1450,
#                     "route": "::/0",
#                     "table": "688"
#                 }
#             ]
#         }
#     },
#     "bb_iface": "eth0",
#     "bb_ip": "2a02:6b8:b000:657:922b:34ff:fec1:d06a",
#     "bb_mtu": 9000,
#     "bb_routes": [],
#     "fastbone": {
#         "761": {
#             "created": true,
#             "fb_iface": "vlan761",
#             "fb_ip": "2a02:6b8:f000:a73:922b:34ff:fec1:d06a/64",
#             "method": "eui-64",
#             "routes": [
#                 {
#                     "dev": "vlan761",
#                     "gw": "fe80::1",
#                     "mtu": 8910,
#                     "route": "2a02:6b8:f030::/44"
#                 },
#                 {
#                     "dev": "vlan761",
#                     "gw": "fe80::1",
#                     "mtu": 8910,
#                     "route": "2a02:6b8:f010:1f02::/64"
#                 },
#                  ....
#             ]
#         }
#         ....
#     }
# }


def run(netconfig_path=None, interfaces_path=None):
    service_name = '_'.join((SERVICE_NAME, 'diff'))
    try:
        diff = stateutil.RoutesDiff(netconfig_path, interfaces_path)
        jugglerutil.push_local(
            'OK',
            '',
            service_name,
            ['OK'],
        )
    except master.NetconfigError as e:
        jugglerutil.push_local(
            'CRIT',
            e.message,
            service_name,
            ['check_netconfig_exception'],
        )
        return

    logging.info("New routes: %s\n" % json.dumps(diff.new_routes))
    logging.info("Deleted routes: %s\n" % json.dumps(diff.deleted_routes))
    logging.info("Changed routes: %s\n" % json.dumps(diff.changed_routes))
    logging.info("New vlans: %s\n" % json.dumps(diff.new_vlans))
    logging.info("Deleted vlans: %s\n" % json.dumps(diff.deleted_vlans))

    yasmutil.push_local([
        ("netconfig_have_new_routes_thhh", 1.0 if len(diff.new_routes) else 0.0),
        ("netconfig_new_routes_thhh", float(sum(map(lambda x: len(x), diff.new_routes.values())))),

        ("netconfig_have_deleted_routes_thhh", 1.0 if len(diff.deleted_routes) else 0.0),
        ("netconfig_deleted_routes_thhh", float(sum(map(lambda x: len(x), diff.deleted_routes.values())))),

        ("netconfig_have_changed_routes_thhh", 1.0 if len(diff.changed_routes) else 0.0),
        ("netconfig_changed_routes_thhh", float(sum(map(lambda x: len(x), diff.changed_routes.values())))),

        ("netconfig_have_new_vlans_thhh", 1.0 if len(diff.new_vlans) else 0.0),
        ("netconfig_new_vlans_thhh", float(len(diff.new_vlans))),

        ("netconfig_have_deleted_vlans_thhh", 1.0 if len(diff.deleted_vlans) else 0.0),
        ("netconfig_deleted_vlans_thhh", float(len(diff.deleted_vlans))),
    ])

    # Deleted routes check
    service_name = '_'.join((SERVICE_NAME, 'deleted'))
    if len(diff.deleted_routes) or len(diff.deleted_vlans) or len(diff.changed_routes):
        jugglerutil.push_local(
            'CRIT',
            "Outdated routes found",
            service_name,
            ['deleted_routes', 'changed_routes'],
        )
    else:
        jugglerutil.push_local(
            'OK',
            '',
            service_name,
            ['OK'],
        )

    # New routes check
    service_name = '_'.join((SERVICE_NAME, 'new'))
    if len(diff.new_routes) or len(diff.new_vlans):
        jugglerutil.push_local(
            'WARN',
            "New routes found",
            service_name,
            ['new_routes', 'changed_routes'],
        )
    else:
        jugglerutil.push_local(
            'OK',
            '',
            service_name,
            ['OK'],
        )
