import traceback
import glob
import sys
import os.path
import logging

from infra.netconfig.lib import jugglerutil
from infra.netconfig.lib import lldputil
from infra.netconfig.lib import master

SERVICE_NAME = "netconfig_switch_port_changes"

logger = logging.getLogger('switch_check_runs')


def run(path=master.RUN_NETWORK_DIR, host64_ifnames_path=master.HOST64_IFNAMES_PATH):
    try:
        changed_interfaces = []
        switch_port_file_list = glob.glob(os.path.join(path, '*.switch_port'))
        if not switch_port_file_list:
            jugglerutil.push_local(
                # OK for now (RTCNETWORK-16)
                'OK',
                'No switch_port files has been found',
                SERVICE_NAME,
                # ['switch_port_files_not_found'],
                ['OK'],
            )
            return True

        for switch_port_file in switch_port_file_list:
            interface = os.path.basename(switch_port_file).split('.')[0]
            with open(switch_port_file, 'r') as f:
                switch_port = f.readline().strip()

            parsed_lldp = lldputil.parse_lldp_packet(interface)
            port_name = parsed_lldp.get('port_name')
            if not port_name:
                raise master.NetconfigError('Could not get "port_name" from lldp packet parsing')
            logger.info('port_name: {}'.format(port_name))

            switch_name = parsed_lldp.get('switch_name')
            if not switch_name:
                raise master.NetconfigError('Could not get "switch_name" from lldp packet parsing')

            # get a short name from fqdn like sas1-s313.ydnx.net
            switch_name = switch_name.split('.')[0]
            logger.info('switch_name: {}'.format(switch_name))

            port_index = master.get_port_index(port_name, host64_ifnames_path)
            logger.info('port_index: {}'.format(port_index))

            if port_index != switch_port:
                changed_interfaces.append(interface)

        if changed_interfaces:
            jugglerutil.push_local(
                'CRIT',
                'Switch/Port has been changed since last interface configuration: {}'.format(changed_interfaces),
                SERVICE_NAME,
                ['switch_port_changed'],
            )
            return False

        jugglerutil.push_local(
            'OK',
            '',
            SERVICE_NAME,
            ['OK'],
        )
        return True

    except Exception:
        jugglerutil.push_local(
            'CRIT',
            'Got exception: {}'.format(
                '\n'.join(traceback.format_exception(*sys.exc_info()))
            ),
            SERVICE_NAME,
            ['check_exception'],
        )
        return False
