import netaddr
import ipaddr
import logging
import pprint
import re
import os
import socket

from infra.netconfig.lib import master
from infra.netconfig.lib import yasmutil
from infra.netconfig.lib import aggregate_routes
from infra.netconfig.proto import fixutil_pb2
from pyroute2 import IPRoute

DEF_ROUTE_MTU = 1450
NOT_MTN_RULES_MSG = 'Found rules with prefixes not from MTN. Won\'t fix!'
UNKNOWN_ROUTES_MSG = 'Unknown routes found'

RULE_RE_COMPILED = re.compile(r'^(?P<prio>[0-9]+)\:\s+from\s+(?P<src>[a-f0-9:/]+)\s+(to\s+(?P<dst>[a-f0-9:/]+)\s+)?lookup\s+(?P<table>(main|[0-9]+))$')


class FixallPrefix6(Exception):
    pass


def is_physical_interface(iface_name):
    return os.path.exists(os.path.join('/sys/class/net', iface_name, 'device/vendor'))


def get_interface_kind(iface_name):
    logging.info("Getting {} kind".format(iface_name))
    if is_physical_interface(iface_name):
        return 'physical'

    link_kind = ''
    with IPRoute() as ipr:
        try:
            indx = ipr.link_lookup(ifname=iface_name)
            link_kind = ipr.get_links(indx)[0].get_attr('IFLA_LINKINFO').get_attr('IFLA_INFO_KIND')
        except IndexError:
            msg = 'Could not find interface {}'.format(iface_name)
            logging.warning(msg)
        except Exception as e:
            logging.warning(e)

    logging.info("{} kind: {}".format(iface_name, link_kind))
    return link_kind


def get_vlan_id(iface_name):
    logging.info("Getting {} vlan_id".format(iface_name))
    vlan_id = None
    with IPRoute() as ipr:
        try:
            indx = ipr.link_lookup(ifname=iface_name)
            vlan_id = ipr.get_links(indx)[0].get_attr('IFLA_LINKINFO').get_attr('IFLA_INFO_DATA').get_attr('IFLA_VLAN_ID')
        except IndexError:
            msg = 'Could not find interface {}'.format(iface_name)
            logging.warning(msg)
        except Exception as (e):
            logging.warning(e)

    logging.info("{} vlan_id: {}".format(iface_name, vlan_id))
    return vlan_id


def prepare_ipv6_eui64_address(interface, prefix):
    """ Generate IPv6 EUI-64 address to emulate getting address from RA """
    mac = master.get_mac_address(interface)
    mac = netaddr.EUI(mac, dialect=netaddr.mac_unix)
    host_id = mac.modified_eui64()
    prefix = netaddr.IPNetwork(prefix)
    ipv6_eui64 = str(netaddr.IPAddress(prefix.first + int(host_id)))
    return str(ipaddr.IPAddress(ipv6_eui64))


def prepare_ipv6_backbone_global(iface, prefix):
    """ Return ipv6 address that should be on interface """
    if ipaddr.IPNetwork(prefix) in ipaddr.IPNetwork(master.MTN_BACKBONE_PREFIX):
        cidr = str(ipaddr.IPNetwork(master.get_projectid_backbone_address(iface, prefix)))
        backbone_ipv6 = str(ipaddr.IPNetwork(cidr).ip)
    else:
        backbone_ipv6 = prepare_ipv6_eui64_address(iface.name, prefix)
        cidr = str(ipaddr.IPNetwork('/'.join((backbone_ipv6, prefix.split('/')[1]))))
    return backbone_ipv6, cidr


def extend_bb_with_routes(iface_name, iface_proto, data, routes_groups):
    """ In:
          - interface name
          - interface protobuf
          - data - the data from l3 segments for specified network
          - routes_groups - route groups from l3-segments
        Out:
          - updated version of interface protobuf, extended with routes
    """

    bb_routes = routes_groups.get(data.get('backbone_routes', ''), {}).items()
    bb_routes = master.convert_routes(bb_routes, iface_proto.cidr, iface_name)
    # Extend backbone routes and set bb_ip_static if 'prefix6' in network data
    if 'prefix6' in data:
        raise FixallPrefix6('prefix6 in network data')

    # Set backbone interface routes in protobuf here
    for r in bb_routes:
        iface_proto.routes.extend([route_to_proto(r)])

    return iface_proto


def compare_l3_segment_prefixes_w_aggregates(l3_segments_routes, aggregate_routes):
    unknown_routes = []
    l3_segments_prefixes_set = {ipaddr.IPNetwork(route['route']) for route in l3_segments_routes}
    aggregate_prefixes_set = {ipaddr.IPNetwork(route['route']) for route in aggregate_routes}
    diff = l3_segments_prefixes_set - aggregate_prefixes_set
    if diff:
        for l3_route in l3_segments_routes:
            found_in_aggr = False
            for aggr_prefix in aggregate_prefixes_set:
                if ipaddr.IPNetwork(l3_route['route']) in aggr_prefix:
                    found_in_aggr = True
                    break

            if not found_in_aggr:
                unknown_routes.append(l3_route)
    return unknown_routes


def target_states_from_l3segments(iface, interface_proto):
    """ Update interface_proto with backbone interface routes
        and return TargetStates() with physical interface
        and vlans based on info from l3-segments.

        Probably, should be splited in functions.
    """

    # Set some variables that can be usefull in future
    # method = None
    project_id = None
    project_id_host_method = None
    # bb_ip_static = None
    switch_port = None

    # Find network in l3segments (copypaste from master)
    networks_info = master.get_network_info()
    searchip = ipaddr.IPNetwork(interface_proto.cidr).ip
    foundnet = None
    for network in networks_info['data']:
        net = ipaddr.IPNetwork(network)
        if searchip not in net:
            continue
        if foundnet is not None and net.compare_networks(foundnet) < 0:
            continue
        foundnet = net
        logging.info('FOUND: {}'.format(network))
        data = networks_info['data'][network]

    # data = networks_info['data'][network]  # data of found network
    # logging.info('DATA:\n{}'.format(pprint.pformat(data)))
    routes_groups = networks_info['route_groups']  # all groups of routes

    # set 'method' for future probably
    # if data.get('mtn') and iface.option('method') == 'auto':
    #     method = 'project_id'

    # set project_id and project_id_host_method for future
    if master.dict_has_value(data, 'mtn', 1):
        project_id = master.get_project_id(iface)
        project_id_host_method = iface.option('ya-netconfig-project-id-host-method') or 'hostname'

    # rewrite host64 value if needed
    host64_override_value = iface.numeric_option('ya-netconfig-host64-override')
    if host64_override_value is not None:
        master.dict_override_value(data, 'host64', host64_override_value)

    # if host64 somewhere in network data, get project_id and switch_port
    if master.dict_has_value(data, 'host64', 1):
        project_id = master.get_project_id(iface)

        switch_port = master.get_lldp_port_index(iface.name)

    # Get routes for backbone interface
    interface_proto = extend_bb_with_routes(iface.name, interface_proto, data, routes_groups)

    # trying to get backbone vlans and create protobuf InterfaceState for it
    vlans_target_state = fixutil_pb2.TargetState()

    bbvlans = data.get('backbone_vlans')
    if bbvlans and not iface.bool_option('ya-netconfig-bb-disable'):
        for vlan, opts in bbvlans.items():
            # create net proto.InterfaceState for bb_vlan interface
            vlan_iface_proto = fixutil_pb2.InterfaceState()
            vlan_iface_proto.name = ''.join(('vlan', vlan))  # 'vlan688' for example
            vlan_iface_proto.mtu = interface_proto.mtu  # use mtu of physical iface
            vlan_iface_proto.group = 'bb'

            # We need to create vlan interfaces to get ips and other info for them
            try:
                master.make_subinterface(interface_proto.name, vlan, vlan_iface_proto.group, iface)
                vlan_iface_proto.kind = get_interface_kind(vlan_iface_proto.name)
            except master.NetconfigMTUError as e:
                # RTCNETWORK-70
                logging.info('{}'.format(e))
                logging.info('Trying to create subinterface {} with MTU 1500'.format(interface_proto.name))
                master.make_subinterface(interface_proto.name, vlan, vlan_iface_proto.group, iface, rewrite_mtu=1500)

            mtn_ips = master.get_host64_address(
                vlan_iface_proto.name,
                switch_port
            )

            vlan_iface_proto.ipv6_address = str(ipaddr.IPNetwork(mtn_ips['mtn_global']).ip)
            vlan_iface_proto.ipv6_link_local = str(ipaddr.IPNetwork(mtn_ips['mtn_local']))
            vlan_iface_proto.prefix = str(ipaddr.IPNetwork(mtn_ips['mtn_prefix']))

            vlan_iface_proto.mtn_host64 = ipaddr.IPNetwork(vlan_iface_proto.prefix) in ipaddr.IPNetwork(master.MTN_BACKBONE_PREFIX)

            vlan_iface_proto.network = str(ipaddr.IPNetwork(mtn_ips['mtn_net']))
            formed_cidr = '/'.join((vlan_iface_proto.ipv6_address, vlan_iface_proto.network.split('/')[1]))
            vlan_iface_proto.cidr = str(ipaddr.IPNetwork(formed_cidr))

            l3_segments_routes = routes_groups.get('routes', {}).items()
            l3_segments_routes = master.convert_routes(
                l3_segments_routes,
                vlan_iface_proto.prefix,
                vlan_iface_proto.name,  # vlan688
                vlan  # 688 for example - used to fill 'table' field
            )

            vlan_routes = aggregate_routes.BACKBONE_AGGR_ROUTES.items()
            vlan_routes = master.convert_routes(
                vlan_routes,
                vlan_iface_proto.prefix,
                vlan_iface_proto.name,  # vlan688
                vlan
            )

            unknown_routes = compare_l3_segment_prefixes_w_aggregates(l3_segments_routes, vlan_routes)
            for ur in unknown_routes:
                vlan_iface_proto.unknown_routes.extend([route_to_proto(ur)])

            # For backbone mtn add default gw
            vlan_routes.append({
                'dev': vlan_iface_proto.name,
                'gw': 'fe80::1',
                'mtu': master.MTU_DEFAULT_ROUTE,
                'route': '::/0',
                'table': vlan
            })
            for vr in vlan_routes:
                vlan_iface_proto.routes.extend([route_to_proto(vr)])
            vlans_target_state.interfaces.extend([vlan_iface_proto])

    fbvlans = data.get('fastbone_vlans')
    if fbvlans and not iface.bool_option('ya-netconfig-fb-disable'):
        for vlan, opts in fbvlans.items():
            routes = opts.get('routes', '')
            # No routes - skip interface (logic from master)
            if not routes:
                continue
            vlan_routes = aggregate_routes.FASTBONE_AGGR_ROUTES.items()

            vlan_iface_proto = fixutil_pb2.InterfaceState()
            vlan_iface_proto.name = ''.join(('vlan', vlan))  # 'vlan688' for example
            vlan_iface_proto.mtu = interface_proto.mtu  # use mtu of physical iface
            vlan_iface_proto.group = 'fb'

            # We need to create vlan interfaces to get ips and other info for them
            try:
                master.make_subinterface(interface_proto.name, vlan, vlan_iface_proto.group, iface)
                vlan_iface_proto.kind = get_interface_kind(vlan_iface_proto.name)
            except master.NetconfigMTUError as e:
                # RTCNETWORK-70
                logging.info('{}'.format(e))
                logging.info('Trying to create subinterface {} with MTU 1500'.format(interface_proto.name))
                master.make_subinterface(interface_proto.name, vlan, vlan_iface_proto.group, iface, rewrite_mtu=1500)

            # mtn for host64
            if opts.get('host64', None):

                # TODO: migrate this code to function. It's the same as in backbones vlans.
                # The only diff is 'fb'/'bb' in master.make_subinterface(...)
                # and for 'bb' we add default gw route.
                mtn_ips = master.get_host64_address(
                    vlan_iface_proto.name,
                    switch_port
                )
                vlan_iface_proto.ipv6_address = str(ipaddr.IPNetwork(mtn_ips['mtn_global']).ip)
                vlan_iface_proto.ipv6_link_local = str(ipaddr.IPNetwork(mtn_ips['mtn_local']))
                vlan_iface_proto.prefix = str(ipaddr.IPNetwork(mtn_ips['mtn_prefix']))
                vlan_iface_proto.mtn_host64 = True
                vlan_iface_proto.network = str(ipaddr.IPNetwork(mtn_ips['mtn_net']))
                formed_cidr = '/'.join((vlan_iface_proto.ipv6_address, vlan_iface_proto.network.split('/')[1]))
                vlan_iface_proto.cidr = str(ipaddr.IPNetwork(formed_cidr))
                master.set_sysctl('net.ipv6.conf.all.forwarding', '1')

                l3_segments_routes = routes_groups.get(routes, {}).items()
                l3_segments_routes = master.convert_routes(
                    l3_segments_routes,
                    vlan_iface_proto.prefix,
                    vlan_iface_proto.name,  # vlan688
                    vlan  # 688 for example - used to fill 'table' field
                )

                vlan_routes = master.convert_routes(
                    vlan_routes,
                    vlan_iface_proto.prefix,
                    vlan_iface_proto.name,
                    vlan
                )

                unknown_routes = compare_l3_segment_prefixes_w_aggregates(l3_segments_routes, vlan_routes)
                for ur in unknown_routes:
                    vlan_iface_proto.unknown_routes.extend([route_to_proto(ur)])

                for vr in vlan_routes:
                    vlan_iface_proto.routes.extend([route_to_proto(vr)])
                vlans_target_state.interfaces.extend([vlan_iface_proto])
                continue

            if opts.get('mtn', None):
                project_ip = master.get_projectid_address(
                    vlan_iface_proto.name,
                    project_id,
                    project_id_host_method
                )

                logging.info("Fastbone Project IP: {}".format(project_ip))

                vlan_iface_proto.ipv6_address = str(ipaddr.IPNetwork(project_ip).ip)

                l3_segments_routes = routes_groups.get(routes, {}).items()
                l3_segments_routes = master.convert_routes(
                    l3_segments_routes,
                    str(ipaddr.IPNetwork(project_ip).network),  # prefix
                    vlan_iface_proto.name,  # vlan688
                    vlan  # 688 for example - used to fill 'table' field
                )

                vlan_routes = master.convert_routes(
                    vlan_routes,
                    str(ipaddr.IPNetwork(project_ip).network),  # prefix
                    vlan_iface_proto.name,
                )
                vlan_iface_proto.cidr = str(ipaddr.IPNetwork(project_ip))

                unknown_routes = compare_l3_segment_prefixes_w_aggregates(l3_segments_routes, vlan_routes)
                for ur in unknown_routes:
                    vlan_iface_proto.unknown_routes.extend([route_to_proto(ur)])

                for vr in vlan_routes:
                    vlan_iface_proto.routes.extend([route_to_proto(vr)])
                vlans_target_state.interfaces.extend([vlan_iface_proto])
                continue

            master.enable_ipv6_ra(vlan_iface_proto.name)
            # Don't check IP flag set, because of the RA on that iface
            logging.info('Set \'dont_check_ip\' flag for iface {} because of RA is enabled for this iface'.format(vlan_iface_proto.name))
            vlan_iface_proto.dont_check_ip = True
            vlan_iface_proto.cidr = str(ipaddr.IPNetwork(master.get_ipv6_ra(vlan_iface_proto.name, False, False)))
            vlan_iface_proto.ipv6_address = str(ipaddr.IPNetwork(vlan_iface_proto.cidr).ip)

            l3_segments_routes = routes_groups.get(routes, {}).items()
            l3_segments_routes = master.convert_routes(
                l3_segments_routes,
                str(ipaddr.IPNetwork(vlan_iface_proto.ipv6_address).network),  # prefix
                vlan_iface_proto.name,
            )

            vlan_routes = master.convert_routes(
                vlan_routes,
                str(ipaddr.IPNetwork(vlan_iface_proto.ipv6_address).network),  # prefix
                vlan_iface_proto.name,
            )

            unknown_routes = compare_l3_segment_prefixes_w_aggregates(l3_segments_routes, vlan_routes)
            for ur in unknown_routes:
                vlan_iface_proto.unknown_routes.extend([route_to_proto(ur)])

            for vr in vlan_routes:
                vlan_iface_proto.routes.extend([route_to_proto(vr)])
            vlans_target_state.interfaces.extend([vlan_iface_proto])

    interfaces_target_states = fixutil_pb2.TargetState()
    interfaces_target_states.interfaces.extend([interface_proto])
    for viface in vlans_target_state.interfaces:
        interfaces_target_states.interfaces.extend([viface])

    return interfaces_target_states


def prepare_base_phys_interface_proto(iface):
    master.check_is_configurable(iface, reload_mode=True)
    interface_proto = fixutil_pb2.InterfaceState()
    interface_proto.name = iface.name
    interface_proto.prefix = master.get_ra_prefix(iface.name)
    bb_ipv6, cidr = prepare_ipv6_backbone_global(iface, interface_proto.prefix)
    interface_proto.ipv6_address = str(ipaddr.IPAddress(bb_ipv6))
    interface_proto.cidr = str(ipaddr.IPNetwork(cidr))
    interface_proto.mtu = iface.numeric_option('mtu') or master.MTU_IFACE
    interface_proto.group = 'backbone'
    interface_proto.kind = get_interface_kind(iface.name)

    for k, v in iface.options.items():
        interface_proto.options[k] = str(v)

    # We should set mtu of phys ifaces in the begining of iteration,
    # to set up proper mtu on vlan ifaces.
    master.set_mtu(interface_proto.name, interface_proto.mtu)

    return interface_proto


def fixall(target_states, dry_run=True):

    results = []  # needs for juggler message
    juggler_mode = None
    routes_metrics = {
        'new': 0,
        'old': 0,
        'changed': 0,
    }

    for iface_ts in target_states.interfaces:
        # Force fix link mtu, and set group of interface
        mtu_res = fix_link_and_mtu(iface_ts, False)  # dry_run=False
        if mtu_res:
            results.append(':'.join((iface_ts.name, 'link_mtu')))
            juggler_mode = 'CRIT'

        # Fix IP
        ip_res = fix_ip(iface_ts, dry_run)
        if ip_res:
            results.append(':'.join((iface_ts.name, 'ip')))
            juggler_mode = 'CRIT'

        # count routes diffs for metrics
        diff = routes_diffs(iface_ts, return_len=True)
        routes_metrics['new'] += diff[0]
        routes_metrics['old'] += diff[1]
        routes_metrics['changed'] += diff[2]
        # Fix routes
        routes_res = fix_routes(iface_ts, dry_run)
        if routes_res:
            results.append(':'.join((iface_ts.name, 'routes')))
            juggler_mode = 'CRIT'

        # Check if some strange routes not from aggregate prefixes exists
        logging.info('Checking for unknown routes')
        if iface_ts.unknown_routes:
            logging.info('{}: {}'.format(UNKNOWN_ROUTES_MSG, iface_ts.unknown_routes))
            results.append(':'.join((iface_ts.name, UNKNOWN_ROUTES_MSG)))
            if not juggler_mode:
                juggler_mode = 'WARN'

    rules_res = fix_rules(target_states.interfaces, dry_run)
    if rules_res:
        results.append(rules_res)
        if not juggler_mode:
            juggler_mode = 'WARN'

    logging.info('ROUTES_METRICS: {}'.format(routes_metrics))
    yasmutil.push_local([
        ('netconfig_have_new_routes_thhh', 1.0 if routes_metrics['new'] - routes_metrics['changed'] else 0.0),
        ('netconfig_new_routes_thhh', float(routes_metrics['new'] - routes_metrics['changed'])),

        ('netconfig_have_deleted_routes_thhh', 1.0 if routes_metrics['old'] - routes_metrics['changed'] else 0.0),
        ('netconfig_deleted_routes_thhh', float(routes_metrics['old'] - routes_metrics['changed'])),

        ('netconfig_have_changed_routes_thhh', 1.0 if routes_metrics['changed'] else 0.0),
        ('netconfig_changed_routes_thhh', float(routes_metrics['changed'])),
    ])

    return results, juggler_mode


def fix_link_and_mtu(iface_ts, dry_run):
    # Force set groups to interface
    master.set_interface_group(iface_ts.name, iface_ts.group)

    # Force set base reach time / retrans time on missmatch with default
    base_reach_time = iface_ts.options.get('ya-netconfig-base-reach-time', master.DEF_BASE_REACH_TIME)
    master.set_base_neigh_reach_time(iface_ts.name, val=base_reach_time)

    retrans_time = iface_ts.options.get('ya-netconfig-retrans-time', master.DEF_RETRANS_TIME)
    master.set_retrans_time_ms(iface_ts.name, val=retrans_time)

    current_mtu = master.get_mtu(iface_ts.name)
    if iface_ts.mtu != current_mtu:
        if dry_run:
            # Return True to mark that we need fix
            return True
        else:
            # Fix here
            master.set_mtu(iface_ts.name, iface_ts.mtu)
            return True  # Return True to mark it in juggler.

    # No fixes needed
    return False


def get_current_addresses_on_iface(ifname, scope=None):
    current_addresses = set()
    addresses = []
    with IPRoute() as ipr:
        iface = ipr.link_lookup(ifname=ifname)[0]
        addresses = ipr.get_addr(index=iface, scope=scope) if scope is not None else ipr.get_addr(index=iface)

    for ip_addr in addresses:
        # We need to canonize current ip due to old pyroute2 in arcadia
        address = ip_addr.get_attr('IFA_ADDRESS')
        prefix_length = ip_addr.get('prefixlen')
        canonized_ip = str(ipaddr.IPAddress(address))
        logging.debug('IPRoute_ip_addr: {}/{}'.format(canonized_ip, prefix_length))
        canonized_tuple = (canonized_ip, prefix_length)
        logging.info('Adding to current_addresses: {}'.format(canonized_tuple))
        current_addresses.add(canonized_tuple)
    return current_addresses


def match_prefixes(ip1, ip2):
    return netaddr.IPNetwork(ip1).value >> 64 == netaddr.IPNetwork(ip2).value >> 64


def match_hostids(ip1, ip2):
    return netaddr.IPNetwork(ip1).value & 0xFFFFFFFFFFFFFFFF == netaddr.IPNetwork(ip2).value & 0xFFFFFFFFFFFFFFFF


def has_mtn_fb_bb_prefix_ip(set_of_addresses):
    # Format of set_of_addresses is {(addr, prefix), ...}

    for ipv6 in set_of_addresses:
        # find ip with perfix in MTN
        if ipaddr.IPAddress(ipv6[0]) in master.IPADDR_MTN_BACKBONE_PREFIX \
                or ipaddr.IPAddress(ipv6[0]) in master.IPADDR_MTN_FASTBONE_PREFIX:

            return ipv6


def fix_ip(interface_target_state, dry_run):
    if interface_target_state.dont_check_ip:
        logging.info('Don\'t check IP flag set for iface {}, so skip it'.format(interface_target_state.name))
        return False

    link_local_eui64 = prepare_ipv6_eui64_address(interface_target_state.name, 'fe80::/64')
    logging.info('LINK_LOCAL_EUI64: {}'.format(link_local_eui64))

    current_addresses = get_current_addresses_on_iface(interface_target_state.name)
    logging.info('Current addresses of {}: {}, type: {}'.format(interface_target_state.name, current_addresses, type(current_addresses)))

    # Remove link local eui64 address from current adresses on iface
    try:
        current_addresses.remove((link_local_eui64, 64))
        logging.info('Remove link local eui64 from set {}: {}'.format(interface_target_state.name, current_addresses))
    except KeyError:
        # what to do here ?
        logging.info('There is no predicted link local eui64 in current addresses')

    logging.info('TARGET_IP: {}'.format(interface_target_state.ipv6_address))
    logging.info('TARGET_CIDR: {}'.format(interface_target_state.cidr))
    logging.info('TARGET_LINK_LOCAL: {}'.format(interface_target_state.ipv6_link_local))
    # Check if link local IPv6 is on interface if interface_target_state has ipv6_link_local
    if interface_target_state.ipv6_link_local:
        addr, prefix = interface_target_state.ipv6_link_local.split('/')
        addr = str(ipaddr.IPNetwork(addr).ip)  # canonize
        link_local_ipv6 = (addr, int(prefix))
        logging.info('target link_local_ipv6: {}'.format(link_local_ipv6))
        if link_local_ipv6 in current_addresses:
            logging.info('Remove {} from set: {}'.format(link_local_ipv6, current_addresses))
            current_addresses.remove(link_local_ipv6)
            logging.info('Current addresses: {}'.format(current_addresses))
        else:
            if dry_run:
                # mark that we need fix
                logging.info('There is no {} on {}. Need to add.'.format(link_local_ipv6, interface_target_state.name))
                return True
            else:
                # Fix here
                logging.info('Setting {} on {}'.format(link_local_ipv6, interface_target_state.name))
                master.set_ipv6_address(interface_target_state.name, interface_target_state.ipv6_link_local)

    # Check if global IPv6 is on interface
    addr, prefix = interface_target_state.cidr.split('/')
    addr = str(ipaddr.IPAddress(addr))
    global_ipv6 = (addr, int(prefix))
    logging.info('target global_ipv6: {}'.format(global_ipv6))

    # Check if we have MTN prefix on physical iface but hostid part of IP is SLAAC generated
    if is_physical_interface(interface_target_state.name):
        logging.info("Check if we have MTN prefix on physical iface but hostid part of IP is SLAAC generated.")
        found_ip = has_mtn_fb_bb_prefix_ip(current_addresses)
        if not found_ip:
            logging.info("Not found any IP on {} with MTN prefix".format(interface_target_state.name))
            # If no MTN prefixes on physical iface, it means that we are not in project-id network.
            # And that means, that we have to get addresses/routes by RA.
            # So, force set net.ipv6.conf.<iface>.accept_ra=2
            # https://wiki.yandex-team.ru/users/olegsenin/SPI-8127/
            master.set_sysctl('net.ipv6.conf.{}.accept_ra'.format(interface_target_state.name), 2)
        else:
            logging.info("Found IP with MTN prefix on physical iface: {}".format(found_ip))
            # In project-id network we set addresses and routes by ourself.
            # So, force set net.ipv6.conf.<iface>.accept_ra=0
            # https://wiki.yandex-team.ru/users/olegsenin/SPI-8127/
            master.set_sysctl('net.ipv6.conf.{}.accept_ra'.format(interface_target_state.name), 0)

            if not match_prefixes(found_ip[0], interface_target_state.cidr) or \
               not match_hostids(found_ip[0], interface_target_state.cidr):

                logging.info(
                    "Looks like current ip {} is SLAAC and missmatches with target ip {}".format(
                        found_ip[0],
                        interface_target_state.cidr
                    )
                )
                logging.info("Force replace {} to {}".format(found_ip[0], interface_target_state.cidr))
                # We can't set default route with new address as prefsrc, if new address in tentative state.
                # So, disable DAD.
                master.set_sysctl('net.ipv6.conf.{}.accept_dad'.format(interface_target_state.name), '0')
                master.set_ipv6_address(interface_target_state.name, interface_target_state.cidr)
                # Because of disabling accept_ra, we should write default route by ourself.
                logging.info("Working on default route for {}".format(interface_target_state.name))
                logging.info("Pref src IP: {}".format(interface_target_state.ipv6_address))
                try:
                    with IPRoute() as ipr:
                        link_id = ipr.link_lookup(ifname=interface_target_state.name)[0]
                        logging.info("LINK_ID: {}".format(link_id))
                        ipr.route(
                            'replace',
                            family=socket.AF_INET6,
                            oif=link_id,
                            dst='::/0',
                            table=254,
                            gateway='fe80::1',
                            priority=1,
                            prefsrc=interface_target_state.ipv6_address,
                            metrics={"mtu": 1450}
                        )
                except Exception as e:
                    raise master.NetconfigError(e)

    # If vlan iface is mtn_host64, then match prefixes only
    if interface_target_state.mtn_host64:
        logging.info('Match prefixes only')

        found_ip = has_mtn_fb_bb_prefix_ip(current_addresses)
        if not found_ip:
            logging.info('Not found any global IP on iface {}. Need to add {}'.format(interface_target_state.name, interface_target_state.cidr))
            if dry_run:
                return True
            else:
                logging.info('Setting {} on {}'.format(interface_target_state.cidr, interface_target_state.name))
                master.set_ipv6_address(interface_target_state.name, interface_target_state.cidr)
        else:
            if not match_prefixes(found_ip[0], interface_target_state.cidr):
                logging.info('Prefixes don\'t match {} {}'.format(found_ip[0], interface_target_state.prefix))
                if dry_run:
                    return True
                else:
                    addr_to_add = netaddr.IPNetwork(interface_target_state.network)
                    addr_to_add.value = addr_to_add.value | master.MTN_HOST64_HOSTID  # badcab1e
                    logging.info('Setting {} on {}'.format(addr_to_add, interface_target_state.name))
                    master.set_ipv6_address(interface_target_state.name, interface_target_state.cidr)
            else:
                logging.info('Prefixes are matches {} {}'.format(found_ip[0], interface_target_state.cidr))
                current_addresses.remove(found_ip)

    else:
        if global_ipv6 in current_addresses:
            logging.info('Remove {} from set: {}'.format(global_ipv6, current_addresses))
            current_addresses.remove(global_ipv6)
            logging.info('Current addresses: {}'.format(current_addresses))
        else:
            if dry_run:
                # mark that we need fix
                logging.info('There is no {} on {}. Need to add.'.format(global_ipv6, interface_target_state.name))
                return True
            else:
                # Fix here
                logging.info('Setting {} on {}'.format(global_ipv6, interface_target_state.name))
                master.set_ipv6_address(interface_target_state.name, interface_target_state.cidr)

    # If current_addresses is not empty it means, that we have some old ips on iface, we should delete them.
    # If current interface is physical, check that we won't delete the last one global address first.
    if current_addresses:
        logging.info('There are some old ips on {}: {}'.format(interface_target_state.name, current_addresses))
        if is_physical_interface(interface_target_state.name):
            # At this point, we probably added all needed ips on iface.
            # Let's check that we won't delete the last one global address from physical iface.
            # Don't bother about link local addresses, get global addresses only,
            # so set scope=0 (see /etc/iproute2/rt_scopes)
            new_set_of_addresses = get_current_addresses_on_iface(interface_target_state.name, scope=0)
            logging.info('Got new set of addresses: {}'.format(new_set_of_addresses))
            diff = new_set_of_addresses - current_addresses
            msg = 'These global addresses should stay (new_set_of_addresses - current_addresses): {}'.format(diff)
            logging.info(msg)

            # Probably, pretty naive logic
            if not diff:
                msg = ("Interface {} is physical, we should delete some adresses from it,"
                       "but there is no diff => don't do anything".format(interface_target_state.name))
                logging.info(msg)
                return False

        if dry_run:
            # mark that we need fix here
            return True
        else:
            for addr_prefix in current_addresses:  # addr_prefix is like ('fe80::a:9', 64)
                addr = '/'.join((addr_prefix[0], str(addr_prefix[1])))
                logging.info('Delete {} from {}'.format(addr, interface_target_state.name))
                stdout, _, ret = master.run_cmd([
                    'ip', '-6', 'addr', 'del', addr, 'dev', interface_target_state.name
                ])
                if ret:
                    logging.info('Could not delete address {} from interface {}'.format(addr, interface_target_state.name))
    # No fixes needed
    return False


def form_set_from_ts_routes(iface_ts_routes):
    set_of_routes = set()
    for route in iface_ts_routes:
        prefix = route.route
        if prefix == "::/0":
            prefix = "default"
        table = route.table
        if not table:
            table = 'main'
        route_struct = (route.dev, route.gw, route.mtu, prefix, table)
        set_of_routes.add(route_struct)
    return set_of_routes


def form_set_of_current_routes(iface_name):

    # we need to use regexes because different versions of iproute2 spits different results :(
    mtu_re = r'mtu\s(\w+\s)?(?P<mtu>\d+)'
    gw_re = r'via\s(?P<gw>(\w+:?)+\::\w+)'
    table_re = r'table\s(?P<table>\w+)'

    set_of_routes = set()
    stdout, _, ret = master.run_cmd(
        ['ip', '-o', '-d', '-6', 'route', 'show', 'table', 'all', 'dev', iface_name]
    )
    if ret or not stdout:
        return set_of_routes

    # ignore prefixes, created by kernel, such as:
    # - ff00::/8 - multicast
    # - fe80::/64 - link-local
    # - for example, 2a02:6b8:b000:657::/64 - creates on adding IP to interface
    routes_txt = [route for route in stdout.splitlines() if 'via' in route]
    set_of_routes = set()
    for route in routes_txt:
        route_sp = route.split()
        prefix = route_sp[0]
        if prefix == 'unicast':  # don't write regex for prefixes => use dirty hack for newer versions of iproute2 :-|
            prefix = route_sp[1]

        gw = re.search(gw_re, route).group('gw')
        # try for olders versions of iproute2
        try:
            table = re.search(table_re, route).group('table')
        except AttributeError:
            logging.info('There is no \'table\' for route:\n{}\nSet to \'main\'.'.format(route))
            table = 'main'

        try:
            mtu = int(re.search(mtu_re, route).group('mtu'))
        except:
            logging.info('Could not get MTU for route {} so skip it'.format(route))
            continue

        dev = iface_name
        set_of_routes.add((dev, gw, mtu, prefix, table))

    return set_of_routes


def routes_diffs(interface_target_state, return_len=False):
    set_of_ts_routes = form_set_from_ts_routes(interface_target_state.routes)
    set_of_current_routes = form_set_of_current_routes(interface_target_state.name)
    old_routes = set_of_current_routes - set_of_ts_routes
    new_routes = set_of_ts_routes - set_of_current_routes
    changed_routes = set()
    # TODO: check if this is good logic
    for new_r in new_routes:
        for old_r in old_routes:
            # check if prefix is the same but gateway or mtu has been changed
            if old_r[3] == new_r[3] and (old_r[1] != new_r[1] or old_r[2] != new_r[2]):
                changed_routes.add(old_r)

    if return_len:
        len_old_routes = len(old_routes)
        for table in ('main', '6010'):
            formed_route = (interface_target_state.name, 'fe80::1', DEF_ROUTE_MTU, 'default', table)
            if formed_route in old_routes:
                len_old_routes -= 1

        return (len(new_routes), len_old_routes, len(changed_routes))
    else:
        return new_routes, old_routes, changed_routes


def fix_routes(interface_target_state, dry_run):

    # Form sets from states
    set_of_ts_routes = form_set_from_ts_routes(interface_target_state.routes)
    logging.info('SET_OF_TS_ROUTES:\n{}'.format(pprint.pformat(set_of_ts_routes)))
    set_of_current_routes = form_set_of_current_routes(interface_target_state.name)
    logging.info('SET_OF_CURRENT_ROUTES:\n{}'.format(pprint.pformat(set_of_current_routes)))

    if not set_of_current_routes:
        if dry_run:
            return True
        else:
            # Apply routes from set of target_state routes
            for route in set_of_ts_routes:
                # example of route: ('vlan688', 'fe80::1', 8910, '2a02:6b8::/32', '688')
                # master.add_route(route, gateway, mtu, dev, table=None):
                master.add_route(route[3], route[1], route[2], route[0], route[-1])
            return False

    # Get diffs
    new_routes, old_routes, changed_routes = routes_diffs(interface_target_state)

    logging.info('OLD_ROUTES:\n{}'.format(pprint.pformat(old_routes)))
    logging.info('NEW_ROUTES:\n{}'.format(pprint.pformat(new_routes)))
    logging.info('CHANGED_ROUTES:\n{}'.format(pprint.pformat(changed_routes)))

    if fix_def_routes(old_routes, dry_run) and dry_run:
        # Need to fix
        return True

    if len(new_routes) == len(old_routes) == 0:
        # No need to fix
        return False

    # Don't remove routes if iface is physical
    if is_physical_interface(interface_target_state.name) and len(new_routes) == 0:
        logging.info('{} is physical - don\'t remove any of routes'.format(interface_target_state.name))
        return False

    if dry_run:
        return True
    else:
        # Add new routes
        for route in new_routes:
            # example of route: ('vlan688', 'fe80::1', 8910, '2a02:6b8::/32', '688')
            # master.add_route(route, gateway, mtu, dev, table=None):
            route_prefix = route[3]

            # cast 'default' prefix to '::/0' because master.add_route() check it via master.get_ipaddress_version()
            if route_prefix == 'default':
                route_prefix = '::/0'
            formed_route = (route_prefix, route[1], route[2], route[0], route[-1])
            master.add_route(*formed_route)

        # Don't delete anything if device is physical
        if is_physical_interface(interface_target_state.name):
            logging.info('Interface {} is physical, don\'t delete routes'.format(interface_target_state.name))
            return False

        # Delete old routes
        for route in old_routes:
            # master.del_route(route, gateway, iface, table=None):
            if route not in changed_routes:
                master.del_route(route[3], route[1], route[0], route[-1])

    return False


def fix_rules(target_states_interfaces, dry_run):

    mtn_vlan_ifaces = [iface for iface in target_states_interfaces if iface.network]

    if not mtn_vlan_ifaces:
        return False

    # Form rules that should exist
    target_rules = set()
    template_for_main_table = str(master.PBR_PRIORITY) + ': from {} to {} lookup main'
    template_for_vlanid_table = str(master.PBR_PRIORITY + 5) + ': from {} lookup {}'
    for mtn_vlan in mtn_vlan_ifaces:
        logging.info('MTN_VLAN: {}'.format(mtn_vlan))
        vlan_id = mtn_vlan.name.replace('vlan', '')

        logging.info(template_for_main_table.format(mtn_vlan.prefix, mtn_vlan.network))
        target_rules.add(template_for_main_table.format(mtn_vlan.prefix, mtn_vlan.network))

        logging.info(template_for_vlanid_table.format(mtn_vlan.prefix, vlan_id))
        target_rules.add(template_for_vlanid_table.format(mtn_vlan.prefix, vlan_id))

    logging.info('SET_OF_TARGET_RULES:\n{}'.format(target_rules))
    rules = master.run_cmd(['ip', '-6', 'rule'])[0].split('\n')
    logging.info('SET_OF_CURRENT_RULES_BEFORE_FILTER: {}'.format(rules))
    rules = {' '.join(rule.strip().split()) for rule in rules if 'from all' not in rule and rule}
    logging.info('SET_OF_CURRENT_RULES_AFTER_FILTER: {}'.format(rules))

    # Check if there are rules with prefixes, that are not from MTN_BACKBONE_PREFIX
    ipn_mtn_backbone = ipaddr.IPNetwork(master.MTN_BACKBONE_PREFIX)
    ipn_mtn_fastbone = ipaddr.IPNetwork(master.MTN_FASTBONE_PREFIX)
    for cur_rule in rules:
        cur_prefix = cur_rule.split()[2]
        ipn_cur_prefix = ipaddr.IPNetwork(cur_prefix)
        logging.info('CUR_PREFIX: {}'.format(cur_prefix))
        if ipn_cur_prefix not in ipn_mtn_backbone and ipn_cur_prefix not in ipn_mtn_fastbone:
            # We don't want to do anything if there are such prefixes, so always return True (dry_run flag is ignored)
            return NOT_MTN_RULES_MSG

    new_rules = target_rules - rules
    old_rules = rules - target_rules
    logging.info('OLD_RULES: {}\nNEW_RULES: {}\n'.format(old_rules, new_rules))
    if new_rules or old_rules:
        if dry_run:
            return 'Found diff in rules'
        else:
            for new_rule in new_rules:
                add_rule(new_rule)
            for old_rule in old_rules:
                del_rule(old_rule)


def add_rule(rule):
    logging.info('Add rule: {}'.format(rule))

    rule_match = RULE_RE_COMPILED.match(rule)

    from_net = rule_match.group('src')
    to_net = rule_match.group('dst')
    table = rule_match.group('table')

    master.add_pbr_rule(from_net, table, to_net)


def del_rule(rule):
    logging.info('Delete rule: {}'.format(rule))

    rule_match = RULE_RE_COMPILED.match(rule)

    priority = int(rule_match.group('prio'))
    from_net = rule_match.group('src')
    to_net = rule_match.group('dst')
    table = rule_match.group('table')
    if table == 'main':
        table = 254
    table = int(table)

    with IPRoute() as ipr:
        ipr.rule(
            "del",
            family=socket.AF_INET6,
            src=from_net,
            dst=to_net,
            table=table,
            priority=priority,
        )


def route_to_proto(route_dict):
    route_proto = fixutil_pb2.Route()
    route_proto.dev = route_dict.get('dev')
    route_proto.gw = route_dict.get('gw')
    route_proto.mtu = route_dict.get('mtu')
    route_proto.route = route_dict.get('route')
    route_proto.table = route_dict.get('table', '')
    return route_proto


def get_interfaces(path=None):
    interfaces = master.DebianInterfaces(path)
    interface_instances = []
    for _, ifaces in interfaces.ifaces.items():
        for i in ifaces:
            interface_instances.append(i)
    return interface_instances


def fix_def_routes(set_of_old_routes, dry_run=True):
    # check if default gateway is fe80::1 and mtu is 1450 on it
    for route in set_of_old_routes:
        dev = route[0]
        table = route[-1]  # vlan_id or 'main'
        prefix = route[3]  # prefix
        if prefix == 'default' and table in ('main', '688'):
            gw = route[1]
            old_mtu = route[2]
            if gw != 'fe80::1' or old_mtu != DEF_ROUTE_MTU:
                logging.info("Default route is incorrect!")
                if dry_run:
                    # Need to fix
                    return True
                else:
                    logging.info("Working on default route for {}".format(table))
                    master.add_route('::/0', 'fe80::1', DEF_ROUTE_MTU, dev, table)
    # Don't need to fix or already added route
    return False


def need_to_fix(results):

    # In fact, there is no way to get multiple NOT_MTN_RULES_MSG in results list,
    # but made this cycle just for insurance.
    num_of_not_mtn_msgs = results.count(NOT_MTN_RULES_MSG)

    return num_of_not_mtn_msgs < len(results)


def set_link_down(iface_name):
    with IPRoute() as ipr:
        try:
            link_id = ipr.link_lookup(ifname=iface_name)
            ipr.link('set', index=link_id, state='down')
        except Exception as e:
            logging.error("Could not set {} down:\n{}".format(iface_name, e))


def enforce_sysctls():
    master.set_sysctl('net.ipv6.conf.all.proxy_ndp', '0')
