#!/usr/bin/env python
# pylint: disable=W0703
# -*- coding: utf-8 -*-
"""Configures backbone and fastbone interfaces with routes."""

from collections import defaultdict
import errno
from glob import glob
import datetime
import hashlib
import fcntl
import json
import logging
import os
import netaddr
import pprint
import struct
import socket
import subprocess
import sys
import time
import traceback

import ipaddr

# Monkey patch:
# urllib3 is too smart,
# it considers that we have no ipv6
# when it fails to bind on ::1.
#
# The noc-export is ipv6-only, and if loopback iface
# is down on the moment of invocation, we cannot
# setup network on host image bootstrap stage.
#
# See contrib/python/urllib3/urllib3/util/connection.py:130

import urllib3
urllib3.util.connection.HAS_IPV6 = True
import requests

from pyroute2 import IPRoute

from infra.netconfig.lib.retrying import retry
from infra.netconfig.lib import lldputil
from infra.netconfig.lib import aggregate_routes
from infra.netconfig.lib.exceptions import NetconfigError
from infra.netconfig.lib.exceptions import NetconfigSkip
from infra.netconfig.lib.exceptions import NetconfigNoSysctl
from infra.netconfig.lib.exceptions import NetconfigMTUError

DENY_IFACE_NAMES = (
    '--all', 'lo', 'vlan', 'tun', 'tunl',
    'lxcbr', 'mc', 'tap', 'vif', 'pflog',
    'virbr', 'plip', 'ip6tnl', 'ip4tnl', 'dummy'
)

WAIT_CARRIER_SECONDS = 10  # wait interface for connection
WAIT_CARRIER_AFTER_MTU_SECONDS = 30  # wait for interface to wake up after MTU set
WAIT_RA_SECONDS = 36  # wait ra

IFUPDOWN_CONFIG_TIMEOUT = 900

RUN_NETWORK_DIR = '/run/network/ya-netconfig'
NETCONFIG_STATE_PATH = '%s/ya_netconfig_state.json' % RUN_NETWORK_DIR

NETWORKS_CONFIG_URL = 'https://noc-export.yandex.net/rt/l3-segments2.json'
NETWORKS_CONFIG_CACHE = '/var/cache/network/ya-netconfig/networks.json'
NETWORKS_CONFIG_TRIES_TO_DOWNLOAD = 5
NETWORKS_CONFIG_WAIT_CONNECTION_SECONDS = 5
NETWORKS_CONFIG_WAIT_BETWEEN_TRIES_SECONDS = 3
NETWORKS_CONFIG_EXPIRE_SECONDS = 3600

HOST64_IFNAMES_PATH = '/usr/lib/yandex-netconfig/host-64-ifnames.json'

NETWORK_CONFIG_PATCHES_PATH = '/usr/lib/yandex-netconfig/patches'
NETWORK_CONFIG_PROJECT_ID = '/etc/network/projectid'

MTU_IFACE = 9000
MTU_ROUTES = 8910
MTU_DEFAULT_ROUTE = 1450

PBR_PRIORITY = 16383

MTN_BACKBONE_PREFIX = "2a02:6b8:c00::/40"
IPADDR_MTN_BACKBONE_PREFIX = ipaddr.IPNetwork(MTN_BACKBONE_PREFIX)

MTN_FASTBONE_PREFIX = "2a02:6b8:fc00::/40"
IPADDR_MTN_FASTBONE_PREFIX = ipaddr.IPNetwork(MTN_FASTBONE_PREFIX)

LLDP_GET_RETRY_COUNT = 300

SIOCGIFMTU = 0x8921  # ioctl to get MTU
SIOCSIFMTU = 0x8922  # ioctl to set MTU

# This is host id part of global ipv6 mtn host64 address of vlan iface.
MTN_HOST64_HOSTID = int('badcab1e', 16)

DEF_BASE_REACH_TIME = '3600000'
DEF_RETRANS_TIME = '30000'


def usage():
    """Show usage information."""
    print("""Usage:
        ya-netconfig [action [interface]]

    Where:
        action      start|stop|status|state|fetch
        interface   name of physical active interface

    You may also specify action as MODE env variable and interfaces as IFACE.
""")


def dict_has_value(d, key, value):
    """
    Gets specified key value from anywhere in dictionary.

    Used to find if any VLAN has option specified in network info.
    """
    for k, v in d.items():
        if k == key and v == value:
            return True
        elif isinstance(v, dict):
            if dict_has_value(v, key, value):
                return True
    return False


def dict_override_value(d, key, value):
    if key in d:
        d[key] = value

    for k, v in d.items():
        if isinstance(v, dict):
            dict_override_value(v, key, value)


def switch_symbols(some_str, new, old):
    # Snached from procps package sysctl.c
    new_str = ""
    for i in some_str:
        if i == old:
            new_str += new
            continue
        if i == new:
            new_str += old
            continue
        new_str += i
    return new_str


def set_sysctl(name, value):
    logging.info('Setting sysctl %s=%s', name, value)
    name = switch_symbols(name, '.', '/')
    try:
        with open('/proc/sys/%s' % (name,), 'wb') as f:
            f.write('%s\n' % (value,))
    except IOError as e:
        if e.errno == errno.ENOENT:
            raise NetconfigNoSysctl(name)
        else:
            raise NetconfigError("IOError while setting %s=%s: %s" % (name, value, e))
    except Exception as e:
        raise NetconfigError("Failed to set %s=%s: %s" % (name, value, e))


def get_sysctl(name):
    logging.debug('Getting sysctl %s', name)
    name = switch_symbols(name, '.', '/')
    try:
        with open('/proc/sys/%s' % (name,), 'r') as f:
            return f.read(4096).strip()
    except IOError as e:
        if e.errno == errno.ENOENT:
            raise NetconfigNoSysctl(name)
        else:
            raise NetconfigError("IOError while getting %s: %s" % (name, e))
    except Exception as e:
        raise NetconfigError("Failed to get %s: %s" % (name, e))


def get_mtu(ifname):
    """
    Returns MTU set on specified interface.
    """
    s = socket.socket(socket.SOCK_DGRAM)
    try:
        ifr = ifname + '\x00' * (32 - len(ifname))
        ifs = fcntl.ioctl(s, SIOCGIFMTU, ifr)
        return struct.unpack('<i', ifs[16:20])[0]
    except Exception as e:
        raise NetconfigError('Unable to get mtu on %s: %s', ifname, e)
    finally:
        s.close()


class Timer(object):
    """
    Simple timer implementation. Has basic defence against misuse.
    """

    def __init__(self, start=False):
        self._is_running = False
        self._start = 0
        self._stop = 0
        if start:
            self.start()

    def format_elapsed(self):
        return "{0:.5f} sec.".format(self.elapsed)

    @property
    def started_at(self):
        return self._start

    @property
    def stopped_at(self):
        return self._stop

    @property
    def elapsed(self):
        return self._stop - self._start

    def start(self):
        if self._is_running:
            raise RuntimeError('timer is already running')
        self._start = time.time()
        self._is_running = True

    def stop(self):
        if not self._is_running:
            raise RuntimeError('timer is not running')
        self._stop = time.time()
        self._is_running = False

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stop()


class Interface(object):
    """
    Network interface information.
    """

    @staticmethod
    def from_env(name):
        """
        Creates interface from environment variables.
        """
        interface = Interface(name)
        interface.set_option('addrfam', os.getenv('ADDRFAM', None))
        interface.set_option('method', os.getenv('METHOD', None))
        for var, val in os.environ.items():
            if var.startswith('IF_'):
                interface.set_option(var, val)
        return interface

    @staticmethod
    def _option_name(name):
        """
        Universalize option name.
        """
        option = name.strip().upper().replace('-', '_')
        if option.startswith('IF_') or option in ('METHOD', 'ADDRFAM'):
            return option
        return 'IF_' + option

    def __init__(self, name):
        self.name = name
        self.options = {}

    def option(self, name):
        """
        Returns an interface option.
        """
        option = self._option_name(name)
        if option in self.options:
            return self.options[option]
        return None

    def bool_option(self, name):
        """
        Returns boolean interfaces option value or None
        """
        return is_enabled(self.option(name))

    def numeric_option(self, name):
        """
        Returns integer interface option value or None
        """
        option = self._option_name(name)
        if option in self.options:
            value = self.options[option]
            try:
                return int(value)
            except ValueError:
                logging.warning('Invalid numeric value %s for option %s', value, option)

        return None

    def set_option(self, name, value):
        """Set interface option."""
        self.options[self._option_name(name)] = value
        return self

    def show(self):
        """Show information about interface."""
        print('%s has option %s' % (self.name, self.options))

    def __str__(self):
        return self.name

    def __repr__(self):
        return self.name


class DebianInterfaces(object):
    """
    An object representation of Debian specific network
        interfaces configuration file.
    """

    def __init__(self, path=None):
        self.stanzas = ('iface', 'mapping', 'auto', 'allow', 'source')
        self.ifaces = defaultdict(list)
        self.auto = []
        self.parsedconfigs = []
        self._parse(path or '/etc/network/interfaces')

    def get_iface_definitions(self, name):
        """Return a list of interface, by given name."""
        if name not in self.ifaces:
            return []
        return self.ifaces[name]

    def _parse(self, config):
        """Parse configuration file."""
        abspath = os.path.abspath(config)
        if not os.path.isfile(abspath) or abspath in self.parsedconfigs:
            return
        self.parsedconfigs += [abspath]
        for stanza in self._get_stanzas(abspath):
            head = stanza[0]
            if head.startswith(('auto', 'allow-auto')):
                self._parse_auto(head)
            elif head.startswith('iface'):
                self._parse_iface(stanza)
            elif head.startswith('source'):
                self._parse_source(head, os.path.dirname(abspath))

    def _parse_auto(self, stanza):
        """
        Parses 'auto' stanza.
        """
        self.auto += stanza.replace('allow-', '').replace('auto', '').split()
        return self

    def _parse_iface(self, stanza):
        """
        Parses iface stanza.
        """
        _, name, addrfam, method = stanza[0].split()
        iface = Interface(name)
        iface.set_option('method', method)
        iface.set_option('addrfam', addrfam)
        for params in stanza[1:]:
            opt = params.split(None, 1)
            if len(opt) > 1:
                iface.set_option(opt[0], opt[1])
        self.ifaces[name].append(iface)
        return self

    def _parse_source(self, stanza, curdir):
        """Parse 'source' stanza with including sources."""
        splitted = stanza.split(None, 1)
        if len(splitted) != 2:
            return self
        source_type, search = splitted
        if not search.startswith('/'):
            search = curdir + '/' + search
        found = glob(search)
        if source_type.startswith('source-directory'):
            for found_dir in found:
                if not os.path.isdir(found_dir):
                    continue
                for config in os.listdir(found_dir):
                    self._parse(found_dir + '/' + config)
            return self
        for config in found:
            self._parse(config)
        return self

    def _get_stanzas(self, config):
        """
        Returns a generator of stanzas by given config.
        """
        with open(config) as conf:
            lines = []
            for line in conf:
                strline = line.strip()
                if not strline or strline.startswith('#'):
                    continue
                if not strline.startswith(self.stanzas):
                    lines.append(strline)
                    continue
                if lines:
                    yield lines
                lines = [strline]
        if lines:
            yield lines


def is_enabled(value):
    """Check, that given variable is enabled.

    Args:
        value: a string or None, checked value.

    Returns:
        A boolean, True if given value is enabled.

    """
    if not value:
        return False
    val = value.strip().lower()
    if ((val.isdigit() and int(val) > 0)
            or val == 'yes' or val == 'enable'
            or val == 'on' or val == 'true'):
        return True
    return False


def is_env_enabled(env):
    """Check, that OS environment variables is set and value is enable.

    Args:
        env: a string, name of checked environment variable.

    Returns:
        A boolean, True if given variable exists and it's value is true.

    """
    return is_enabled(os.getenv(env, None))


def get_log_level_for(action):
    """
    Choose log level depending on current action.
    E.g. do not log everything on 'fetch' because it is started from cron.
    """
    if action == 'fetch':
        return logging.ERROR
    return logging.DEBUG


def setup_logging(level):
    """
    Setup log messages output format.
    """
    debug = (is_env_enabled('DEBUG') or
             is_env_enabled('IF_DEBUG') or
             is_env_enabled('VERBOSITY'))
    if debug:
        level = logging.DEBUG
    logging.basicConfig(format='[%(asctime)s] %(levelname)s: %(message)s',
                        level=level)


def run_cmd(cmd):
    """Safety run command.

    Args:
        cmd: a list of commands and it's args.

    Returns:
        A list of stdout, stderr, returncode.
        If can't run a commnand returncode will be -1.

    """
    try:
        proc = subprocess.Popen(
            cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE
        )
        stdout, stderr = proc.communicate()
        returncode = proc.returncode
        logging.debug(
            'Run: %s\n\tstdout: %s\n\tstderr: %s\n\texit: %s',
            ' '.join(cmd), stdout, stderr, returncode
        )
    except Exception as error:
        logging.warn('Command `%s` has an error %s', ' '.join(cmd), error)
        stdout = ''
        stderr = str(error)
        returncode = -1
    return stdout, stderr, returncode


def make_dirs(path):
    """
    Safely create a full path dirs.

    Args:
        path: a string, an ended path to directory

    Returns:
        A boolean, True if success, False otherwise.

    """
    if os.path.isdir(path):
        return True
    try:
        os.makedirs(path, mode=0755)
        return True
    except EnvironmentError as e:
        # Directory might have already been created,
        # but it can be a file, so check before returning.
        if e.errno == errno.EEXIST:
            return os.path.isdir(path)
        logging.error('Failed to create "%s": %s', path, e)
        return False


def is_iface_created(iface):
    return os.path.exists('/sys/class/net/%s' % iface)


def is_carrier_up(path):
    try:
        with open(path, 'r') as carrier:
            if '1' in carrier.read(128):
                logging.debug('%s is up', path)
                return True
    except Exception:
        pass
    return False


def is_iface_connected(iface, wait=WAIT_CARRIER_SECONDS):
    """
    Waits seconds while interface carrier.

    Args:
        iface: a string, name of checkecked interface.
        wait: an integer, time for waiting interface carrier.

    Returns:
        A boolean, True if interface is connected.
    """

    carrier_file = '/sys/class/net/%s/carrier' % iface

    if wait <= 0:
        return is_carrier_up(carrier_file)

    while wait > 0:
        if is_carrier_up(carrier_file):
            return True
        time.sleep(1)
        wait -= 1

    logging.debug('Interface %s is not connected', iface)
    return False


def is_interface_allowed(interface):
    """Check that given interface in a yanetconfig.

    Args:
        interface: an object of Interface class.

    Returns:
        A boolean, True if given interface in allow-yanetconfig group.
        Also, if there are no allow-yanetconfig string always return True.

    """
    if interface.bool_option('ya-netconfig-disable'):
        logging.debug('ya-netconfig-disable on for %s', interface)
        return False

    stdout = run_cmd(
        ['ifquery', '--allow', 'yanetconfig', '--list']
    )[0]

    if stdout and (str(interface) not in stdout.splitlines()):
        logging.debug('Interface %s not in yanetconfig class', interface)
        return False

    return True


def get_global_ipaddress(iface, ip_version):
    """ Get global ip address via pyroute.
        Migrate to this function in future.
    """
    logging.info('Getting global ip address.')
    ip_version_2_family_map = {
        4: 2,
        6: 10,
    }

    try:
        with IPRoute() as ipr:
            link_index = ipr.link_lookup(ifname=iface)[0]
            addr_info = ipr.get_addr(index=link_index, scope=0, family=ip_version_2_family_map[ip_version])[0]
            ip_addr = addr_info.get_attr('IFA_ADDRESS')
            prefixlen = addr_info['prefixlen']
            return '/'.join((ip_addr, str(prefixlen)))
            # return netaddr.IPNetwork('/'.join((ip_addr, str(prefixlen))))
    except Exception as e:
        logging.info('Could not get global ip addr.\n{}'.format(e))
        return ""


def get_mac_address(iface):
    """ Get a mac address of given interface via pyroute.
        Migrate to this in future.
    """
    try:
        with IPRoute() as ipr:
            link_index = ipr.link_lookup(ifname=iface)[0]
            iface_info = ipr.get_links(link_index)[0]
            mac_addr = iface_info.get_attr('IFLA_ADDRESS')
            return mac_addr
            # return netaddr.EUI(mac_addr, dialect=netaddr.mac_unix)

    except Exception as e:
        raise NetconfigError('Could not get MAC for interface: {}\n{}'.format(iface, e))


def get_my_hostname():
    """Return a my hostname."""
    # simplest way to get hostname from /etc/hostname
    return os.uname()[1]


def get_mac_hostid(mac):
    try:
        netaddr.EUI(mac, dialect=netaddr.mac_unix)
    except:
        raise Exception(
            'Looks like MAC address is not correct {}'.format(mac)
        )
    hostid = int(mac.replace(':', '')[-8:], 16)
    return hostid


def get_hostname_hostid(hostname=None):
    if not hostname:
        hostname = get_my_hostname()

    hostid = int(hashlib.md5(hostname).hexdigest()[:8], 16)
    return hostid


def get_manual_hostid(hostid):
    try:
        hostid_int = int(hostid, 16)
        if hostid_int <= 4294967295:  # '0x100000000'
            return hostid_int
        else:
            return int(hashlib.md5(hostid).hexdigest()[:8], 16)
    except ValueError:
        return int(hashlib.md5(hostid).hexdigest()[:8], 16)


def get_projectid_address(iface, project_id, host_method, ra_prefix=None):
    """Get ip address for project id.

    Args:
        iface: interface name
        project_id: a project_id
        host_method: a method to generate host address
        reset: reset previous address

    Returns:
        A string, IPv6 address or empty.

    """

    if ra_prefix is None:
        ra_prefix = get_ra_prefix(iface)

    try:
        ra_prefix = netaddr.IPNetwork(ra_prefix)
    except ValueError as err:
        raise NetconfigError('Could not parse ra_prefix: {}'.format(err))

    if ra_prefix.prefixlen < 48:
        raise NetconfigError('Got incompatible RA prefix, prefix_length < 48: {}'.format(ra_prefix.prefixlen))

    if host_method == 'mac':
        hostid = get_mac_hostid(get_mac_address(iface))
    elif host_method == 'hostname':
        hostid = get_hostname_hostid()
    else:
        hostid = get_manual_hostid(host_method)

    ip_part = netaddr.IPAddress('::')

    if project_id:
        project_id = int(project_id, 16)
    else:
        raise NetconfigError('Project ID Not Set')

    if project_id > 4294967295:
        raise NetconfigError('Project ID is more than 4 bytes')

    ip_part.value = project_id << 32 | hostid
    ip_part.value = ra_prefix.ip.value | ip_part.value

    result = netaddr.IPNetwork(ip_part)
    result.prefixlen = ra_prefix.prefixlen

    return str(result)


def get_bridge_interfaces(iface):
    """Return a list of bridge interfaces.

    This function use command like 'brctl show br0', wich return next output:
    bridge name	bridge id		    STP enabled	    interfaces
    br0		    8000.fcaa14d9a2d2	no		        eth0
                                                    eth1

    Args:
        iface: a string, name of bridge interface.

    Returns:
        A list of interfaces, that included in a specified bridge.

    """
    ifaces = []

    stdout, stderr, _ = run_cmd(['brctl', 'show', iface])
    if stderr or not stdout:
        return ifaces

    # remove header line
    lines = stdout.splitlines()[1:]
    # first output line has a specific format, choose last iface
    if lines:
        ifaces.append(lines[0].split()[3])
        lines = lines[1:]
    for line in lines:
        ifaces.append(line.split()[0])
    return ifaces


def disable_ipv6_privext(iface):
    """Disable IPv6 private extension on interface.

    This function also know about bridge interfaces, and disable 'use_tempaddr'
    for all bridge ports.

    Args:
        iface: a string, name of configureable interface.

    Returns:
        A boolean, True always.

    """
    if iface.startswith('br'):
        for i in get_bridge_interfaces(iface):
            disable_ipv6_privext(i)
        time.sleep(10)  # this is magic wait, change it if you know what to do
    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.use_tempaddr', '0')


def get_ipv6_ra(interface, accept_default_route=True, reset=False):
    """Enable IPv6 Router Advertisement on iface.

    Args:
        iface: a string, name of configureable interface.
        accept_default_route: a boolean, a flag that accept default router
            from RA.
        reset: a boolean, a flat to set sysctl to enable ipv6 ra

    Returns:
        A boolean, True if address was successfully received and setuped.

    """
    if isinstance(interface, Interface):
        iface = interface.name
    else:
        iface = interface

    logging.debug('Configure RA on %s', iface)

    if reset:
        set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.disable_ipv6', '1')
        enable_ipv6_ra(iface)

    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.accept_ra', '2')
    disable_ipv6_privext(iface)
    if accept_default_route:
        set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.accept_ra_defrtr', '1')
        try:
            set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.ra_default_route_mtu', '1450')
        except NetconfigNoSysctl:
            # Yandex-specific sysctl with optional semantics
            logging.error("Ignoring non-existing sysctl ra_default_route_mtu")
    else:
        set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.accept_ra_defrtr', '0')
    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.accept_dad', '0')
    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.dad_transmits', '0')

    # Ensure ipv6 is enabled
    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.disable_ipv6', '0')

    # Ensure that network interface is up
    # Not sure if it is needed, but we have issues when
    # no RA for unknown reasons
    is_iface_connected(iface, WAIT_CARRIER_AFTER_MTU_SECONDS)

    # run rdisc to force RA request
    prefix = get_ra_prefix(iface)

    if ipaddr.IPNetwork(prefix) in ipaddr.IPNetwork(MTN_BACKBONE_PREFIX):
        logging.debug('Project ID network')
        disable_ipv6_ra(iface, True)
        cidr = get_projectid_backbone_address(interface, prefix)
        set_ipv6_address(iface, cidr)
        time.sleep(3)
        addr = cidr.split('/')[0]
        run_cmd([
            'ip', '-6', 'route', 'replace', 'default',
            'via', 'fe80::1', 'dev', iface,
            'src', addr, 'metric', '1', 'mtu', '1450'
        ])
        return cidr
    else:
        wait = WAIT_RA_SECONDS
        while wait > 0:
            try:
                return get_global_ipaddress(iface, 6)
            except NetconfigError:
                pass

            time.sleep(2)
            wait -= 2

    raise NetconfigError('IPv6 global unicast not found on %s after RA', iface)


def enable_ipv6_ra(iface):
    """
    Enables autoconf IPv6 address for given interface.
    """
    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.autoconf', '1')
    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.accept_ra', '2')


def disable_ipv6_ra(iface, reset=False):
    """Disable via sysctl IPv6 RA.

    Args:
        iface: a string, inteface name.
        reset: a boolean, disable IPv6 address before reset.

    Returns:
        None.
    """
    if reset:
        set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.disable_ipv6', '1')
        set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.accept_ra', '0')

    disable_ipv6_privext(iface)

    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.autoconf', '0')
    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.accept_dad', '0')
    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.disable_ipv6', '0')


@retry(retry_on_result=lambda x: x is None,
       wait_fixed=1000, stop_max_delay=60000)
def get_ra_prefix(iface):
    """Get IPv6 RA prefix via rdisc6.

    Args:
        iface: a string, name of interface.

    Returns:
        A string - RA prefix (like '2a02:6b8:b000:b900::/57')
        None if failed.

    """
    out, err, code = run_cmd([
        'timeout', '10', 'rdisc6', '-1',
        '-w', '2000', '-r', '5', '-n', iface
    ])

    if code != 0:
        raise NetconfigError("'rdisc' exit code: %s" % code)

    prefix = [s for s in out.split('\n') if ' Prefix ' in s]
    if not prefix:
        raise NetconfigError("There are no 'Prefix' in RA announce")

    return prefix[0].split(':', 1)[-1].strip()


def get_projectid_backbone_address(iface, prefix):
    """
    Get a projectid backbone address.
    """
    if iface.option('addrfam') != 'inet6':
        raise NetconfigError("Backbone address only for inet6 family")

    project_id = get_project_id(iface, True)
    method = iface.option('ya-netconfig-project-id-host-method') or 'hostname'
    return get_projectid_address(iface.name, project_id, method, prefix)


def get_backbone_address(interface, configure):
    """Get global ip address of specified interface.

    Args:
        interface: an object of Interace class.
        configure: a boolean, if True reset IPv6 address, before RA.

    Returns:
        A string, IPv4/IPv6 address or empty.

    """
    iface = interface.name
    addrfam = interface.option('addrfam')
    method = interface.option('method')
    anymethod = interface.bool_option('ya-netconfig-enable')
    accept_ra_gw = not interface.bool_option('ya-netconfig-ra-disable-default-gw')

    if None in (addrfam, method):
        raise NetconfigError(
            'One of ADDRFAM, METHOD variable not defined.'
            ' This is not ifupdown run script behaviour'
        )

    if not addrfam.startswith('inet'):
        raise NetconfigSkip('Unsupported protocol definition for ya-netconfig')

    if addrfam == 'inet' and (method == 'static' or anymethod):
        return get_global_ipaddress(iface, 4)

    if addrfam == 'inet':
        raise NetconfigSkip('Unsupported method definition for ya-netconfig')

    # now addrfam only inet6
    if method == 'auto':
        return get_ipv6_ra(interface, accept_ra_gw, configure)

    if method == 'static':
        if interface.bool_option('accept_ra'):
            try:
                get_ipv6_ra(interface, accept_ra_gw, False)
            except NetconfigError:
                pass

        return get_global_ipaddress(iface, 6)

    if anymethod:
        if method == 'manual':
            try:
                return get_global_ipaddress(iface, 6)
            except NetconfigError:
                return get_ipv6_ra(interface, accept_ra_gw)

        return get_global_ipaddress(iface, 6)

    raise NetconfigSkip('Unsupported method definition for ya-netconfig')


def atomic_write(path, data):
    """Write data to temporary file and move it into 'path'."""

    cache_dir = os.path.dirname(path)
    tmp_path = path + ".tmp"
    try:
        make_dirs(cache_dir)
        open(tmp_path, "w").write(data)
        os.rename(tmp_path, path)
        return True
    except Exception as e:
        logging.warn("Failed to write data to file '{}': {}".format(path, e))

    return False


def format_exception():
    etype, evalue, _ = sys.exc_info()
    return "\n".join(traceback.format_exception_only(etype, evalue))


class RemoteJsonConfig(object):

    def __init__(self, url, cache_path, cache_ttl, socket_timeout,
                 retry_kwargs):
        self.url = url
        self.cache_path = cache_path
        self.cache_ttl = cache_ttl
        self.socket_timeout = socket_timeout
        retry_ = retry(**retry_kwargs)
        self._fetch_and_validate = retry_(self._fetch_and_validate)

    def _get_from_cache(self):
        try:
            if os.path.exists(self.cache_path):
                cache_age = time.time() - os.path.getmtime(self.cache_path)
                if cache_age < self.cache_ttl:
                    msg = "Cache file '{}' not expired, using it."
                    logging.info(msg.format(self.cache_path))
                    self.config = json.load(open(self.cache_path))
                    return True
        except Exception:
            msg = "Unable to get data from cache: "
            logging.error(msg + format_exception() + ".")

        return False

    def _fetch_and_validate(self):
        self.data = requests.get(self.url, timeout=self.socket_timeout).content
        self.config = json.loads(self.data)

    def _fetch(self):
        logging.info("Fetching data from <%s>", self.url)
        try:
            self._fetch_and_validate()
        except Exception:
            msg = "Unable to fetch data: "
            logging.error(msg + format_exception() + ".")
            return False
        return True

    def get(self, store=True):
        if not self._get_from_cache():
            if self._fetch():
                if store:
                    atomic_write(self.cache_path, self.data)
            else:
                if not os.path.exists(self.cache_path):
                    raise NetconfigError('Unable to download file')

                logging.warning(
                    "Using expired cache file '{}'.".format(self.cache_path)
                )
                try:
                    self.config = json.load(open(self.cache_path))
                except Exception:
                    msg = "Unable to get data from cache."
                    logging.exception(msg)
                    raise NetconfigError(msg)

        return self.config

    def update_cache(self):
        if self._fetch():
            return atomic_write(self.cache_path, self.data)
        return False


def get_project_id(interface, silent=False):
    """Return a projectId of network.
    Args:
        interface: an object of Interface class.
        silent: don't show error message

    Returns:
        A string, project id for current interface
        None if failed.

    """
    project_id = interface.option('project-id')
    if project_id:
        return project_id

    try:
        with open(NETWORK_CONFIG_PROJECT_ID, 'r') as prj:
            return prj.read().strip()
    except Exception as error:
        if not silent:
            logging.error(
                '{} Failed to get project id from {}: {}'.format(interface.name, NETWORK_CONFIG_PROJECT_ID, error)
            )

    return 0


def get_port_index(port_name, host64_ifnames_path=HOST64_IFNAMES_PATH):
    try:
        host64ifnames = json.load(open(host64_ifnames_path, 'r'))
    except Exception as error:
        raise NetconfigError('Error while parsing json config: %s' % error)
    port_index = host64ifnames.get(port_name, {}).get('index', None)
    if port_index is None:
        raise NetconfigError('Unknown port %s name' % port_name)

    return hex(int(port_index)).split('0x')[1]


def get_lldp_port_index(interface):
    """ interface: string with name of interface (ex. eth0) """

    parsed_data = lldputil.parse_lldp_packet(interface)
    port_name = parsed_data['port_name']

    port_index = get_port_index(port_name)
    return port_index


def get_network_info(interface=None, store=True):
    networks_url = NETWORKS_CONFIG_URL
    if interface:
        networks_url = (interface.option('ya-netconfig-networks-url') or
                        NETWORKS_CONFIG_URL)

    networks_retry_kwargs = {
        "stop_max_attempt_number": NETWORKS_CONFIG_TRIES_TO_DOWNLOAD,
        "wait_fixed": NETWORKS_CONFIG_WAIT_BETWEEN_TRIES_SECONDS
    }
    return RemoteJsonConfig(
        networks_url, NETWORKS_CONFIG_CACHE, NETWORKS_CONFIG_EXPIRE_SECONDS,
        NETWORKS_CONFIG_WAIT_CONNECTION_SECONDS, networks_retry_kwargs
    ).get(store=store)


def prepare_network_info(networks_config, interface, cidr):
    """Find network configuration by given IP in configuration file.

    Args:
        interface: an object of Interface class.
        reset_previous_address: a flag, that reset and setup backbone again.

    Returns:
        An object, with the most specific network record in file.
        If network will is not found, returns None.
        An object has a next structure:
        {
            'ip'  : ip,
            'cidr' : 'cidr notation of found ip',
            'info' : {
                "fastbone_vlans": {
                    "vlandID": {
                        "routes": "route_group_name"
                    },
                    "vlanID2": {
                        ...
                    },
                    ...
                ],
                "backbone_vlans": "route_group_name"
            },
            "switch_port": lldp_port,
            "project_id": project_id,
            "routes": {
                "route_group_name": {
                    "route_in_cidr_format": {
                        "gateway": "ip.ad.dr.es"
                        "mtu": 8910
                    },
                    ...
                }
            }
        }

        If more than one network will found,
            the smallest network (with most specific mask/prefix)
            will be returned.
    """

    def _check_project_id_network(info):
        """Check that found network required project_id."""
        return bool(info.get('mtn', 0))

    def _check_interface_is_auto(interface):
        """Check that address on interface has static_ip."""
        method = interface.option('method')
        return method == 'auto'

    info = find_network_info(cidr, networks_config)
    if info is None:
        raise NetconfigError('Could not find network info by given address')

    if _check_project_id_network(info.get('info', {})) and \
            _check_interface_is_auto(interface):
        info['method'] = 'project_id'

    logging.info('Found global backbone IP address %s', cidr)

    if dict_has_value(info, 'mtn', 1):
        info['project_id'] = get_project_id(interface)
        info['project_id_host_method'] = interface.option(
            'ya-netconfig-project-id-host-method'
        ) or 'hostname'

    host64_override_value = interface.numeric_option('ya-netconfig-host64-override')
    if host64_override_value is not None:
        dict_override_value(info, 'host64', host64_override_value)

    if dict_has_value(info, 'host64', 1):
        info['project_id'] = get_project_id(interface)
        # With default 'group_fwd_mask' we cannot get LLDP information from
        # bridge device. So we'll try to get it from first device in this
        # bridge.
        # TODO: Detect and use physical device.
        bridged_ifaces = get_bridge_interfaces(interface.name)
        if not bridged_ifaces:
            lldp_iface = interface.name
        else:
            lldp_iface = bridged_ifaces[0]

        logging.info('Using \'raw lldp\' scheme.')
        try:
            info['switch_port'] = get_lldp_port_index(lldp_iface)
        except Exception:
            logging.error('{}'.format(format_exception()))
            raise NetconfigError('Could not get raw lldp packet')
    return info


def find_network_info(cidr, networks_info):
    """Find network information by given IP address."""

    def patch_info(networks_info, patch):
        """Patch info from custom file."""
        if not os.path.isfile(patch):
            return networks_info
        try:
            pinfo = json.load(open(patch, 'r'))
            for section in ('route_groups', 'data'):
                if section in pinfo:
                    networks_info[section].update(pinfo[section])
            return networks_info
        except Exception as error:
            logging.error(
                'Failed to patch networks info by %s: %s', patch, error
            )
            return None

    try:
        searchip = ipaddr.IPNetwork(cidr).ip
        info = None
        for patch in glob(NETWORK_CONFIG_PATCHES_PATH + '/*.json'):
            networks_info = patch_info(networks_info, patch) or networks_info
        foundnet = None
        for network in networks_info['data']:
            net = ipaddr.IPNetwork(network)
            if searchip not in net:
                continue
            if foundnet is not None and net.compare_networks(foundnet) < 0:
                continue
            foundnet = net
            logging.debug('Found network is "%s"', network)
            info = {
                'ip': str(searchip),
                'cidr': cidr,
                'info': networks_info['data'][network],
                'routes': networks_info['route_groups']
            }

    except Exception as e:
        raise NetconfigError('Failed to get networks info: %s' % e)

    return info


def set_mtu(iface, mtu):
    """Sets up MTU on specified interface.

    Args:
        iface: a string, name of interface.
        mtu: an integer, maximum transfer unit


    Raise an exception in case of failure
    """

    mtu = int(mtu)  # Just to be sure
    for i in get_bridge_interfaces(iface):
        set_mtu(i, mtu)

    logging.debug('Setup MTU %s on %s', mtu, iface)

    prev_mtu = get_mtu(iface)

    if mtu == prev_mtu:
        logging.info('MTU %s already set on %s', mtu, iface)
        return

    s = socket.socket(socket.SOCK_DGRAM)
    try:
        ifr = struct.pack('<16si', iface, mtu) + '\x00' * 12
        ifs = fcntl.ioctl(s, SIOCSIFMTU, ifr)
        mtu = struct.unpack('<i', ifs[16:20])[0]

    except Exception as e:
        raise NetconfigMTUError('Failed to set mtu %s on %s: %s' % (mtu, iface, e))

    finally:
        s.close()

    logging.info('Successfully set MTU %s on %s', mtu, iface)
    logging.info('Waiting for iface %s up, after MTU set.', iface)

    is_iface_connected(iface, WAIT_CARRIER_AFTER_MTU_SECONDS)


def set_ipv6_address(iface, ip):
    """ Set IPv6 address via pyroute.
        It looks like the best argument 'ip' is netaddr.IPNetwork.
        Leave it as string for now.

        IPRoute can only add/delete ips, so we can't use 'replace' here.
        Because of that, we must check if address is already on iface.
    """
    logging.info('Settings IPv6 {} on {}'.format(ip, iface))

    ipaddr, prefixlen = ip.split('/')

    ip_netaddr_need_to_add = netaddr.IPNetwork(ip)

    try:
        with IPRoute() as ipr:
            link_index = ipr.link_lookup(ifname=iface)[0]
            found = False
            for addr_info in ipr.get_addr(index=link_index):
                cur_ipaddr = addr_info.get_attr('IFA_ADDRESS')
                cur_prefixlen = str(addr_info['prefixlen'])
                ip_netaddr_current = netaddr.IPNetwork('/'.join((cur_ipaddr, cur_prefixlen)))
                if ip_netaddr_need_to_add.value == ip_netaddr_current.value:
                    logging.info('{} is already on {}'.format(ip, iface))
                    found = True
                    break

            if not found:
                ipr.addr('add', link_index, address=ipaddr, prefixlen=int(prefixlen))
                logging.info('Added {} on {}'.format(ip, iface))

    except Exception as e:
        raise NetconfigError('Failed to set {} address on iface {}\n{}'.format(ip, iface, e))


def get_ipaddress_version(ip):
    """Return a type of IP address.

    Args:
        ip: a sting, checked IP address/network (may be in CIDR format).

    Returns:
        An integer:
            4 - if IPv4 address;
            6 - if IPv6 address;
            0 - otherwise.

    """
    try:
        return ipaddr.IPNetwork(ip).version
    except Exception:
        return 0


def add_route(route, gateway, mtu, dev, table=None):
    """Add route to routing table.

    Args:
        route: a string, route.
        gateway: a string, IP address of gateway for given route.
        mtu: an integer, default MTU for route.
        dev: a string, name of interface.

    Returns:
        A boolean, True if success.

    """

    # Debugging missing routes is very complicated task.
    # Let's fail right away
    route_type = get_ipaddress_version(gateway)  # It seems like it's better to check gateway as it is always IP.
    if route_type == 0:
        logging.info('Could not determine route type: {}'.format(route_type))
        return False
    mss = int(mtu) - 10 * route_type

    cmd = [
        'ip', '-' + str(route_type), 'route', 'replace', route,
        'via', gateway, 'dev', dev, 'mtu', str(mtu), 'advmss', str(mss)
    ]
    if table:
        cmd += ['table', str(table)]

    _, stderr, returncode = run_cmd(cmd)

    # may be a route already exists, so try to replace it.
    # for ipv6 command like 'ip route replace ...' didn't help.

    if returncode != 0 and del_route(route, gateway, dev, table):
        _, stderr, returncode = run_cmd(cmd)

    if returncode != 0:
        raise NetconfigError(
            'Fail to add route by command "%s", because: %s' %
            (' '.join(cmd), stderr)
        )

    logging.info(
        'Add route to %s by %s with MTU %s of %s table %s',
        route, gateway, mtu, dev, 'default' if table is None else table
    )


def del_route(route, gateway, iface, table=None):
    """Remove a route.

    Args:
        route: a string, destination network (route).
        gateway: a string, gateway for specified route.
        iface: a string, name of destination interface.

    Returns:
        A boolean, True if successfully remove.

    """

    # Tolerate route removal errors

    route_type = get_ipaddress_version(route)
    if route_type == 0:
        return False

    logging.info(
        'Remove route to %s by %s of %s table %s',
        route, gateway, iface, 'default' if table is None else table
    )
    cmd = [
        'ip', '-' + str(route_type), 'route', 'del', route,
        'via', gateway, 'dev', iface
    ]
    if table:
        cmd += ('table', str(table))

    stderr, returncode = run_cmd(cmd)[1:]

    if returncode != 0:
        logging.warn(
            'Fail to remove a route by command "%s", because: %s',
            ' '.join(cmd), stderr
        )
        return False

    return True


def get_gateway(gateway, network):
    """Return a gateway.

    Args:
        gateway: a string, gateway.
        network: a string, basic network in cidr format.
            This param need for calculate 'last' and 'first' address of network.

    Returns:
        A string, gateway address. Empty string if has an error.

    """
    try:
        if 'first' in gateway:
            return str(ipaddr.IPNetwork(network).network + 1)
        elif 'last' in gateway:
            return str(ipaddr.IPNetwork(network).broadcast - 1)
    except Exception as error:
        logging.warn(
            'Problem with detect right gateway "%s" for network "%s": %s',
            gateway, network, error
        )
        return ''

    return gateway


def convert_routes(routes, net, iface, table=None):
    """Add or remove routes by given list to interface.

    Args:
        routes: a dictionary, format should be:
            [ {
                'ip.ad.dr.es': {
                    'mtu': 8910,
                    'gateway': 'fe80::1'
                },
              },
              {
                ...
              }
            ].
        net: a string, cidr format network.
        iface: a string, given interface name.
        table: a string, name of route table.

    Returns:
        A list of dictionaries configured routes.

    Throws:
        May throw Exception, if given params aren't correct.

    """

    result = []
    for (route, route_config) in routes:
        mtu = route_config.get('mtu', MTU_ROUTES)
        gateway = get_gateway(route_config.get('gateway'), net)
        if not gateway:
            continue

        route_info = {
            'route': route, 'gw': gateway,
            'mtu': mtu, 'dev': iface
        }
        if table:
            route_info['table'] = table

        result.append(route_info)

    return result


def up_interface(iface, mtu=MTU_IFACE):
    """Trying to up interface.

    Args:
        iface: a string, name of interface.

    Returns:
        A boolean, True if interface up and carrier.

    """
    iface = str(iface)
    # already up?
    if is_iface_connected(iface, 1):
        return True
    # try to up via ifup
    run_cmd(['timeout', '30', 'ifup', iface])
    if is_iface_connected(iface, 1):
        return True
    # try to up manually
    # set_mtu(iface, mtu)
    disable_ipv6_privext(iface)
    # if command to up successfully executed, waiting for connection
    # _, _, ret = run_cmd(['ip', 'link', 'set', iface, 'up'])
    try:
        with IPRoute() as ipr:
            link_index = ipr.link_lookup(ifname=iface)[0]
            ipr.link('set', index=link_index, mtu=mtu, state='up')

        return is_iface_connected(iface)
    except Exception:
        return False


def create_vlan(base_iface, iface, vlan, mtu=MTU_IFACE):
    """Create vlan on specified interface."""

    if is_iface_created(iface):
        if is_iface_connected(iface, 1):
            logging.warn(
                'Interface %s already created and up for vlan %s on %s',
                iface, vlan, base_iface
            )
            return

        logging.warn("Iface {} is not connected - recreate it".format(iface))
        with IPRoute() as ipr:
            link_index = ipr.link_lookup(ifname=iface)[0]
            ipr.link('del', index=link_index)

    logging.info(
        'Create vlan interface %s on %s', iface, base_iface
    )

    _, stderr, error_code = run_cmd([
        'ip', 'link', 'add', 'link', base_iface, 'name', iface,
        'type', 'vlan', 'id', str(vlan)
    ])

    if error_code:
        raise NetconfigError("Failed to create vlan iface %s on %s" % (iface, base_iface))

    set_sysctl('net.ipv6.conf.' + iface.replace('.', '/') + '.accept_ra_defrtr', '0')

    set_mtu(iface, mtu)
    disable_ipv6_privext(iface)
    run_cmd(['ip', 'link', 'set', iface, 'up'])
    # Magic wait a few seconds,
    # While kernel up interface and setup default addresses
    time.sleep(2)
    if not is_iface_connected(iface):
        raise NetconfigError("Failed to ensure vlan iface %s connected" % iface)


GROUP_MAP = defaultdict(
    lambda: 0,
    {
        "backbone": 1,
        "bb": 1,
        "fastbone": 2,
        "fb": 2,
    }
)
REVERSE_GROUP_MAP = defaultdict(
    lambda: 'default',
    {
        1: "backbone",
        2: "fastbone",
    }
)


def set_interface_group(iface, group):
    """
    iface: string with interface name, like 'eth0'
    group: key from GROUP_MAP
    """
    try:
        with IPRoute() as ipr:
            link_index = ipr.link_lookup(ifname=iface)[0]
            ipr.link('set', index=link_index, group=GROUP_MAP[group])
        logging.info("Successfully set {} group {}".format(iface, group))
    except Exception as e:
        raise NetconfigError("Could not set interface group {} for {}: {}".format(group, iface, e))


def make_subinterface(base_iface, vlan, itype, opts, rewrite_mtu=None):
    """Trying up sub interface, or create interface.

    Args:
        base_iface: a main interface.
        vlan: a string, VLAN id for sub interface.
        itype: a string 'fb' or 'bb' expected.
        opts: an object of Inteface class, with specific options.

    Returns:
        A dictionary. Examples:
            - {'bb_iface': 'vlanXXX', 'created': False}
            - {'fb_iface': 'vlanXXX', 'created': True}

    """
    is_disabled = opts.option(
        'ya-netconfig-{0}-iface-vlan{1}-disable'.format(itype, vlan)
    )
    if is_disabled:
        logging.info(
            'Skip to configure vlan %s, because its disabled)', vlan
        )
        return {}

    iface = opts.option(
        'ya-netconfig-{0}-iface-vlan{1}'.format(itype, vlan)
    ) or 'vlan{0}'.format(vlan)

    # RTCNETWORK-70
    mtu = rewrite_mtu or opts.numeric_option('mtu') or MTU_IFACE

    if is_iface_created(iface) and up_interface(iface, mtu):
        return {'{0}_iface'.format(itype): iface, 'created': False}

    create_vlan(base_iface, iface, vlan, mtu)

    set_interface_group(iface, itype)

    return {
        '{0}_iface'.format(itype): iface,
        '{0}_base'.format(itype): base_iface,
        'created': True
    }


def get_host64_address(iface, port):
    """
    Gets MTN host64 address information.

    Args:
        iface: a string, name of interface.
        port: a string, name of connected switch port.

    Returns:
        A dictionary:
        {
            'mtn_global': ip:ad:dr:ess::1
            'mtn_local': ip:ad:dr:ess::1
            'mtn_prefix': pr:ef:ix::
        }
    """
    try:
        prefix = get_ra_prefix(iface)
        prefix = netaddr.IPNetwork(prefix)
    except netaddr.core.AddrFormatError as e:
        raise NetconfigError('Invalid RA prefix: {}'.format(e))

    if port is None:
        logging.debug('Skipping setup ip6 address to %s', iface)
        return None

    try:
        if prefix.prefixlen < 48 or prefix.prefixlen > 58:
            raise NetconfigError('Unsupported prefix length')

        # NOCDEV-3908
        port = int(port, 16)
        if prefix.prefixlen == 58 and port >= 32:
            port -= 32

        net = netaddr.IPNetwork('::')
        net.value = prefix.value + (port << 64)
        net.prefixlen = 64

        gaddr = netaddr.IPNetwork('::')
        gaddr.prefixlen = 64
        gaddr.value = (net.value | MTN_HOST64_HOSTID)
        logging.info("GADDR_VAL: {}".format(gaddr))

        laddr = netaddr.IPNetwork('fe80::/64')
        laddr.value += (int('a', 16) << 16) + port

    except Exception as error:
        raise NetconfigError('Error while calculate host64 ipaddr: %s', error)

    return {
        'mtn_global': str(gaddr), 'mtn_local': str(laddr),
        'mtn_prefix': str(prefix), 'mtn_net': str(net),
    }


def get_setup_schema(interface, info):
    """Return a setup schema

    Args:
        interface: an object of Interface class.
        info: network infromation object.

    Returns:
        An object - configures network information.

    """
    result = get_backbone_schema(interface, info)
    backbone_vlans = get_backbone_vlans_schema(interface, info)
    if backbone_vlans:
        result['backbone'] = backbone_vlans
    fastbone_vlans = get_fastbone_vlans_schema(interface, info)
    if fastbone_vlans:
        result['fastbone'] = fastbone_vlans

    return result


def get_backbone_schema(interface, net_info):
    """Return a backbone schema for given interface

    Args:
        interface: an object of Interface class.abs
        net_info: network information object.abs

    Returns:
        An object - configures network information.abs

    """
    result = {}
    routes = net_info['routes']
    info = net_info['info']

    # temporary fix for non existing project_id routes
    if info.get('backbone_routes', None) is None and \
            net_info.get('method', '') == 'project_id':
        info['backbone_routes'] = 'bb_default'

    result['bb_cidr'] = net_info['cidr']
    result['bb_ip'] = net_info['ip']
    result['bb_iface'] = interface.name
    result['bb_mtu'] = interface.numeric_option('mtu') or MTU_IFACE
    result['bb_routes'] = convert_routes(
        routes.get(info.get('backbone_routes', ''), {}).items(),
        net_info['cidr'], interface.name
    )

    # IPv4 schema, with static IPv6 address
    #   generate address like IPv6Prefix::IPv4
    if 'prefix6' in info:
        prefix6 = info['prefix6'].get('cidr')
        if prefix6 and info['prefix6'].get('method') == 'ipv4':
            ipv6 = prefix6.replace('/', result['bb_ip'] + '/')
            result['bb_ip_static'] = str(ipaddr.IPNetwork(ipv6))
            result['bb_routes'].extend(
                convert_routes(
                    routes.get('bb_default').items(),
                    prefix6, interface.name
                )
            )
    return result


def get_bridge_vlan_schema(iface, vlan, itype, interface):
    """Return a bridge vlan schema."""
    bridge_info = make_subinterface(iface, vlan, itype, interface)
    if not bridge_info:
        return {}
    bridge_info['bridge'] = True
    return bridge_info


def get_host64_vlan_schema(iface, vlan, itype, interface, net_info, route):
    """
    Return MTN host64 network schema.
    """

    port = net_info.get('switch_port', None)
    if not (port and route):
        return {}

    mtn_info = make_subinterface(iface, vlan, itype, interface)
    if not mtn_info:
        return {}

    mtn_ips = get_host64_address(mtn_info['{}_iface'.format(itype)],
                                 port)

    mtn_info.update(mtn_ips)

    aggr_routes = None
    if itype == 'bb':
        aggr_routes = aggregate_routes.BACKBONE_AGGR_ROUTES
    elif itype == 'fb':
        aggr_routes = aggregate_routes.FASTBONE_AGGR_ROUTES
    else:
        raise('Unknown itype: {}'.format(itype))

    mtn_info['routes'] = convert_routes(
        aggr_routes.items(),
        mtn_info['mtn_prefix'],
        mtn_info['{}_iface'.format(itype)],
        vlan
    )
    # For backbone mtn add default gw
    if itype == 'bb':
        mtn_info['routes'].append({
            'dev': mtn_info['bb_iface'],
            'gw': 'fe80::1',
            'mtu': MTU_DEFAULT_ROUTE,
            'route': '::/0',
            'table': vlan
        })
    mtn_info['host64'] = True
    return mtn_info


def get_backbone_vlans_schema(interface, net_info):
    """Return a backbone vlans schema.

    Args:
        interface: an object of Interface class.
        net_info: ntework information object.

    Returns:
        An object - configures network information.

    """
    result = {}
    info = net_info['info']
    bbvlans = info.get('backbone_vlans')

    if not bbvlans or interface.bool_option('ya-netconfig-bb-disable'):
        return result

    iface = interface.option('ya-netconfig-bb-iface') or interface.name
    mtu = interface.numeric_option('mtu') or MTU_IFACE

    if not up_interface(iface, mtu):
        raise NetconfigError('Base backbone interface %s not connected', iface)

    # Backbone VLANS may be two types:
    #  - bridge (only create interface)
    #  - mtn host64
    for vlan, opts in bbvlans.items():
        bb_info = {}
        if opts.get('bridge', None):
            bb_info = get_bridge_vlan_schema(iface, vlan, 'bb', interface)
        if opts.get('host64', None):
            bb_info = get_host64_vlan_schema(
                iface, vlan, 'bb', interface,
                net_info, opts.get('routes', None)
            )
        if bb_info:
            result[vlan] = bb_info
    return result


def get_fastbone_vlans_schema(interface, net_info):
    """Return a fastbone schema for given interface

    Args:
        interface: an object of Interface class.
        net_info: network information object.

    Returns:
        An object - configures network information.

    """
    result = {}
    info = net_info['info']
    fbvlans = info.get('fastbone_vlans')

    if not fbvlans or interface.bool_option('ya-netconfig-fb-disable'):
        return result

    iface = interface.option('ya-netconfig-fb-iface') or interface.name
    mtu = interface.numeric_option('mtu') or MTU_IFACE

    if not up_interface(iface, mtu):
        logging.warn('Base fastbone interfaces %s not connected', iface)
        return result

    # Fastbone VLANS types:
    #  - bridge (create)
    #  - mtn for host64 (/64 on host)
    #  - default IPv6
    #     - prefix6 + backbone IPv4 (cidr + method = ipv4)
    #     - prefix6 + project_id
    #     - RA
    for vlan, opts in fbvlans.items():
        fb_info = {}
        routes = opts.get('routes', None)

        # bridge
        if opts.get('bridge', None):
            fb_info = get_bridge_vlan_schema(iface, vlan, 'fb', interface)
            if fb_info:
                result[vlan] = fb_info
            continue

        # skip if no routes
        if not routes:
            continue

        # mtn for host64
        if opts.get('host64', None):
            fb_info = get_host64_vlan_schema(
                iface, vlan, 'fb', interface,
                net_info, routes
            )
            if fb_info:
                result[vlan] = fb_info
            continue

        # fastbone static -- IPv6_prefix::IPv4_Backbone
        # todo: make a dedicated function for this method
        if opts.get('prefix6', {}).get('method', None) == 'ipv4':
            global_ip = net_info.get('ip', None)
            prefix = opts.get('prefix6', {}).get('cidr', None)
            if not (global_ip and prefix):
                logging.warn(
                    'Fastbone vlan %s has method IPv4, '
                    'but no global ip and prefix6/cidr specified', vlan
                )
                continue

            if int(get_ipaddress_version(global_ip)) != 4:
                logging.warn(
                    'Fastbone vlan %s has method IPv4, '
                    'but main backbone address is not IPv4', vlan
                )
                continue

            fb_info = make_subinterface(iface, vlan, 'fb', interface)
            if not fb_info:
                continue

            fb_info['fb_ip'] = prefix.replace('/', global_ip + '/')
            fb_info['static'] = True
            fb_info['method'] = 'prefix6 + ipv4'
            fb_info['routes'] = convert_routes(
                aggregate_routes.FASTBONE_AGGR_ROUTES.items(),
                prefix, fb_info['fb_iface']
            )
            result[vlan] = fb_info
            continue

        fb_info = make_subinterface(iface, vlan, 'fb', interface)
        if not fb_info:
            continue

        # fastbone with project_id
        # todo: make a dedicated function for this method
        if opts.get('mtn', None):
            project_ip = get_projectid_address(
                fb_info['fb_iface'],
                net_info['project_id'],
                net_info['project_id_host_method']
            )

            if not project_ip:
                logging.warn(
                    'Fastbone interface %s has method ProjectID, '
                    'but did not get RA prefix', fb_info['fb_iface']
                )
                if fb_info.get('created', False):
                    result[vlan] = fb_info
                    continue

            fb_info['fb_ip'] = project_ip
            fb_info['static'] = True
            fb_info['method'] = 'project_id'
            fb_info['routes'] = convert_routes(
                aggregate_routes.FASTBONE_AGGR_ROUTES.items(),
                str(ipaddr.IPNetwork(project_ip).network),
                fb_info['fb_iface']
            )
            result[vlan] = fb_info
            continue

        # fastbone get address from RA
        enable_ipv6_ra(fb_info['fb_iface'])
        fb_ip_addr = get_ipv6_ra(fb_info['fb_iface'], False, False)
        if not fb_ip_addr:
            logging.warn(
                'Fastbone interface %s did not get address from RA',
                fb_info['fb_iface']
            )
            if fb_info.get('created', False):
                result[vlan] = fb_info
            continue

        fb_info['fb_ip'] = fb_ip_addr
        fb_info['method'] = 'eui-64'
        fb_info['routes'] = convert_routes(
            aggregate_routes.FASTBONE_AGGR_ROUTES.items(),
            str(ipaddr.IPNetwork(fb_ip_addr).network),
            fb_info['fb_iface']
        )

        result[vlan] = fb_info

    return result


def unsetup_vlans(vlans, itype):
    """Remove created vlans and routes"""
    for vlan, info in vlans:
        for route in info.get('routes', []):
            del_route(
                route['route'], route['gw'],
                route['dev'], route.get('table', None)
            )

        if info.get('host64', None):
            run_cmd(['ip', '-6', 'rule', 'del', 'table', str(vlan)])

        if info.get('created', False):
            iface = info['{}_iface'.format(itype)]
            logging.info('Remove interface %s', iface)
            run_cmd(['ip', 'link', 'set', 'dev', iface, 'down'])
            run_cmd(['ip', 'link', 'delete', iface, 'type', 'vlan'])


def add_pbr_rule(from_net, table, to_net=None):
    """Add ip -6 rule to specified table."""

    with IPRoute() as ipr:
        rules = ipr.get_rules(family=socket.AF_INET6)

    split_from_net = from_net.split('/')

    if to_net and table == 'main':
        int_main_table = 254  # according to /etc/iproute2/rt_tables, 254 == 'main'

        search = 'from {0} to {1} lookup {2}'.format(from_net, to_net, table)
        logging.info("Searching for \'{}\'".format(search))

        split_to_net = to_net.split('/')

        found_rule = False
        for rule in rules:
            if (rule.get_attrs('FRA_TABLE')[0] == int_main_table and
                    rule.get_attrs('FRA_SRC') == [split_from_net[0]] and
                    rule.get('src_len') == int(split_from_net[1]) and
                    rule.get_attrs('FRA_DST') == [split_to_net[0]] and
                    rule.get_attrs('FRA_PRIORITY')[0] == PBR_PRIORITY and
                    rule.get('dst_len') == int(split_to_net[1])):
                found_rule = True
                break

        if not found_rule:
            with IPRoute() as ipr:
                ipr.rule(
                    "add",
                    family=socket.AF_INET6,
                    src=from_net,
                    dst=to_net,
                    table=254,
                    priority=PBR_PRIORITY,
                )
    else:
        search = 'from {0} lookup {1}'.format(from_net, table)
        logging.info("Searching for \'{}\'".format(search))

        found_rule = False
        for rule in rules:
            if (rule.get_attrs('FRA_TABLE')[0] == int(table) and
                    rule.get_attrs('FRA_SRC') == [split_from_net[0]] and
                    rule.get_attrs('FRA_PRIORITY')[0] == PBR_PRIORITY+5 and
                    rule.get('src_len') == int(split_from_net[1])):
                found_rule = True
                break

        if not found_rule:
            with IPRoute() as ipr:
                ipr.rule(
                    "add",
                    family=socket.AF_INET6,
                    src=from_net,
                    table=int(table),
                    priority=PBR_PRIORITY+5,
                )


def ensure_ra_gateway(interface):
    """
    Check that RA default gateway is fe80::1

    Raises exception if nothing worked out
    """
    # check ra gateway only for (inet6 auto) options
    addrfam = interface.option('addrfam')
    method = interface.option('method')
    if addrfam != 'inet6' or method != 'auto':
        return

    # is check option is enabled
    if not interface.bool_option('ya-netconfig-ra-check-gw'):
        return

    if interface.bool_option('ya-netconfig-ra-disable-default-gw'):
        logging.info(
            'Option "ya-netconfig-ra-check-gw" enabled, '
            'but option "ya-netconfig-ra-disable-default-gw" '
            'enabled too. Skip this check.')
        return

    default_routes, _, _ = run_cmd(
        ['ip', '-6', 'route', 'list', 'dev', str(interface.name), 'default']
    )

    if 'default via fe80::1' in default_routes:
        return

    raise NetconfigError(
        'Check default RA gateway failed. '
        'You should ask a NOC to reconfigure default RA gw to fe80::1. '
        'Or disable ya-netconfig-ra-check-gw option. '
        'See https://st.yandex-team.ru/YTADMIN-7901 for more information.'
    )


def setup_schema(schema, previous_schema=None):
    """Setup network for specified schema.

    Args:
        schema: an object (dict) with needed to setup schema.
        previous_schema: an object (dict), with previous state.

    Returns:
        An object (dict), configured state.

    """

    def setup_vlan(vlan, info, iface):
        """Setup vlan address and routes.
        Args:
            vlan: a string vlan id
            info: an object (dict), vlan schema
            iface: a string, name of vlan interface
        Returns:
            A list of successfully added routes.

        """
        if info.get('bridge', False):
            if info.get('created', False):
                disable_ipv6_ra(iface, True)
            return None

        if info.get('host64', False):
            if info.get('created', False):
                disable_ipv6_ra(iface, True)
            set_ipv6_address(iface, info['mtn_global'])
            set_ipv6_address(iface, info['mtn_local'])
            set_sysctl('net.ipv6.conf.all.forwarding', '1')
            add_pbr_rule(info['mtn_prefix'], 'main', info['mtn_net'])
            add_pbr_rule(info['mtn_prefix'], str(vlan))

        routes = []
        for route in info.get('routes', []):
            add_route(
                route['route'], route['gw'], route['mtu'],
                route['dev'], route.get('table', None)
            )

            routes.append(route)

        return routes

    state = schema
    if previous_schema is None:
        previous_schema = {}

    bb_ip_static = state.get('bb_ip_static', '')
    prev_bb_ip_static = previous_schema.get('bb_ip_static', '')
    if bb_ip_static != prev_bb_ip_static:
        if prev_bb_ip_static:
            logging.info(
                'Remove previous global ip static %s', prev_bb_ip_static
            )
            run_cmd([
                'ip', '-6', 'addr', 'del', prev_bb_ip_static,
                'dev', state['bb_iface']
            ])

        if bb_ip_static:
            disable_ipv6_ra(state['bb_iface'], True)
            set_ipv6_address(state['bb_iface'], bb_ip_static)
            add_route('::/0', 'fe80::1', MTU_DEFAULT_ROUTE, state['bb_iface'])

    # append new routes (or replace existed)
    if 'bb_routes' in state:
        state['bb_routes'] = [
            r for r in state['bb_routes']
            if add_route(
                r['route'], r['gw'], r['mtu'],
                r['dev'], r.get('table', None)
            )
        ]

    # remove previous routes
    for route in previous_schema.get('bb_routes', []):
        if route not in state['bb_routes']:
            del_route(
                route['route'], route['gw'],
                route['dev'], route.get('table', None)
            )

    # fastbone vlans
    previous_fastbone_schema = previous_schema.get('fastbone', {})
    for vlan, info in state.get('fastbone', {}).items():
        previous_vlan_schema = previous_fastbone_schema.get(vlan, {})

        if info.get('static', False):
            if info.get('created', False):
                disable_ipv6_ra(info['fb_iface'], True)
            set_ipv6_address(info['fb_iface'], info['fb_ip'])

        routes = setup_vlan(vlan, info, info['fb_iface'])
        if routes is not None:
            state['fastbone'][vlan]['routes'] = routes

        # in a previous state ya-netconfig may create interface
        if previous_vlan_schema.get('created', False):
            state['fastbone'][vlan]['created'] = True

        previous_routes = previous_vlan_schema.get('routes', [])
        for route in previous_routes:
            if route not in state['fastbone'][vlan]['routes']:
                del_route(
                    route['route'], route['gw'],
                    route['dev'], route.get('table', None)
                )

    # remove obsolete vlans
    for vlan, info in previous_schema.get('fastbone', {}).items():
        if vlan not in state.get('fastbone', {}):
            unsetup_vlans({vlan: info}.items(), 'fb')

    # backbone vlans
    previouse_backbone_schema = previous_schema.get('backbone', {})
    for vlan, info in state.get('backbone', {}).items():
        previous_vlan_schema = previouse_backbone_schema.get(vlan, {})

        routes = setup_vlan(vlan, info, info['bb_iface'])
        if routes is not None:
            state['backbone'][vlan]['routes'] = routes

        # in a previous state ya-netconfig may create interface
        if previous_vlan_schema.get('created', False):
            state['backbone'][vlan]['created'] = True

        previous_routes = previous_vlan_schema.get('routes', [])
        for route in previous_routes:
            if route not in state['backbone'][vlan]['routes']:
                del_route(
                    route['route'], route['gw'],
                    route['dev'], route.get('table', None)
                )

    # remove obsolete vlans
    for vlan, info in previous_schema.get('backbone', {}).items():
        if vlan not in state.get('backbone', {}):
            unsetup_vlans({vlan: info}.items(), 'bb')

    return state


def write_netconfig_state(netconfig_state_descr):
    atomic_write(
        NETCONFIG_STATE_PATH,
        json.dumps(netconfig_state_descr)
    )


def load_netconfig_state(path=NETCONFIG_STATE_PATH):
    try:
        return json.loads(open(path).read())
    except Exception as e:
        logging.warning("Failed to load netconfig state at {}: {}".format(path, e))
        return []


class NetconfigState(object):
    """
    Example:
    {
        [
            {
                "interface": "eth0"
                "setup_stages": {
                    "bootstrap_net": {
                        finished: True
                        timestamp: <...>
                        status: "OK"
                        exception: null
                        traceback: null
                        message: "..."
                    }
                    "vlans_setup": {
                        ...
                    }
                    ...
                }
            },
        ...
        ]
    }
    """

    def __init__(self, state_descr, iface, stage, descr=""):
        self._stage = stage
        self._state_descr = state_descr
        self._descr = descr
        self._iface_state = None

        for section in self._state_descr:
            if section["interface"] == iface:
                self._iface_state = section
                break

        if not self._iface_state:
            self._iface_state = {"interface": iface}
            self._state_descr.append(self._iface_state)

        stages = self._iface_state.setdefault("setup_stages", {})
        self._stage_state = stages.setdefault(stage, {})

    def __enter__(self):
        self._stage_state["finished"] = False
        self._stage_state["timestamp"] = str(datetime.datetime.now())
        self._stage_state["status"] = "INPROGRESS"
        self._stage_state["exception"] = None
        self._stage_state["traceback"] = None
        self._stage_state["message"] = self._descr

        write_netconfig_state(self._state_descr)

    def __exit__(self, etype, value, tb):
        self._stage_state["finished"] = True
        self._stage_state["timestamp"] = str(datetime.datetime.now())

        ret = True

        if etype is None and value is None and tb is None:
            self._stage_state["status"] = "OK"
            logging.info("%s - OK", self._descr)
        else:
            if etype is NetconfigError:
                self._stage_state["status"] = "ERROR"
                self._stage_state['exception'] = "".join(traceback.format_exception_only(etype, value))
                self._stage_state['traceback'] = "".join(traceback.format_tb(tb))
                logging.exception("%s - FAILED", self._descr)
            elif etype is NetconfigSkip:
                del self._iface_state[self._stage]
                logging.debug("%s - SKIP: %s", self._descr, value)
                ret = False
            else:
                self._stage_state["status"] = "EXCEPTION"
                ret = False

        write_netconfig_state(self._state_descr)
        return ret


def set_state(iface, state):
    """Save interface state.

    Args:
        iface: a string, name of interface.
        state: an object, configuration state.

    """
    if not state:
        return None

    state["timestamp"] = str(datetime.datetime.now())

    fstate = '%s/%s.state' % (RUN_NETWORK_DIR, iface)
    try:
        json.dump(state, open(fstate, 'w'), indent=4, sort_keys=True)
    except Exception as error:
        logging.error('Failed to save interface state file %s: %s' % (fstate, error))


def get_state(iface):
    """Return a configured state for given interface.

    Args:
        iface: a string, name of interface.

    Returns:
        An object, configuration state.

    """
    fstate = '%s/%s.state' % (RUN_NETWORK_DIR, iface)
    try:
        if not os.path.exists(fstate):
            return {}
        return json.load(open(fstate, 'r'))
    except Exception as error:
        logging.error(
            'Failed to load state file %s: %s', fstate, error
        )
        return {}


def remove_state(iface):
    """Remove a configuration state for given interface.

    Args:
        iface: a string, name of interface.

    """
    fstate = '%s/%s.state' % (RUN_NETWORK_DIR, iface)
    try:
        if os.path.exists(fstate):
            os.remove(fstate)
    except Exception as error:
        logging.error('Failed to remove state file %s: %s', fstate, error)


def print_state(iface):
    """Pretty print interface state."""
    state = get_state(iface)
    if state:
        print json.dumps(state, indent=4, sort_keys=True)


def get_configured_interfaces(iface=None):
    """Return a list of configured interfaces."""
    try:
        ifaces = [
            i[:-6] for i in os.listdir(RUN_NETWORK_DIR)
            if i.endswith('.state')
        ]
    except Exception as error:
        logging.error('Failed to get configuration interfaces: %s', error)
        return []

    if iface:
        if iface in ifaces:
            return [iface]
        return []

    return ifaces


def set_base_neigh_reach_time(iface, val=DEF_BASE_REACH_TIME):
    """
    iface: string
    val: string
    """
    iface = iface.replace('.', '/')
    for i in ('ipv4', 'ipv6'):
        try:
            sysctl = 'net.'+i+'.neigh.'+iface+'.base_reachable_time_ms'
            if get_sysctl(sysctl) != val:
                set_sysctl(sysctl, val)
        except Exception as e:
            logging.warning(e)


def set_retrans_time_ms(iface, val=DEF_RETRANS_TIME):
    """
    iface: string
    val: string
    """
    iface = iface.replace('.', '/')
    # in salt we had only ipv6 version of this sysctl, so set only ipv6 version
    sysctl = 'net.ipv6.neigh.'+iface+'.retrans_time_ms'
    try:
        if get_sysctl(sysctl) != val:
            set_sysctl(sysctl, val)
    except Exception as e:
        logging.warning(e)


def check_is_configurable(iface, reload_mode):
    """
    Checks, whether we need to configure the interface
    Args:
        iface: an object, instance of Interface.
    """
    if iface is None:
        raise NetconfigSkip('Interface not specified')

    name = iface.name
    if get_configured_interfaces(name) and not reload_mode:
        raise NetconfigSkip('Skipping configure %s because already configured' % name)

    if name.startswith(DENY_IFACE_NAMES):
        raise NetconfigSkip('Interface %s has denied name' % name)

    # This is a workaround for early booting, without /sys system
    if not is_iface_connected(name):
        if reload_mode:
            raise NetconfigSkip('Interface %s is not connected' % iface)

        logging.debug('Interface %s seems like not connected, '
                      'but we will try to continue', name)

    if not is_interface_allowed(iface):
        raise NetconfigSkip('ya-netconfig setup for this iface is not allowed')


def setup_interface(iface, cidr, reload_mode=False):
    """
    Configure various settings exported by NOC for specified interface
    Args:
        iface: an object, instance of Interface
    """
    network_info = prepare_network_info(get_network_info(iface), iface, cidr)
    if network_info is None:
        raise NetconfigError('Network information not found')

    logging.info('Acquired network info: %s', pprint.pformat(network_info))

    try:
        switch_port = network_info['switch_port']
        switch_port_file = os.path.join(RUN_NETWORK_DIR, '.'.join((iface.name, 'switch_port')))
        logging.info('Writing switch_port {} to file {}'.format(switch_port, switch_port_file))
        atomic_write(switch_port_file, switch_port)
    except KeyError:
        logging.info('Section of network info has no \'host64\', skip writing switch_port info to state')

    mtu = iface.numeric_option('mtu') or MTU_IFACE
    set_mtu(iface.name, mtu)

    if not reload_mode:
        ensure_ra_gateway(iface)
        set_interface_group(iface.name, 'backbone')

    schema = get_setup_schema(iface, network_info)
    if not schema:
        raise NetconfigError('No setup schema obtained')

    state = setup_schema(schema, get_state(iface.name) if reload_mode else None)
    if state.get("bb_routes", '') or \
            state.get('fastbone', '') or \
            state.get('backbone', ''):
        set_state(iface.name, state)


def configure_interface(iface, reload_mode):
    """
    Configure specified interface.
    Args:
        iface: an object, instance of Interface.
    Returns:
        None
    """
    netconfig_state = load_netconfig_state()
    if netconfig_state:
        netconfig_state = [
            section for section in netconfig_state
            if section["interface"] != iface
        ]

    cidr = None
    try:
        check_is_configurable(iface, reload_mode)

        base_reach_time = iface.option('ya-netconfig-base-reach-time') or DEF_BASE_REACH_TIME
        set_base_neigh_reach_time(iface.name, val=base_reach_time)

        retrans_time_ms = iface.option('ya-netconfig-retrans-time') or DEF_RETRANS_TIME
        set_retrans_time_ms(iface.name, val=retrans_time_ms)

        with NetconfigState(
                netconfig_state, iface.name, "bootstrap_net",
                "Bootstrap network interface %s" % iface
        ):
            cidr = get_backbone_address(iface, not reload_mode)
            if not cidr:
                raise NetconfigError('Cannot obtain corresponding backbone address')

    except NetconfigSkip:
        return

    if not cidr:
        return

    with NetconfigState(
            netconfig_state, iface.name, "vlans_routes",
            "Setup vlans and routes on %s" % iface
    ):
        setup_interface(iface, cidr, reload_mode)


def deconfigure_interface(iface):
    """Clear configured interface.

    Args:
        iface: a string, name of configured interface.

    Returns:
        A boolean, True if success.

    """

    state = get_state(iface)
    if not state:
        logging.error('Iface %s is not configured' % iface)
        return False

    # Remove iface from netconfig state file
    netconfig_state = load_netconfig_state()
    if netconfig_state:
        netconfig_state = [
            section for section in netconfig_state
            if section["interface"] != iface
        ]
        write_netconfig_state(netconfig_state)

    try:
        if 'fastbone' in state:
            unsetup_vlans(state['fastbone'].items(), 'fb')

        if 'backbone' in state:
            unsetup_vlans(state['backbone'].items(), 'bb')

        for route in state.get('bb_routes', []):
            # don't remove default route
            if route['route'].startswith(('::0', '0.0.0.0', 'default')):
                continue
            del_route(route['route'], route['gw'], route['dev'])

        if 'bb_ip_static' in state:
            run_cmd([
                'ip', 'addr', 'del', state['bb_ip_static'],
                'dev', state['bb_iface']
            ])

        return True
    except Exception as error:
        logging.exception(
            'Error while deconfigure interface %s: %s',
            iface, error
        )

    return False


def run_start(iface):
    """Start specified interface.

    Args:
        iface: a string, name of interface.
    """

    l = logging.getLogger('start')
    if iface and os.getenv('IFACE') == iface:
        # ifupdown mode
        l.info('Configure %s', iface)

        iface_config = Interface.from_env(iface)
        timeout = iface_config.numeric_option('ya-netconfig-ifupdown-config-timeout') or IFUPDOWN_CONFIG_TIMEOUT
        deadline = time.time() + timeout

        netconfig_state = load_netconfig_state()
        if netconfig_state:
            netconfig_state = [
                section for section in netconfig_state
                if section["interface"] != iface
            ]

        backbone_cidr = None

        while time.time() < deadline and not backbone_cidr:
            try:
                check_is_configurable(iface_config, False)

                base_reach_time = iface_config.option('ya-netconfig-base-reach-time') or DEF_BASE_REACH_TIME
                set_base_neigh_reach_time(iface, val=base_reach_time)

                retrans_time_ms = iface_config.option('ya-netconfig-retrans-time') or DEF_RETRANS_TIME
                set_retrans_time_ms(iface, val=retrans_time_ms)

                with NetconfigState(
                        netconfig_state, iface, "bootstrap_net",
                        "Bootstrap network interface %s" % iface
                ):
                    backbone_cidr = get_backbone_address(iface_config, True)
                    if not backbone_cidr:
                        raise NetconfigError('Cannot obtain corresponding backbone address')
            except NetconfigSkip:
                return

            time.sleep(2)

        while time.time() < deadline and backbone_cidr:
            with NetconfigState(
                    netconfig_state, iface, "vlans_routes",
                    "Setup vlans and routes on %s" % iface
            ):
                setup_interface(iface_config, backbone_cidr)
                return

            time.sleep(2)

    # Oneshot behaviour for direct invocation
    elif iface:
        l.info('Start for %s', iface)

        # Executed directly for specific interface.
        interfaces = DebianInterfaces()
        for i in interfaces.get_iface_definitions(iface):
            configure_interface(i, False)

    else:
        l.info('Start for all ifaces')

        # Executed directly for all interfaces.
        interfaces = DebianInterfaces()
        for iname, ifaces in interfaces.ifaces.items():
            for i in ifaces:
                configure_interface(i, False)


def run_stop(iface):
    """Deconfigure network interface.

    Args:
        iface: a sting, name of configured interface or a None (all).

    """
    ifaces = get_configured_interfaces(iface)
    if not ifaces and not iface:
        logging.info('There are no configured interfaces')

    for i in ifaces:
        if deconfigure_interface(i):
            remove_state(i)


def run_reload(iface):
    """Reload configuration.
    It's simplify remake routes on interfaces (and make new fastbone vlans).

    Args:
        iface: a string, name of interface.
    """
    ifaces = get_configured_interfaces(iface)
    if not ifaces:
        if iface:
            logging.info('Interface %s not configured', iface)
        else:
            logging.info('There are no configured interfaces (use "start")')

        return

    interfaces = DebianInterfaces()
    for iname in ifaces:
        for i in interfaces.get_iface_definitions(iname):
            configure_interface(i, True)


def run_restart(iface):
    """
    Fully restart specified interface.

    Args:
        iface: a string, name of configured interface or a None.
    """
    ifaces = get_configured_interfaces(iface)
    if not ifaces and iface:
        logging.info('Interface %s not configured', iface)
        return

    for i in ifaces:
        run_stop(i)

    run_start(iface or None)


def run_status(iface):
    """Print line by line configured interfaces list."""
    for i in get_configured_interfaces(iface):
        print i


def run_state(iface):
    """Print configured state."""
    for i in get_configured_interfaces(iface):
        print_state(i)


def run_fetch():
    """Fetch networks.json file (with cache updated).
    Exit with code 0 if success, 1 - otherwise.

    """
    networks_config_url = os.getenv('IF_YA_NETCONFIG_NETWORKS_URL',
                                    NETWORKS_CONFIG_URL)
    networks_retry_kwargs = {
        "stop_max_attempt_number": NETWORKS_CONFIG_TRIES_TO_DOWNLOAD,
        "wait_fixed": NETWORKS_CONFIG_WAIT_BETWEEN_TRIES_SECONDS
    }
    networks_config_updated = RemoteJsonConfig(
        networks_config_url, NETWORKS_CONFIG_CACHE,
        NETWORKS_CONFIG_EXPIRE_SECONDS,
        NETWORKS_CONFIG_WAIT_CONNECTION_SECONDS,
        networks_retry_kwargs
    ).update_cache()

    if not networks_config_updated:
        sys.exit(1)
    sys.exit(0)


def main():
    action = ''.join(sys.argv[1:2]) or os.getenv('MODE', '')
    action = action.strip().lower()
    setup_logging(get_log_level_for(action))
    iface = ''.join(sys.argv[2:3]) or os.getenv('IFACE', None)

    make_dirs(RUN_NETWORK_DIR)

    # FIXME: leave "all-success" approach for now
    if action == "start":
        run_start(iface)
    elif action == "stop":
        run_stop(iface)
    elif action == "restart":
        run_restart(iface)
    elif action == "reload":
        run_reload(iface)
    elif action == "status":
        run_status(iface)
    elif action == "fetch":
        run_fetch()
    elif action == "state":
        run_state(iface)
    else:
        usage()
    logging.info('\'{}\' action is finished correctly'.format(action))

    return 0


if __name__ == "__main__":
    t = Timer(start=True)
    try:
        raise SystemExit(main())
    finally:
        t.stop()
        logging.info("ya-netconfig execution took: %s", t.format_elapsed())
