from __future__ import unicode_literals

import re
import sys
import copy
import itertools
import hashlib
import datetime
import traceback
import ipaddr

import master
from infra.netconfig.lib.hostutil import get_uptime


def calculate_network_config_hash(orig_network_config):
    """
    Hash exported NOC configuration

    Args:
        orig_network_config: map, exported by NOC
    Returns: (hash str, unhashed_fields set)
    """

    def store_leftovers(value_dict, fields):
        for k in value_dict.keys():
            fields.add(k)

    def hash_fields(ctx, fields, record):
        for key, default in fields.items():
            value = record.pop(key, default)
            if isinstance(value, unicode):
                ctx.update(value.encode('utf-8'))
            else:
                ctx.update(str(value))

    h = hashlib.sha256()
    network_config = copy.deepcopy(orig_network_config)
    unhashed_fields = {}
    route_groups = network_config.pop("route_groups", {})
    for group in sorted(route_groups.keys()):
        h.update(group)
        routes = route_groups.get(group, {})
        for prefix in sorted(routes.keys()):
            h.update(prefix)
            params = routes.pop(prefix, {})
            h.update(params.pop("gateway", "INVALID"))
            h.update(str(params.pop("mtu", 42)))
            store_leftovers(unhashed_fields, params)

    h.update("data")
    subnets = network_config.pop("data", None)
    for prefix in sorted(subnets.keys()):
        h.update(prefix)
        settings = subnets.get(prefix, {})
        fields = {
            "vlan": 7255,
            "datacenter_id": 7255,
            "datacenter_name": "INVALID",
            "l2_domain_id": 42,
            "l2_domain_name": "INVALID",
            "backbone_routes": "INVALID",
            "mtn": 0,
            "host64": 0,
        }
        hash_fields(h, fields, settings)

        fastbones = settings.pop("fastbone_vlans", {})
        for vlan in sorted(fastbones.keys()):
            h.update(vlan)
            fields = {
                "routes": "INVALID",
                "mtn": 0,
                "host64": 0
            }
            params = fastbones.get(vlan, {})
            hash_fields(h, fields, params)
            store_leftovers(unhashed_fields, params)

        backbones = settings.pop("backbone_vlans", {})
        for vlan in sorted(backbones.keys()):
            h.update(vlan)
            fields = {
                "routes": "INVALID",
                "mtn": 0,
                "host64": 0
            }
            params = backbones.get(vlan, {})
            hash_fields(h, fields, params)
            store_leftovers(unhashed_fields, params)

        store_leftovers(unhashed_fields, settings)
    store_leftovers(unhashed_fields, network_config)
    return (h.hexdigest(), unhashed_fields)


def get_bb_cidr_from_state(iface_name, iface_state):
    bb_cidr = iface_state.get("bb_cidr", None)
    if not bb_cidr:
        bb_iface = iface_state.get("bb_iface", None)
        if not bb_iface:
            raise master.NetconfigError("Cannot find bb_iface key in %s state" % iface_name)

        bb_ip = iface_state.get("bb_ip", None)
        if not bb_ip:
            raise master.NetconfigError('No backbone ip configured for %s' % iface_name)

        stdout, _, ret = master.run_cmd(
            ['ip', '-6', '-o', 'addr', 'show', 'dev', iface_name, 'scope', 'global']
        )
        if ret:
            raise master.NetconfigError("Cannot retrieve bb iface %s cidr" % iface_name)

        m = re.compile("^%s/[0-9]{1,3}$" % bb_ip)
        for col in stdout.split():
            if m.match(col):
                bb_cidr = col

        if not bb_cidr:
            raise master.NetconfigError("Cannot match bb iface %s cidr" % iface_name)
    return bb_cidr


def verify_state(iface_name, iface_state):
    def ensure_link(name, addr_str, mtu):
        # canonicalization
        addr = str(ipaddr.IPNetwork(addr_str).ip)
        stdout, _, ret = master.run_cmd(
            ['ip', '-6', '-o', 'link', 'show', 'dev', name]
        )
        if ret or not stdout:
            raise master.NetconfigError("Iface %s not found" % name)
        if not "mtu %s" % mtu in stdout:
            raise master.NetconfigError("Iface %s config does not match real mtu %s" % (name, mtu))

        stdout, _, ret = master.run_cmd(
            ['ip', '-6', '-o', 'addr', 'show', 'dev', name, 'scope', 'global']
        )
        if ret or not stdout:
            raise master.NetconfigError("No global addresses on iface %s" % name)
        if addr not in stdout:
            raise master.NetconfigError("Ip %s not set on iface %s" % (addr, name))

    def ensure_routes(name, routes):
        stdout, _, ret = master.run_cmd(
            ['ip', '-o', '-d', '-6', 'route', 'show', 'table', 'all', 'dev', name]
        )
        if ret or not stdout:
            raise master.NetconfigError("Cannot get routes for iface %s" % name)

        routes_txt = stdout.splitlines()
        for route in routes:
            prefix = route["route"]
            table = route.get("table", None)
            match = False
            if prefix == "::/0":
                prefix = "default"

            for line in routes_txt:
                if prefix in line:
                    if (
                        route["dev"] == name and
                        "mtu %s" % route["mtu"] in line and
                        "via %s" % route["gw"] in line and
                        (not table or "table %s" % table in line)
                    ):
                        match = True
                        break
            if not match:
                raise master.NetconfigError("route %s not found" % route)

    def verify_bb():
        bb_ip = iface_state["bb_ip"]
        bb_mtu = iface_state["bb_mtu"]
        bb_iface = iface_state["bb_iface"]
        ensure_link(bb_iface, bb_ip, bb_mtu)
        # FIXME: bb_routes is empty for now

    def verify_bb_vlans(mtu):
        for (vlan, config) in iface_state.get("backbone", {}).items():
            bb_iface = config["bb_iface"]
            host64 = config.get("host64", False)
            if host64:
                mtn_global = config["mtn_global"]
                ensure_link(bb_iface, mtn_global, mtu)
            else:
                bb_ip = config["bb_ip"]
                ensure_link(bb_iface, bb_ip, mtu)
            ensure_routes(bb_iface, config["routes"])

    def verify_fb_vlans(mtu):
        for (vlan, config) in iface_state.get("fastbone", {}).items():
            bb_iface = config["fb_iface"]
            host64 = config.get("host64", False)
            if host64:
                mtn_global = config["mtn_global"]
                ensure_link(bb_iface, mtn_global, mtu)
            else:
                bb_ip = config["fb_ip"]
                ensure_link(bb_iface, bb_ip, mtu)
            ensure_routes(bb_iface, config["routes"])

    tstr = iface_state.get("timestamp", None)
    if tstr:
        try:
            ts = datetime.datetime.strptime(tstr.split(".")[0], "%Y-%m-%d %H:%M:%S")
            if (datetime.datetime.now() - ts).total_seconds() > get_uptime():
                raise master.NetconfigError("Outdated %s configure state" % iface_name)
        except ValueError:
            pass
    try:
        verify_bb()
        verify_bb_vlans(iface_state["bb_mtu"])
        verify_fb_vlans(iface_state["bb_mtu"])
    except KeyError as e:
        raise master.NetconfigError("Missed state key: %s" % e)


class RoutesDiff(object):
    def __init__(self, netconfig_path=None, interfaces_path=None):
        self.new_routes = {}
        self.deleted_routes = {}
        self.changed_routes = {}
        self.new_vlans = []
        self.deleted_vlans = []
        self.orig_hash = None
        self.iface_name = None
        self.cidr = None
        self.route_in_table = {'bb': False}
        self.iface_state = None
        self.build_diff(netconfig_path, interfaces_path)

    def build_diff(self, netconfig_path=None, interfaces_path=None):
        kwargs = {}
        if netconfig_path:
            kwargs["path"] = netconfig_path

        netconfig = master.load_netconfig_state(**kwargs)

        # FIXME: There is one unique iface in nearly all cases, so let's just bail out for now
        for iface_name in master.get_configured_interfaces():
            if netconfig:
                for section in netconfig:
                    if section.get("interface", "") == iface_name:
                        found = True
                        break

                if not found:
                    raise master.NetconfigError(
                        'Discrepancy between state and configured iface %s' % iface_name
                    )

            self.iface_state = master.get_state(iface_name)
            if not self.iface_state:
                raise master.NetconfigError('No configured %s state file' % iface_name)

            verify_state(iface_name, self.iface_state)
            self.cidr = get_bb_cidr_from_state(iface_name, self.iface_state)
            bb_routes = self.iface_state.get("bb_routes", [])
            iface_obj = None
            try:
                iface_obj = master.DebianInterfaces(
                    path=interfaces_path
                ).get_iface_definitions(iface_name)[0]
                self.iface_name = iface_name

                network_config = master.get_network_info(iface_obj)
                self.orig_hash, _ = calculate_network_config_hash(network_config)
                network_info = master.prepare_network_info(network_config, iface_obj, self.cidr)
            except Exception:
                raise master.NetconfigError(
                    'Cannot get network info for %s (%s): %s' % (
                        iface_name, self.cidr,
                        "\n".join(traceback.format_exception(*sys.exc_info()))
                    )
                )

            # Parse NOC export
            self.route_in_table = {'bb': False}
            vlan_routes = {
                "bb": copy.deepcopy(
                    network_info.get("routes", {}).get(
                        network_info.get("info", {}).get("backbone_routes", ""),
                        {}
                    )
                )}

            for vlan, descriptor in network_info.get("info", {}).get("backbone_vlans", {}).items():
                vlan_routes[vlan] = copy.deepcopy(
                    network_info.get("routes", {}).get(
                        descriptor.get("routes", {}),
                        {}
                    )
                )
                self.route_in_table[vlan] = True if descriptor.get("host64", 0) > 0 else False

            for vlan, descriptor in network_info.get("info", {}).get("fastbone_vlans", {}).items():
                vlan_routes[vlan] = copy.deepcopy(
                    network_info.get("routes", {}).get(
                        descriptor.get("routes", {}),
                        {}
                    )
                )
                self.route_in_table[vlan] = True if descriptor.get("host64", 0) > 0 else False

            # Traverse netconfig configured state
            for vlan, vlan_section in itertools.chain(
                    self.iface_state.get("backbone", {}).items(),
                    self.iface_state.get("fastbone", {}).items(),
                    [("bb", {"routes": bb_routes})]
            ):
                if vlan not in vlan_routes:
                    self.deleted_vlans.append(vlan)
                    continue

                for iface_route in vlan_section.get("routes", []):
                    iface_prefix = iface_route.get("route", "")
                    if not iface_prefix:
                        continue

                    noc_route = vlan_routes.get(vlan, {}).get(iface_prefix, None)

                    if (
                        noc_route and
                        (not self.route_in_table[vlan] or iface_route.get("table", None) == vlan)
                    ):
                        noc_gw = master.get_gateway(noc_route.get("gateway", ""), self.cidr)
                        if (
                                noc_route.get("mtu", 0) == iface_route.get("mtu", 0) and
                                noc_gw == iface_route.get("gw", "")
                        ):
                            # Match
                            vlan_routes.get(vlan, {}).pop(iface_prefix, None)

                        else:
                            # Other gw/mtu
                            self.changed_routes.setdefault(vlan, []).append(
                                {
                                    "prefix": iface_prefix,
                                    "mtu": noc_route.get("mtu", 0),
                                    "gateway": noc_gw,
                                    "table": vlan if self.route_in_table[vlan] else None
                                }
                            )

                    elif iface_prefix != "default" and iface_prefix != "::/0":
                        self.deleted_routes.setdefault(vlan, []).append(
                            {
                                "prefix": iface_prefix,
                                "mtu": iface_route.get("mtu", 0),
                                "gateway": iface_route.get("gw", ""),
                                "table": iface_route.get("table", None)
                            }
                        )

            for vlan, routes in vlan_routes.items():
                if (
                    vlan != "bb" and
                    vlan not in self.iface_state.get("backbone", {}) and
                    vlan not in self.iface_state.get("fastbone", {})
                ):
                    self.new_vlans.append(vlan)

                for prefix, route in routes.items():
                    self.new_routes.setdefault(vlan, []).append(
                        {
                            "prefix": prefix,
                            "mtu": route.get("mtu", 0),
                            "gateway": route.get("gateway", ""),
                            "table": vlan if self.route_in_table[vlan] else None
                        }
                    )

            # FIXME - ya-netconfig does not store main bb routes
            self.new_routes.pop("bb", None)

    def add_new_routes(self, iface_state_routes):
        for (vlan, routes) in self.new_routes.items():
            for route in routes:
                table = route.get("table", None)
                master.add_route(
                    route["prefix"], route["gateway"],
                    route["mtu"], "vlan%s" % vlan, table
                )
                state_route = {
                    "route": route["prefix"],
                    "gw": route["gateway"],
                    "mtu": route["mtu"],
                    "dev": "vlan%s" % vlan
                }
                if table:
                    state_route["table"] = table

                iface_state_routes[vlan].append(state_route)

    def mod_changed_routes(self, iface_state_routes):
        for (vlan, routes) in self.changed_routes.items():
            for route in routes:
                master.add_route(
                    route["prefix"], route["gateway"],
                    route["mtu"], "vlan%s" % vlan, route.get("table", None)
                )
                for state_route in iface_state_routes[vlan]:
                    if state_route["route"] == route["prefix"]:
                        state_route["gw"] = route["gateway"]
                        state_route["mtu"] = route["mtu"]
                        break

    def remove_deleted_routes(self, iface_state_routes):
        for (vlan, routes) in self.deleted_routes.items():
            for route in routes:
                master.del_route(
                    route["prefix"], route["gateway"],
                    "vlan%s" % vlan, route.get("table", None)
                )
                for i in xrange(len(iface_state_routes[vlan])):
                    if route["prefix"] == iface_state_routes[vlan][i]["route"]:
                        iface_state_routes[vlan].pop(i)
                        break

    def apply(self):
        # Extract configured sections data first to reflect new state
        iface_state_routes = {}
        for vlan in itertools.chain(
            self.new_routes.keys(),
            self.deleted_routes.keys(),
            self.changed_routes.keys()
        ):
            if vlan in iface_state_routes:
                continue

            try:
                master.get_mac_address("vlan%s" % vlan)
            except master.NetconfigError:
                raise master.NetconfigError("Cannot find iface vlan%s, something may be broken or ya-netconfig-bb/fb-iface-vlan<num> option set?" % vlan)

            bb_section = self.iface_state.get('backbone', {}).get(vlan, {}).get("routes", [])
            if bb_section:
                iface_state_routes[vlan] = bb_section
                continue

            fb_section = self.iface_state.get('fastbone', {}).get(vlan, {}).get("routes", [])
            if fb_section:
                iface_state_routes[vlan] = fb_section
                continue
            raise master.NetconfigError("No routes section for %s vlan" % vlan)

        # Proceed to action
        netconfig_state = master.load_netconfig_state()
        with master.NetconfigState(
            netconfig_state, self.iface_name, "apply_routes",
            "Apply diff with current route state on %s" % self.iface_name
        ):

            self.add_new_routes(iface_state_routes)
            self.mod_changed_routes(iface_state_routes)
            self.remove_deleted_routes(iface_state_routes)
            master.set_state(self.iface_name, self.iface_state)
