import netaddr
import socket
from pyroute2 import IPRoute

from infra.netconfig.lib import master
from infra.netconfig.lib import fixutil
from infra.netconfig.proto import fixutil_pb2

MTN_VLANS = (688, 788)  # Hardcode for now in the name of Simplicity
LINK_LOCAL_SCOPE = 253


def make_iface_proto(iface):

    iface_name = iface.get_attr('IFLA_IFNAME')
    if not master.is_iface_connected(iface_name, 0):
        return None

    interface_proto = fixutil_pb2.InterfaceState()
    interface_proto.name = iface_name
    interface_proto.mtu = iface.get_attr('IFLA_MTU')
    interface_proto.group = master.REVERSE_GROUP_MAP[iface.get_attr('IFLA_GROUP')]

    kind = fixutil.get_interface_kind(interface_proto.name)
    if kind:
        interface_proto.kind = kind

    vlan_id = fixutil.get_vlan_id(interface_proto.name)
    if vlan_id:
        interface_proto.vlan_id = vlan_id

    with IPRoute() as ipr:
        # It is better to separate tries for ll address, and global address get
        try:
            ll_addr = ipr.get_addr(index=iface.get('index'), scope=LINK_LOCAL_SCOPE)[0]
            interface_proto.ipv6_link_local = ll_addr.get_attr('IFA_ADDRESS')
        except Exception:
            pass

        try:
            gl_addr = ipr.get_addr(index=iface.get('index'), scope=0, family=socket.AF_INET6)[0]
            gl_prefixlen = gl_addr.get('prefixlen')

            gl_cidr = netaddr.IPNetwork(gl_addr.get_attr('IFA_ADDRESS'))
            gl_addr = netaddr.IPAddress(gl_cidr)
            interface_proto.ipv6_address = str(gl_addr)
            gl_cidr.prefixlen = gl_prefixlen
            interface_proto.cidr = str(gl_cidr)

            if interface_proto.vlan_id in MTN_VLANS:  # bad hardcode
                mtn_network = netaddr.IPNetwork('::/64')
                mtn_network_value = gl_addr.value >> 64 << 64
                mtn_network.value = mtn_network_value
                interface_proto.network = str(mtn_network)
        except Exception:
            pass

    return interface_proto
