import porto
import time
import sys
import netaddr
import socket
import random
import logging
from pyroute2 import IPRoute

from infra.netconfig.lib import master as netconfig

MASTER_CONTAINER_NAME = 'pinger_master'
CLIENT_CONTAINER_NAME = 'pinger_client'
# MTN_BACKBONE_PREFIX = netaddr.IPNetwork("2a02:6b8:c00::/40")
MTN_BACKBONE_PREFIX = netaddr.IPNetwork(netconfig.MTN_BACKBONE_PREFIX)

logging.basicConfig(stream=sys.stdout, level=logging.DEBUG)


def form_ip_for_container():
    with IPRoute() as ipr:
        link_id = ipr.link_lookup(ifname='vlan688')[0]
        addrs = ipr.get_addr(index=link_id, scope=0, family=socket.AF_INET6)
        for a in addrs:
            ip = netaddr.IPAddress(a.get_attr('IFA_ADDRESS'))
            if ip in MTN_BACKBONE_PREFIX:
                ip.value = ip.value >> 64
                ip.value = ip.value << 64 | random.randrange(0x1, 0xff)
                result = netaddr.IPNetwork('::/0')
                result.value = ip.value
                result.prefixlen = 128
                return result


def run_client(c_ip, sleep_time=16, wait_for_dead=False):
    c = porto.Connection()

    try:
        r = c.Create(CLIENT_CONTAINER_NAME)
        r.SetProperty('command', '/bin/sleep {}'.format(sleep_time))
        r.SetProperty('net', 'L3 veth')
        r.SetProperty('ip', 'veth {}'.format(c_ip))

        r.Start()
        logging.info('{} started on {} for {} seconds'.format(CLIENT_CONTAINER_NAME, c_ip, sleep_time))
        if not wait_for_dead:
            return r
    except Exception as e:
        print('Could not start container:\n{}'.format(e))
        c.Destroy(CLIENT_CONTAINER_NAME)
        sys.exit(1)

    if wait_for_dead:
        state = c.GetProperty(CLIENT_CONTAINER_NAME, 'state')
        while state != 'dead':
            state = c.GetProperty(CLIENT_CONTAINER_NAME, 'state')
            time.sleep(1)

        c.Destroy(CLIENT_CONTAINER_NAME)


def get_dom0_ip():
    ip = None
    with IPRoute() as ipr:
        try:
            for iface in ('eth0', 'eth1', 'eth2'):
                link_id = ipr.link_lookup(ifname=iface)[0]
                addrs = ipr.get_addr(index=link_id, scope=0, family=socket.AF_INET6)[0]
                ip = netaddr.IPAddress(addrs.get_attr('IFA_ADDRESS'))
        except:
            pass
    return ip


def run_master(remote_container_ip, local_client_sleep_time=16):
    local_client_ip = str(form_ip_for_container())
    client = run_client(local_client_ip, local_client_sleep_time)

    m = porto.Connection()
    master_ip = str(form_ip_for_container())

    dom0_ip = get_dom0_ip()

    r = m.Create(MASTER_CONTAINER_NAME)
    try:
        r.SetProperty('command', '/bin/sh -c "for i in {} {} {} ; do /bin/ping6 -c 1 \$i ; done"'.format(
            remote_container_ip,
            dom0_ip,
            local_client_ip.split('/')[0],
        ))
        r.SetProperty('net', 'L3 veth')
        r.SetProperty('ip', 'veth {}'.format(master_ip))
        r.Start()
        logging.info('{} started on {}'.format(MASTER_CONTAINER_NAME, master_ip))
    except Exception as e:
        print('Could not start container {}:\n{}'.format(MASTER_CONTAINER_NAME, e))
        m.Destroy(MASTER_CONTAINER_NAME)
        client.Destroy()
        sys.exit(1)

    state = m.GetProperty(MASTER_CONTAINER_NAME, 'state')
    while state != 'dead':
        state = m.GetProperty(MASTER_CONTAINER_NAME, 'state')
        time.sleep(1)
    logging.info('STDOUT\n{}'.format(m.GetProperty(MASTER_CONTAINER_NAME, 'stdout')))
    logging.info('STDERR\n{}'.format(m.GetProperty(MASTER_CONTAINER_NAME, 'stderr')))

    m.Destroy(MASTER_CONTAINER_NAME)
    client.Destroy()
