#include "msgbus.h"

#include <library/cpp/netliba/v12/udp_host_connection.h>
#include <util/datetime/base.h>
#include <util/stream/output.h>

using namespace NNetliba_v12;

TBusResolveError::TBusResolveError(int code)
    : code(code)
{
}

TMsgBus::TMsgBus(int port, float timeout)
    : Loop(*this)
    , Connections(100)
    , ResultCallback(nullptr)
    , ResultCallbackObj(nullptr)
{
    Socket = NNetlibaSocket::CreateBestRecvSocket();
    if (Socket->Open(port) != 0) {
        ythrow yexception() << "Unable to open a socket on port " << port << " (errno=" << errno << ")";
    }

    UdpHost = CreateUdpHost(Socket, timeout);
}

TMsgBus::~TMsgBus() {
    Stop();
    Connections.Clear();
    UdpHost.Drop();
    Socket.Drop();
}

void TMsgBus::Start() {
    Loop.Start();
}

void TMsgBus::Stop() {
    Loop.Stop();
}

void TMsgBus::Send(TString msg, TString addr, TString dest_host) {
    auto address = CreateAddress(addr, 0, UAT_ANY);
    if (address.Interface == 0)
        // FIXME (torkve) when dcherednik@ provides errorcode, set it
        ythrow TBusResolveError(-2);

    SendQueue.Enqueue({msg, TDeque<TUdpAddress>(1, address), dest_host});
}

void TMsgBus::Send(TString msg, const TDeque<TUdpAddress>& addrs, TString dest_host) {
    SendQueue.Enqueue({msg, addrs, dest_host});
}

void TMsgBus::SetSendResultCallback(void *cookie, TMsgBus::TSendResultCallback cb)
{
    ResultCallbackObj = cookie;
    ResultCallback = cb;
}

TMsgBus::RecvMsg TMsgBus::Receive(bool block, double timeout) {
    auto&& msg = RecvQueue.Pop(block, timeout);

    return {std::get<0>(msg), GetAddressAsString(std::get<1>(msg)), GetAddressAsString(std::get<2>(msg))};
}

void TMsgBus::SendMessages() {
    TSendMsg msg;
    while (SendQueue.Dequeue(&msg)) {
        auto connection = GetConnection(msg.addresses.front());

        TAutoPtr<TRopeDataPacket> data(new TRopeDataPacket);
        data->WriteNoCopy(msg.data.data(), msg.data.size());

        auto transfer = UdpHost->Send(connection, data, PP_NORMAL, TTos(), DEFAULT_NETLIBA_COLOR);
        // Cout << Now() << " Trying to send " << (+msg.data) << " bytes to " << NNetliba_v12::GetAddressAsString(connection->GetAddress()) << " (transfer " << transfer.Id << ")" << Endl;
        Transfers[transfer] = std::move(msg);
    }
}

void TMsgBus::RecvMessages() {
    while (auto request = UdpHost->GetRequest()) {
        TBlockChainIterator it = request->Data->GetChain();
        TString s;
        ReadArr(&it, &s);

        NNetliba_v12::TUdpAddress myAddress;
        auto conn = dynamic_cast<NNetliba_v12::TConnection*>(request->Connection.Get());
        if (conn != nullptr)
            myAddress = conn->GetMyAddress();
        RecvQueue.Push(std::make_tuple(std::move(s), request->Connection->GetAddress(), myAddress));
        delete request;
    }
}

void TMsgBus::ProcessSendResults() {
    NNetliba_v12::TSendResult res;
    while (UdpHost->GetSendResult(&res)) {
        // Cout << Now() << " Transfer " << res.Transfer.Id << " to address " << NNetliba_v12::GetAddressAsString(res.Transfer.Connection->GetAddress()) << " finished with " << (int)res.Ok << Endl;
        auto it = Transfers.find(res.Transfer);
        if (it == Transfers.end())
            continue;

        auto &&msg = it->second;

        auto shouldRetry = (msg.addresses.size() > 1);
        if (ResultCallback != nullptr) {
            ESendResult r = ESendResult::Ok;
            if (res.Ok != NNetliba_v12::TSendResult::OK)
                r = (
                        shouldRetry
                        ? ESendResult::Retried
                        : ESendResult::Failed
                        );
            ResultCallback(ResultCallbackObj, msg.dest_host, GetAddressAsString(msg.addresses.front()), r);
        }

        if (res.Ok == NNetliba_v12::TSendResult::FAILED && shouldRetry) {
            msg.addresses.pop_front();
            SendQueue.Enqueue(std::move(msg));
        }

        // Cout << "Removing transfer " << it->first.Id << Endl;
        Transfers.erase(it);
    }
}

TMsgBus::TConnectionPtr TMsgBus::GetConnection(const NNetliba_v12::TUdpAddress &address) {
    auto it = Connections.Find(address);
    if (it != Connections.End()) {
        return *it;
    }
    auto settings = TConnectionSettings();
    auto conn = UdpHost->Connect(address, settings);
    Connections.Update(address, conn);
    return conn;
}

int TMsgBus::GetListenPort() {
    return Socket->GetPort();
}

TMsgBus::TLoop::TLoop(TMsgBus& bus)
    : Running(1)
    , Started()
    , Bus(bus)
{
}

TMsgBus::TLoop::~TLoop() {
    Stop();
}

void TMsgBus::TLoop::Start() {
    Thread = SystemThreadFactory()->Run(this);
    Started.Signal();
}

void TMsgBus::TLoop::Stop() {
    AtomicSet(Running, 0);
    // Bus.UdpHost->CancelWait();
    if (Thread) {
        Thread->Join();
    }
}

void TMsgBus::TLoop::DoExecute() {
    Started.Wait();

    while (AtomicGet(Running)) {
        Bus.UdpHost->Wait(0.1);
        Bus.UdpHost->Step();

        Bus.SendMessages();
        Bus.RecvMessages();
        Bus.ProcessSendResults();
    }
}

TMsgBus::TransferState TMsgBus::GetTransferState() {
    return {SendQueue.IsEmpty(), RecvQueue.Size(), Transfers.size()};
}
