#pragma once

#include <util/generic/queue.h>
#include <util/system/condvar.h>
#include <util/system/guard.h>
#include <util/system/mutex.h>

class TSyncQueueTimeout : public yexception {};

template<typename T>
class TSyncQueue {
public:
    void Push(const T& val) {
        with_lock (Mutex) {
            Queue.push(val);
            NotEmpty.Signal();
        }
    }

    void Push(T&& val) {
        with_lock (Mutex) {
            Queue.push(val);
            NotEmpty.Signal();
        }
    }

    T Pop(bool block = true, double secondsTimeout = .0) {
        with_lock (Mutex) {
            double millisecondsTimeout = secondsTimeout * 1000.;
            TInstant deadline = secondsTimeout < 0
                ? TInstant::Max()
                : TDuration::MilliSeconds(millisecondsTimeout).ToDeadLine();

            while (!Queue && block && TInstant::Now() < deadline) {
                NotEmpty.WaitD(Mutex, deadline);
            }

            if (!Queue) {
                ythrow TSyncQueueTimeout() << "Queue is empty";
            }

            T val = Queue.front();
            Queue.pop();
            return val;
        }
        ythrow yexception() << "Unable to acquire the lock";
    }

    size_t Size() const {
        with_lock (Mutex) {
            return Queue.size();
        }
        ythrow yexception() << "Unable to acquire the lock";
    }

private:
    TCondVar NotEmpty;
    TMutex Mutex;

    TQueue<T> Queue;
};

