import logging
import random
import tornado.gen
import tornado.httpclient
import urlparse

from . import const
from . import exceptions
from . import file_cache
from . import settings
from . import utils
from . import _unistat_callback

from infra.yp_service_discovery.api import api_pb2 as sd_api


class NocSlaBackendMaintainer(object):
    """
    Maintains a list of aggregator URLs with resolved IPs and updates it periodically.
    URLs are in the form 'http(s)://[ip]'.

    """

    def __init__(self, sd_enabled):
        self._sd_enabled = sd_enabled
        self._sd_client_name = "netmon-agent/{}".format(const.VERSION)

        parsed_urls = sorted(
            (urlparse.urlparse(url) for url in settings.current().noc_sla_urls),
            key=lambda x: x.hostname
        )
        self._noc_sla_hostnames = tuple(x.hostname for x in parsed_urls)
        self._noc_sla_schemes = tuple(x.scheme for x in parsed_urls)

        if any(host is None for host in self._noc_sla_hostnames):
            raise exceptions.ValidationError("noc_sla_urls have invalid format")

        self._fallback_urls = tuple(settings.current().noc_sla_fallback_urls)

        self._sd_addrs = []
        self._last_sd_timestamp = None

        self._cache = file_cache.FileCache(
            "noc_sla_urls",
            file_cache.FileCache.obsolete_ttl
        )
        # cache.get() returns a list of urls or None if empty
        self._cached_urls = tuple(self._cache.get() or ())
        self._backend_urls = self._cached_urls

    @tornado.gen.coroutine
    def _resolve_backends_sd(self):
        """Resolve backends via YP service discovery."""

        request = sd_api.TReqResolveEndpoints(
            cluster_name=settings.current().yp_sd_cluster_name,
            endpoint_set_id=settings.current().yp_sd_endpoint_set_id,
            client_name=self._sd_client_name
        )
        response = sd_api.TRspResolveEndpoints()
        try:
            yield utils.make_protobuf_request(request, response, settings.current().yp_sd_url)
        except Exception as exc:
            if isinstance(exc, tornado.httpclient.HTTPError):
                logging.warning("Service discovery request failed: %s", exc)
            else:
                logging.exception("Service discovery request failed")
            raise tornado.gen.Return([])

        if response.timestamp <= self._last_sd_timestamp:
            # return last response
            logging.info("Got stale timestamp, returning last SD response")
            raise tornado.gen.Return(self._sd_addrs)
        self._last_sd_timestamp = response.timestamp

        if response.resolve_status == sd_api.EResolveStatus.OK:
            # sort endpoints by fqdn so that all agents will pick the same "master" aggregator
            fqdns_with_addrs = [
                (endpoint.fqdn, endpoint.ip6_address)
                for endpoint in response.endpoint_set.endpoints
                if endpoint.ip6_address
            ]
            self._sd_addrs = [ip for fqdn, ip in sorted(fqdns_with_addrs)]
            if not self._sd_addrs:
                logging.error("Service discovery returned empty endpoint set")
        else:
            self._sd_addrs = []
            logging.warning("Service discovery returned resolve_status %s != OK", response.resolve_status)

        if self._sd_addrs:
            logging.info("Resolved NOCSLA backends via service discovery: %s", self._sd_addrs)
        raise tornado.gen.Return(self._sd_addrs)

    @tornado.gen.coroutine
    def _resolve_backends_dns(self, resolver):
        """Resolve backends via DNS."""
        try:
            logging.info("Resolving self._noc_sla_hostnames: %r", self._noc_sla_hostnames)
            resolved_addresses = yield [
                resolver.try_resolve(url)
                for url in self._noc_sla_hostnames
            ]
            selected_addresses = (random.choice(host_addrs) for host_addrs in resolved_addresses)
            ips = [addr for family, addr in selected_addresses]
        except Exception:
            logging.exception("Failed to resolve NOCSLA backends via DNS")
            raise tornado.gen.Return([])

        logging.info("Resolved NOCSLA backends via DNS: %s", ips)
        raise tornado.gen.Return(ips)

    @tornado.gen.coroutine
    def update_backends(self, dns_resolver=None):
        addrs = []
        if self._sd_enabled:
            addrs = yield self._resolve_backends_sd()
            if not addrs:
                _unistat_callback.push_signal(_unistat_callback.YpSdFails, 1.0)
        if not addrs and dns_resolver is not None and self._noc_sla_hostnames:
            addrs = yield self._resolve_backends_dns(dns_resolver)
            if not addrs:
                _unistat_callback.push_signal(_unistat_callback.DnsFails, 1.0)

        if addrs:
            self._backend_urls = tuple(
                "{}://[{}]".format(scheme, addr)
                for scheme, addr in zip(self._noc_sla_schemes, addrs)
            )
            if (
                self._backend_urls != self._cached_urls and
                self._cache.put(self._backend_urls)
            ):
                self._cached_urls = self._backend_urls
        else:
            if self._fallback_urls:
                logging.warning("Using fallback NOCSLA backend urls: %s", self._fallback_urls)
                self._backend_urls = self._fallback_urls
            else:
                logging.warning("Using previous resolved NOCSLA backend urls: %s", self._backend_urls)

    def get_backends(self):
        return self._backend_urls

    def get_master(self):
        return self._backend_urls[0] if self._backend_urls else ''
