# coding: utf-8
from __future__ import print_function

import yaml
import click

from .agent import AgentService
from .daemon import AgentControl
from .benchmarks import LoopbackBenchmark, PeerToPeerBenchmark
from .const import FULL_VERSION
from .pinger import PingCommand
from .traceroute import TracertCommand
from .settings import Settings
from .topology import HostInfoCommand
from .utils import FASTBONE, BACKBONE


@click.group(context_settings=dict(help_option_names=['-h', '--help'],
                                   max_content_width=click.get_terminal_size()[0],
                                   default_map=Settings.current().to_dict()),
             invoke_without_command=True)
@click.option("--debug/--no-debug", help="Be move verbose.", required=False)
@click.option("--freeze-topology/--no-freeze-topology", help="Don't update topology on run.", required=False)
@click.option("--user", help="Effective user.", required=False)
@click.option("--group", help="Effective group.", required=False)
@click.option("--stats-port", help="Agent stats port.", type=click.INT, required=False)
@click.option("--var-dir", help="Directory for temporary files.",
              type=click.Path(exists=True, dir_okay=True, file_okay=False), required=False)
@click.option("--pid-path", help="Path to pid file.", type=click.Path(exists=False, dir_okay=False), required=False)
@click.option("--log-path", help="Path to log file, stderr if not specified.", type=click.Path(), required=False)
@click.option("--config-path", help="Path to config file.",
              type=click.Path(exists=True, dir_okay=False, file_okay=True), required=False)
@click.option("--hostname", help="Override local hostname.", required=False)
@click.option("--netmon-url", help="Override server URL.", required=False)
@click.option("--noc-sla-url", help="NOC SLA server URLs delimited by commas. If service discovery is enabled, these urls are used as a fallback.", required=False)
@click.option("--noc-sla-fallback-url", help="NOC SLA server URLs delimited by commas. Used as a fallback for --noc-sla-url if DNS resolution fails.", required=False)
@click.option("--provisioning-url", help="Override provisioning server URL.", required=False)
@click.option("--yp-sd-url", help="YP service discovery URL.", required=False)
@click.option("--yp-sd-cluster-name", help="Cluster name for requests to YP SD.", required=False)
@click.option("--yp-sd-endpoint-set-id", help="Endpoint set ID for requests to YP SD.", required=False)
@click.option("--yp-sd-request-interval", help="Interval between requests to YP SD.", type=int, required=False)
@click.option("--echo-port-range", help="Range of UDP ports that should be used for checks.",
              type=(int, int), required=False)
@click.option("--tcp-port-range", help="Range of TCP ports that should be used for checks.",
              type=(int, int), required=False)
@click.option("--check-interval", help="How often agent can spawn checks.", type=(int, int), required=False)
@click.option("--udp-socket-count", help="UDP sockets count used to send probes.", type=int, required=False)
@click.option("--udp-multiport-probes/--no-udp-multiport-probes", help="Use multiple source ports in each UDP probe.", required=False)
@click.option("--packet-count", help="Number of packets to send for each probe.", type=int, required=False)
@click.option("--packet-size", help="Size of each packet.", type=int, required=False)
@click.option("--packet-timeout", help="Timeout before packet is considered as lost.", type=float, required=False)
@click.option("--packet-delay", help="Delay between packets.", type=float, required=False)
@click.option("--packet-ttl", help="Set TTL for packets.", type=int, required=False)
@click.option("--diagnostic-probe-count", help="Number of diagnostic probes that should be started.", type=int, required=False)
@click.option("--diagnostic-packet-count", help="Number of diagnostic packets to send for each probe.", type=int, required=False)
@click.option("--diagnostic-packet-size", help="Size of each diagnostic packet.", type=int, required=False)
@click.option("--diagnostic-packet-ttl", help="TTL of each diagnostic packet.", type=int, required=False)
@click.option("--diagnostic-packet-delay", help="Delay between diagnostic packets.", type=float, required=False)
@click.option("--probe-start-delay", help="Maximum delay to start probe.", type=float, required=False)
@click.option("--check-full-tos/--no-check-full-tos", help="Check full ToS in reply packets instead of 3 first bits.", required=False)
@click.option("--fix-reply-tos/--no-fix-reply-tos", help="Fix recolored ToS in reply packets to avoid further recoloring.", required=False)
@click.option("--max-targets", help="Maximum target count (not affect scheduled targets).", type=int, required=False)
@click.option("--max-scheduled-targets", help="Maximum scheduled target count.", type=int, required=False)
@click.option("--p2p-threshold", help="Maximum host count in group below which all hosts should be treated as targets.",
              type=int, required=False)
@click.option("--protocols", help="Use only specified protocols.",
              multiple=True, type=click.Choice(["icmp", "udp", "tcp"]), required=False)
@click.option("--networks", help="Use only specified networks.",
              multiple=True, type=click.Choice([FASTBONE, BACKBONE, "any"]), required=False)
@click.option("--groups", help="Blinov expressions to work with.", multiple=True, required=False)
@click.option("--vlans", help="List of vlans to work with.", multiple=True, type=(int, int), required=False)
@click.option("--vrfs", help="List of vrfs to work with.", multiple=True, type=(str, str), required=False)
@click.option("--automatic-targets/--no-automatic-targets", help="Always generate targets using network topology.", required=False)
@click.option("--group-ttl", help="Cache groups for specified time.", type=int, required=False)
@click.option("--topology-ttl", help="Cache topology for specified time.", type=int, required=False)
@click.option("--switch-targets", help="Targets count per switch.", type=int, required=False)
@click.option("--queue-targets", help="Targets count per queue.", type=int, required=False)
@click.option("--inside-dc-targets", help="Targets count inside dc.", type=int, required=False)
@click.option("--between-dc-targets", help="Targets count between dc.", type=int, required=False)
@click.option("--max-probability", help="Upper bound for probability.", type=float, required=False)
@click.option("--min-weight", help="Lower bound for weight.", type=float, required=False)
@click.option("--traffic-class", help="Generate probes using following IPV6 DSCP values", type=int, multiple=True, required=False)
@click.option("--use-topology-ips/--no-use-topology-ips", help="Use IP addresses (both v4 and v6) from topology when DNS resolving failed.", required=False)
@click.option("--ignore-ipv4-dns-fails/--no-ignore-ipv4-dns-fails", help="Use all IPv4 src and dst addrs even if they have no DNS record.", required=False)
@click.option("--dns-resolve-ip4/--no-dns-resolve-ip4", help="Resolve DNS A records.", required=False)
@click.option("--unistat-pusher/--no-unistat-pusher", help="Enable unistat pusher", required=False)
@click.option("--use-hw-timestamps/--no-use-hw-timestamps", help="Try to use HW timestamps for link poller if supported by network adapter", required=False)
@click.option("--allow-virtual/--no-allow-virtual", help="Allow virtual network interfaces", required=False)
@click.option("--uniform-probes/--no-uniform-probes", help="Uniformly distribute probes over the check interval. This option overrides --packet-delay and --probe-start-delay", required=False)
@click.option("--link-poller/--no-link-poller", help="Enable link poller", required=False)
@click.option("--link-poller-packet-size", help="Size of each link poller packet.", type=int, required=False)
@click.option("--link-poller-src-ip", help="IP address to use as fake bb source in link poller packets", type=str, required=False)
@click.option("--link-poller-fb-src-ip", help="IP address to use as fake fb source in link poller packets", type=str, required=False)
@click.option("--link-poller-port", help="Src and dst port to use in link poller packets", type=int, required=False)
@click.option("--use-skynet/--no-use-skynet", help="Enable/disable skynet usage", required=False)
@click.version_option(version=FULL_VERSION)
@click.pass_context
def cli(ctx, **kwargs):
    """Agent for Netmon that checks connectivity between hosts."""
    ctx.obj.from_cli(kwargs)

    if ctx.invoked_subcommand is None:
        ctx.invoke(run)


@cli.command()
@click.pass_context
def run(ctx):
    """Run the daemon itself."""
    service = AgentService(ctx.obj)
    service.start()


@cli.command(name='run_once')
@click.pass_context
def run_once(ctx):
    """Run agent once and print result."""
    service = AgentService(ctx.obj, start_delay=0, start_echo=False)
    service.run_once()


@cli.command()
@click.pass_context
def stop(ctx):
    """Stop the daemon itself."""
    control = AgentControl(ctx.obj)
    control.stop(ctx)


@cli.command()
@click.pass_context
def check(ctx):
    """Check that daemon is running."""
    control = AgentControl(ctx.obj)
    control.check(ctx)


@cli.command()
@click.pass_context
def reload(ctx):
    """Reload the daemon configuration."""
    control = AgentControl(ctx.obj)
    control.reload(ctx)


@cli.command(name='bench_loopback')
@click.pass_context
def bench_loopback(ctx):
    """Benchmark that will send probes over loopback."""
    service = LoopbackBenchmark(ctx.obj)
    service.start()


@cli.command(name='bench_peers')
@click.argument("targets", type=click.File("rb"))
@click.pass_context
def bench_peers(ctx, targets):
    """Benchmark that will send probes between specified targets."""
    service = PeerToPeerBenchmark(ctx.obj, targets=yaml.safe_load(targets) or [])
    service.start()


@cli.command()
@click.option("--protocol", type=click.Choice(["icmp", "udp", "tcp"]), default="udp")
@click.option("--json/--no-json", help="Print result in json format.", default=False)
@click.option("--summarize/--no-summarize", help="Print only aggregated results.", default=False)
@click.option("--print-successful/--no-print-successful", help="Print targets of successful probes (used with --summarize).", default=False)
@click.option("--print-failed/--no-print-failed", help="Print targets of failed probes (used with --summarize).", default=False)
@click.option("--hosts-per-sec", help="How many targets to ping per second.", type=int, default=100)
@click.option("--packet-size", help="Size of each packet", type=int, required=False)
@click.option("--allow-mtn-vlan/--no-allow-mtn-vlan", help="Use mtn vlan interfaces.", default=False)
@click.argument("hostnames", nargs=-1, required=True)
@click.pass_context
def ping(ctx, **kwargs):
    """Check specified space separated hostnames manually and print result to stdout."""
    command = PingCommand(ctx.obj, **kwargs)
    command.start()


@cli.command()
@click.option("--protocol", type=click.Choice(["icmp", "udp", "tcp"]), default="udp")
@click.argument("hostname")
@click.pass_context
def tracert(ctx, protocol, hostname):
    """Show hops to given hostname."""
    command = TracertCommand(ctx.obj, protocol, hostname)
    command.start()


@cli.command(name='host_info')
@click.option("--fqdn", help="Find interfaces by fqdn name.", required=False)
@click.option("--dc", help="Find interfaces by datacenter name.", required=False)
@click.option("--queue", help="Find interfaces by queue name.", required=False)
@click.option("--switch", help="Find interfaces by switch name.", required=False)
@click.option("--vlan", help="Find interfaces by VLAN.", required=False)
@click.option("--vrf", help="Find interfaces by VRF.", required=False)
@click.pass_context
def host_info(ctx, **kwargs):
    """Show information about given hostname."""
    kwargs = {k: v for k, v in kwargs.items() if v is not None}
    if len(kwargs) > 1:
        ctx.fail("too many filters specified")
    command = HostInfoCommand(ctx.obj, **kwargs)
    command.start()


@cli.command(name='dump_config')
@click.pass_context
def dump_config(ctx):
    """Simply show resulting configuration."""
    click.echo(yaml.safe_dump(ctx.obj.to_dict()))


def main():
    cli(obj=Settings.current())
