# coding: utf-8
from __future__ import print_function

import signal
import time

import tornado.gen

import click

from . import application
from . import utils


class AgentControl(object):

    def __init__(self, settings):
        self._settings = settings
        self._app = application.Application()

    def stop(self, context, timeout=60):
        deadline = time.time() + timeout

        # try with http api first
        should_send_signal = True
        stats_port = self._settings.stats_port
        if stats_port is not None:
            reply = self._app.run_sync(lambda: tornado.httpclient.AsyncHTTPClient().fetch(
                "http://localhost:{0}/shutdown".format(stats_port),
                raise_error=False
            ))
            should_send_signal = reply.code != 200

        try:
            pid = int(self._settings.pid_path.read_text())
            if should_send_signal and not utils.kill_process(pid, signal.SIGTERM):
                return

            while utils.kill_process(pid, 0) and deadline > time.time():
                time.sleep(0.5)

            if utils.kill_process(pid, 0):
                utils.kill_process(pid, signal.SIGKILL)
                raise Exception("process with pid {0} still alive".format(pid))

        except Exception as exc:
            click.echo("Can't stop agent: {0}".format(exc), err=True)
            context.exit(1)

    def reload(self, context):
        self.check(context)

        try:
            pid = int(self._settings.pid_path.read_text())
            if not utils.kill_process(pid, signal.SIGUSR1):
                raise Exception("process with pid {0} not exists".format(pid))

        except Exception as exc:
            click.echo("Can't reload agent: {0}".format(exc), err=True)
            context.exit(1)

    def check(self, context):
        pid_path = self._settings.pid_path
        try:
            with pid_path.open(mode="a+") as stream:
                if utils.acquire_file_lock(stream):
                    raise Exception("pid {0} not locked".format(pid_path))

            pid = int(pid_path.read_text())
            if not utils.kill_process(pid, 0):
                raise Exception("process with pid {0} not exists".format(pid))

        except Exception as exc:
            click.echo("Agent isn't alive: {0}".format(exc), err=True)
            context.exit(1)
