# coding: utf-8
from __future__ import print_function

import socket

from pyroute2 import IPRoute

import tornado.gen

from infra.netmon.agent.idl import linux_pb2

from .. import ticker
from .. import utils

RESOLUTION = 5


class SystemInfoCollector(object):

    def __init__(self, result, on_changed=None):
        self._result = result
        self._on_changed = on_changed
        self._ip_route = IPRoute()
        self._loop = ticker.LoopingCall("collector_loop", self._one_shot, RESOLUTION)
        self._first_time = True

    @tornado.gen.coroutine
    def _collect_links(self):
        for link in self._ip_route.get_links():
            attrs = dict(link["attrs"])
            stats = attrs.get("IFLA_STATS64", {})
            self._result.Links.add(
                Index=link["index"],
                Name=attrs.get("IFLA_IFNAME"),
                Type=link["ifi_type"],
                OperationState=attrs.get("IFLA_OPERSTATE"),
                MTU=attrs.get("IFLA_MTU"),
                MAC=attrs.get("IFLA_ADDRESS"),
                QueueingDiscipline=attrs.get("IFLA_QDISC"),
                Counters=linux_pb2.TInterfaceCounters(
                    Collisions=stats.get("collisions"),
                    Multicast=stats.get("multicast"),
                    RxBytes=stats.get("rx_bytes"),
                    RxCompressed=stats.get("rx_compressed"),
                    RxCrcErrors=stats.get("rx_crc_errors"),
                    RxDropped=stats.get("rx_dropped"),
                    RxErrors=stats.get("rx_errors"),
                    RxFifoErrors=stats.get("tx_fifo_errors"),
                    RxFrameErrors=stats.get("rx_frame_errors"),
                    RxLengthErrors=stats.get("rx_length_errors"),
                    RxMissedErrors=stats.get("rx_missed_errors"),
                    RxOverErrors=stats.get("rx_over_errors"),
                    RxPackets=stats.get("rx_packets"),
                    TxAbortedErrors=stats.get("tx_aborted_errors"),
                    TxBytes=stats.get("tx_bytes"),
                    TxCarrierErrors=stats.get("tx_carrier_errors"),
                    TxCompressed=stats.get("tx_compressed"),
                    TxDropped=stats.get("tx_dropped"),
                    TxErrors=stats.get("tx_errors"),
                    TxFifoErrors=stats.get("rx_fifo_errors"),
                    TxHeartbeatErrors=stats.get("tx_heartbeat_errors"),
                    TxPackets=stats.get("tx_packets"),
                    TxWindowErrors=stats.get("tx_window_errors")
                ),
                Generated=utils.timestamp()
            )
            yield tornado.gen.moment

    @tornado.gen.coroutine
    def _collect_addresses(self):
        for addr in self._ip_route.get_addr():
            attrs = dict(addr["attrs"])
            self._result.Addresses.add(
                Address=attrs.get("IFA_ADDRESS"),
                Prefix=addr["prefixlen"],
                Family=addr["family"],
                Scope=addr["scope"],
                Index=addr["index"],
                Generated=utils.timestamp()
            )
            yield tornado.gen.moment

    @tornado.gen.coroutine
    def _collect_routes(self):
        for family in (socket.AF_INET, socket.AF_INET6):
            for route in self._ip_route.get_routes(family):
                attrs = dict(route["attrs"])
                self._result.Routes.add(
                    Family=route["family"],
                    Table=route["table"],
                    Protocol=route["proto"],
                    Type=route["type"],
                    Scope=route["scope"],
                    Source=attrs.get("RTA_SRC"),
                    SourcePrefix=route["src_len"],
                    Destination=attrs.get("RTA_DST"),
                    DestinationPrefix=route["dst_len"],
                    Gateway=attrs.get("RTA_GATEWAY"),
                    InputInterfaceIndex=attrs.get("RTA_IIF"),
                    OutputInterfaceIndex=attrs.get("RTA_OIF"),
                    Priority=attrs.get("RTA_PRIORITY"),
                    Metrics={k: str(v) for k, v in attrs["RTA_METRICS"]["attrs"]} if "RTA_METRICS" in attrs else {},
                    Generated=utils.timestamp()
                )
                yield tornado.gen.moment

    @tornado.gen.coroutine
    def _one_shot(self):
        yield self._collect_links()

        if self._first_time:
            yield self._collect_addresses()
            yield self._collect_routes()
            self._first_time = False

        if self._on_changed is not None:
            self._on_changed()

    @tornado.gen.coroutine
    def cancel(self):
        yield self._loop.cancel()
        self._ip_route.close()
