# coding: utf-8
from __future__ import print_function

import hashlib
import logging
import msgpack
import time

from . import settings
from . import utils


class FileCache(object):
    """Simply put some value into it and fetch it while it's fresh."""

    prefix = "pinger3"
    version = 4
    obsolete_ttl = 3 * 24 * 60 * 60

    def __init__(self, key, ttl):
        self._path = settings.current().var_dir.joinpath(self._generate_name(key))
        self._ttl = ttl

    @classmethod
    def _generate_name(cls, key):
        return "{0}.{1}".format(cls.prefix, hashlib.md5(key).hexdigest())

    def get(self):
        should_update = utils.should_file_be_updated(
            self._ttl,
            self._path,
            settings.current().hostname
        )
        if should_update:
            return None

        try:
            with open(str(self._path), "rb") as stream:
                version, data = msgpack.load(stream)
        except Exception as exc:
            logging.debug('Unable to load cache: %r', exc)
        else:
            return None if version != self.version else data

    @classmethod
    def _should_be_deleted(cls, path):
        try:
            stat = path.stat()
        except OSError:
            return True
        else:
            return not stat.st_size or stat.st_mtime + cls.obsolete_ttl <= time.time()

    def _cleanup(self):
        # let's cleanup expired entries
        for path in self._path.parent.glob("{0}.*".format(self.prefix)):
            if path != self._path and self._should_be_deleted(path):
                try:
                    path.unlink()
                except Exception:
                    logging.warning('Unable to unlink %s', path)

    def put(self, value):
        assert value is not None, "value can't be None"

        tmp_path = self._path.parent.joinpath("{0}.tmp".format(self._path.name))
        try:
            with open(str(tmp_path), "wb") as stream:
                msgpack.dump((self.version, value), stream)
            tmp_path.rename(self._path)
        except Exception:
            logging.exception("Can't write cache for %s", self._path)
            return False

        self._cleanup()
        return True
