# coding: utf-8
from __future__ import print_function

import collections

import socket
import ipaddress

import os.path

from pyroute2 import IPRoute


FASTBONE = 'fastbone'
BACKBONE = 'backbone'
NETWORKS = (FASTBONE, BACKBONE)

RT_SCOPE_UNIVERSE = 0
RT_TABLE_MAIN = 254
AF_PACKET = 17


def _get_attrs(obj):
    return obj.get('attrs', []) if isinstance(obj, dict) else []


def _get_address_string(addr):
    address = None
    for key, value in _get_attrs(addr):
        if key == 'IFA_ADDRESS':
            address = value
            break
    return unicode(address) + '/' + str(addr['prefixlen']) if address is not None else u''


def is_mtn_vlan(vlan_id):
    MTN_VLANS = [
        688,
        788
    ]
    return vlan_id in MTN_VLANS


class NetworkInterface:

    __slots__ = ('index', 'name', 'mac', 'kind', 'vlan_id')

    def __init__(self, link, allow_virtual=False):
        self.index = link['index']
        self.name = None
        self.kind = None
        self.vlan_id = None
        self._allow_virtual = allow_virtual

        for key, value in _get_attrs(link):
            if key == 'IFLA_IFNAME':
                self.name = value
            elif key == 'IFLA_ADDRESS':
                self.mac = value
            elif key == 'IFLA_LINKINFO':
                self._parse_linkinfo(value)

    def _parse_linkinfo(self, linkinfo):
        for key, value in _get_attrs(linkinfo):
            if key == 'IFLA_INFO_KIND':
                self.kind = value
            elif key == 'IFLA_INFO_DATA':
                self._parse_info_data(value)

    def _parse_info_data(self, info_data):
        if self.kind == 'vlan':
            for key, value in _get_attrs(info_data):
                if key == 'IFLA_VLAN_ID':
                    self.vlan_id = value

    @property
    def has_vendor(self):
        return os.path.exists('/sys/class/net/' + str(self.name) + '/device/vendor') if self.name is not None else False

    @property
    def is_potentially_host_interface_for_ipv4(self):
        if self.has_vendor and self.kind is None:
            return True
        elif self.kind == 'ip6tnl':
            return False
        elif self.kind == 'tun':
            return True
        elif self.kind is not None and self.kind.startswith('vhost'):
            # Yandex Cloud
            return True
        elif self.kind == 'veth' and self._allow_virtual:
            return True
        else:
            return False

    @property
    def is_potentially_host_interface_for_ipv6(self):
        if self.has_vendor and self.kind is None:
            return True
        elif self.kind == 'vlan':
            return True
        elif self.kind == 'bridge':
            return True
        elif self.kind == 'veth' and self._allow_virtual:
            return True
        else:
            return False


class HostIPv4Address(ipaddress.IPv4Interface):
    def __init__(self, address):
        super(HostIPv4Address, self).__init__(address)


class HostIPv6Address(ipaddress.IPv6Interface):
    # L3 balancers
    BALANCER_NETWORKS = [
        ipaddress.ip_network(u'2a02:6b8:0:3400::/64')
    ]

    FASTBONE_NETWORKS = [
        ipaddress.ip_network(u'2620:10f:d00f::/48'),
        ipaddress.ip_network(u'2a02:6b8:0:a00::/56'),
        ipaddress.ip_network(u'2a02:6b8:0:1603::/64'),
        ipaddress.ip_network(u'2a02:6b8:f000::/36')
    ]

    BACKBONE_NETWORKS = [
        ipaddress.ip_network(u'2620:10f:d000::/44'),
        ipaddress.ip_network(u'2a02:6b8::/32')
    ]

    BACKBONE_EXCLUDE_NETWORKS = [
        ipaddress.ip_network(u'2a02:6b8:0:8000::/50')  # legacy CLOS in SAS
    ]

    def __init__(self, address):
        super(HostIPv6Address, self).__init__(address)

    def _is_subnet(self, networks):
        return any(self.network.subnet_of(network) for network in networks)

    @property
    def is_fastbone(self):
        return not self._is_subnet(self.BALANCER_NETWORKS) and self._is_subnet(self.FASTBONE_NETWORKS)

    @property
    def is_backbone(self):
        return (self._is_subnet(self.BACKBONE_NETWORKS)
                and not self.is_fastbone
                and not self._is_subnet(self.BALANCER_NETWORKS)
                and not self._is_subnet(self.BACKBONE_EXCLUDE_NETWORKS))


def get_host_interfaces(allow_virtual, networks, allow_mtn_vlan):
    HostInterface = collections.namedtuple('HostInterface', ('family', 'address', 'mask', 'mac', 'vlan', 'backbone6', 'fastbone6'))

    host_interfaces = []

    with IPRoute() as ipr:
        for link in ipr.get_links():
            iface = NetworkInterface(link, allow_virtual)

            if iface.is_potentially_host_interface_for_ipv4:
                for addr in ipr.get_addr(family=socket.AF_INET, index=iface.index):
                    if addr['scope'] == RT_SCOPE_UNIVERSE:
                        address = HostIPv4Address(_get_address_string(addr))
                        if not address.is_loopback and not address.is_link_local and not address.is_multicast:
                            host_interfaces.append(
                                HostInterface(
                                    family=socket.AF_INET, address=str(address.ip),
                                    mask=str(address.netmask), mac=str(iface.mac),
                                    vlan=iface.vlan_id,
                                    backbone6=False,
                                    fastbone6=False
                                )
                            )

            if iface.is_potentially_host_interface_for_ipv6:
                for addr in ipr.get_addr(family=socket.AF_INET6, index=iface.index):
                    if addr['scope'] == RT_SCOPE_UNIVERSE:
                        address = HostIPv6Address(_get_address_string(addr))
                        if networks is not None:
                            if not address.is_global or address.is_multicast:
                                continue
                            network_is_allowed = (
                                (address.is_fastbone and FASTBONE in networks) or
                                (address.is_backbone and BACKBONE in networks)
                            )
                            if not network_is_allowed:
                                continue
                        if not allow_mtn_vlan and is_mtn_vlan(iface.vlan_id):
                            continue
                        host_interfaces.append(
                            HostInterface(
                                family=socket.AF_INET6, address=str(address.ip),
                                mask=str(address.netmask), mac=str(iface.mac),
                                vlan=iface.vlan_id,
                                backbone6=address.is_backbone,
                                fastbone6=address.is_fastbone
                            )
                        )
    return host_interfaces


def get_default_gw6():
    with IPRoute() as ipr:
        default = ipr.get_default_routes(family=socket.AF_INET6, table=RT_TABLE_MAIN)
        for route in default:
            for name, value in _get_attrs(route):
                if name == 'RTA_GATEWAY':
                    return value
    return None


def get_neigh6_lladdr(dst, oif):
    if not dst or not oif:
        return None
    with IPRoute() as ipr:
        for neigh in ipr.get_neighbours(family=socket.AF_INET6, ifindex=oif, dst=dst):
            for name, value in _get_attrs(neigh):
                if name == 'NDA_LLADDR':
                    return value
    return None


def get_preferred_src_addr6(dst, oif):
    with IPRoute() as ipr:
        for route in ipr.route('get', dst=dst, ifindex=oif, famliy=socket.AF_INET6):
            for name, value in _get_attrs(route):
                if name == 'RTA_PREFSRC':
                    return value
    return None


def get_ifindex(lladdr, vlan=None):
    with IPRoute() as ipr:
        for link in ipr.get_links():
            if link.get_attr("IFLA_ADDRESS") == lladdr:
                link_info = link.get_attr("IFLA_LINKINFO")
                info_data = link_info.get_attr("IFLA_INFO_DATA") if link_info else None
                vlan_id = info_data.get_attr("IFLA_VLAN_ID") if info_data else None
                if vlan_id == vlan or not vlan_id and not vlan:
                    return link['index']
    return None


def get_lladdr(oif):
    with IPRoute() as ipr:
        links = ipr.get_links(oif)
        if not links:
            return None
        for name, value in _get_attrs(links[0]):
            if name == 'IFLA_ADDRESS':
                return value
    return None
