# coding: utf-8
from __future__ import print_function

import collections
import time
import logging
import logutils.queue
import queue

from . import settings


class LogFormatter(logging.Formatter):

    def formatTime(self, record, datefmt=None):
        ts = self.converter(record.created)
        if datefmt:
            return time.strftime(datefmt, ts)
        else:
            s = time.strftime("%Y-%m-%d %H:%M:%S", ts)
            return "%s.%03d" % (s, record.msecs)


class RateLimitFilter(object):
    """
    Allows approximately one record with particular fingerprint during interval seconds.
    """

    def __init__(self, interval):
        self.interval = interval
        self._seen_fingerprints = set()
        self._last_interval_start = 0

    def filter(self, record):
        now = time.time()
        if self._last_interval_start < (now - self.interval):
            self._seen_fingerprints.clear()
            self._last_interval_start = now

        fingerprint = getattr(record, "event_key", None)
        if fingerprint is not None:
            if fingerprint in self._seen_fingerprints:
                return False
            self._seen_fingerprints.add(fingerprint)

        return True


class LogQueueListener(logutils.queue.QueueListener):
    def dequeue(self, block):
        # Calling Queue.get() without a timeout breaks Ctrl-C handling,
        # and QueueListener.dequeue(block=True) does just that.
        # We override dequeue() to fix it.
        if not block:
            return self.queue.get(block=False)

        while True:
            try:
                return self.queue.get(timeout=60)
            except queue.Empty:
                continue


class LogQueue(queue.Queue):
    STALENESS_THRESHOLD = 60

    def __init__(self, maxlen):
        self._maxlen = maxlen
        queue.Queue.__init__(self)

    def _init(self, _):
        self.queue = collections.deque(maxlen=self._maxlen)
        self._items_dropped = 0

    def _put(self, x):
        if x is logutils.queue.QueueListener._sentinel:
            # listener thread should exit, discard all remaining records
            self.queue.clear()
        if len(self.queue) == self._maxlen:
            self._items_dropped += 1
        self.queue.append(x)

    def _get(self):
        now = time.time()
        ret = self.queue.popleft()

        if ret is logutils.queue.QueueListener._sentinel:
            return ret
        # otherwise self.queue contains only LogRecords

        # Drop all stale records from the queue, and return any record
        # with the highest priority among those.
        last_popped = ret
        while self.queue and now - last_popped.created > self.STALENESS_THRESHOLD:
            last_popped = self.queue.popleft()
            self._items_dropped += 1
            if last_popped.levelno > ret.levelno:
                ret = last_popped

        if self._items_dropped:
            prefix = '(log queue overflow: {} records dropped) '.format(self._items_dropped)
            ret.msg = prefix + ret.msg
            self._items_dropped = 0
        return ret


class Logger(object):

    handlers = []
    queue = None
    listener = None
    direct_logger = logging.getLogger()  # use root logger until Logger.initialize() is called

    @staticmethod
    def _create_formatter():
        return LogFormatter("%(asctime)s\t%(name)-10s\t%(levelname)s\t%(message)s")

    @classmethod
    def _register_local_handler(cls, log_path=None):
        if log_path is not None:
            direct_handler = logging.handlers.TimedRotatingFileHandler(str(log_path), when="midnight", backupCount=3)
        else:
            direct_handler = logging.StreamHandler()
        direct_handler.setFormatter(cls._create_formatter())

        cls.direct_logger = logging.getLogger('direct')
        cls.direct_logger.propagate = False
        cls.direct_logger.addHandler(direct_handler)

        cls.queue = LogQueue(maxlen=100)
        cls.listener = LogQueueListener(cls.queue, direct_handler)

        handler = logutils.queue.QueueHandler(cls.queue)
        cls.handlers.append(handler)
        logging.getLogger().addHandler(handler)

        cls.listener.start()

    @classmethod
    def _remove_old_handlers(cls):
        root = logging.getLogger()
        old_handler_list, cls.handlers = cls.handlers, []
        for old_handler in old_handler_list:
            root.removeHandler(old_handler)
            old_handler.acquire()
            try:
                old_handler.flush()
                old_handler.close()
            except (IOError, ValueError):
                pass
            finally:
                old_handler.release()

        if cls.listener is not None:
            cls.listener.stop()
        if cls.queue is not None:
            cls.queue.clear()

    @classmethod
    def initialize(cls):
        cls._remove_old_handlers()

        current_settings = settings.current()

        cls._register_local_handler(current_settings.log_path)

        level = logging.DEBUG if current_settings.debug else logging.INFO
        logging.getLogger().setLevel(level)
        cls.direct_logger.setLevel(level)
