# coding: utf-8
from __future__ import print_function

import logging

import tornado.gen
import tornado.httpclient

from infra.netmon.agent.idl import api_pb2 as api

from . import application
from . import encoding
from . import exceptions
from . import file_cache
from . import settings
from . import utils


class RpcClient(application.AppMixin, application.Service):

    def __init__(self):
        self._app = None

    @tornado.gen.coroutine
    def _make_request(self, action, request, response, server_url=None, headers=None, connect_timeout=5, request_timeout=15):
        if server_url is None:
            server_url = settings.current().netmon_url
        url = "{0}/api/client/v1/{1}".format(server_url, action)
        logging.debug("Making a request to %s", url)
        yield utils.make_protobuf_request(request, response, url, headers,
                                          connect_timeout, request_timeout)

    @tornado.gen.coroutine
    def expand_group(self, name):
        cache = file_cache.FileCache("group:{0}".format(name), settings.current().group_ttl)
        hosts = cache.get()
        if hosts is not None:
            raise tornado.gen.Return(hosts)

        request = api.TExpandGroupRequest(Expression=encoding.safe_str(name))
        response = api.TExpandGroupResponse()
        yield self._make_request("expand_groups", request, response)
        if not response.Group.Ready:
            raise exceptions.GroupNotReady(name)

        hosts = list(response.Group.Hosts)
        cache.put(hosts)
        raise tornado.gen.Return(hosts)

    @tornado.gen.coroutine
    def send_reports(self, reports, server_url):
        request = api.TSendReportsRequest(Reports=reports)
        response = api.TSendReportsResponse()
        yield self._make_request("send_reports", request, response, server_url,
                                 {"X-Netmon-Reports-Count": str(len(reports))}
                                 )
        raise tornado.gen.Return(response.Accepted)

    @tornado.gen.coroutine
    def enqueued_tasks(self, host):
        request = api.TEnqueuedTasksRequest(Host=host)
        response = api.TEnqueuedTasksResponse()
        yield self._make_request("enqueued_tasks", request, response)
        raise tornado.gen.Return(response.Tasks)

    @tornado.gen.coroutine
    def finish_tasks(self, tasks):
        request = api.TFinishTasksRequest(Tasks=tasks)
        response = api.TFinishTasksResponse()
        yield self._make_request("finish_tasks", request, response)
        if not all(response.Success):
            raise exceptions.TasksNotSaved()
        raise tornado.gen.Return(response.Success)

    @tornado.gen.coroutine
    def provisioning_info(self):
        request = api.TProvisioningRequest(Host=encoding.safe_str(settings.current().hostname))
        response = api.TProvisioningResponse()
        yield self._make_request("provisioning_info", request, response,
                                 server_url=settings.current().provisioning_url)
        raise tornado.gen.Return(response)

    @tornado.gen.coroutine
    def scheduled_probes(self, type):
        request = api.TScheduledProbesRequest(Type=type, Host=encoding.safe_str(settings.current().hostname))
        server_urls = self._app[application.BackendMaintainerService].get_noc_sla_backends()
        master_url = self._app[application.BackendMaintainerService].get_noc_sla_master()
        if not server_urls or not master_url:
            raise exceptions.BackendUrlsNotReady()

        # Send requests to all backends to keep terminated hosts in sync.
        # Only one response is actually used.
        responses = {url: api.TScheduledProbesResponse() for url in server_urls}

        make_request_wrapper = utils.suppress_http_errors(self._make_request)
        response_codes = yield {
            url: make_request_wrapper("scheduled_probes", request, responses[url], server_url=url)
            for url in server_urls
        }
        if response_codes[master_url] == 200:
            raise tornado.gen.Return(responses[master_url])
        else:
            raise RuntimeError("scheduled_probes request to master failed")

    @tornado.gen.coroutine
    def terminated_host(self, server_url):
        request = api.TTerminatedHostRequest(Host=encoding.safe_str(settings.current().hostname))
        response = api.TTerminatedHostResponse()
        yield self._make_request("terminated_host", request, response, server_url)
        raise tornado.gen.Return(response)
